use utf8;
use ExtUtils::testlib;
use Test2::V0;
use URI::Fast qw(uri);

my @uris = (
  '/foo/bar/baz',
  'http://www.test.com',
  'https://test.com/some/path?aaaa=bbbb&cccc=dddd&eeee=ffff',
  'https://user:pwd@192.168.0.1:8000/foo/bar?baz=bat&slack=fnord&asdf=the+quick%20brown+fox+%26+hound#foofrag',
);

ok(uri($_), 'uri') foreach @uris;

is uri(undef), '', 'undef';
is uri(''), '', 'empty string';

is uri('/foo')->scheme, '', 'missing scheme';
is uri('http:'), 'http:', 'non-file scheme w/o host';
is uri('http://test'), 'http://test', 'auth w/ invalid host';

is uri('http://usr:pwd')->usr, '', 'no usr w/o @';
is uri('http://usr:pwd')->pwd, '', 'no pwd w/o @';
is uri('http://usr:pwd')->host, 'usr', 'host w/ invalid port';
is uri('http://usr:pwd')->port, '', 'invalid port number ignored';

is uri('#')->frag, '', 'fragment empty but starts with #';

subtest 'param' => sub{
  is uri('?')->param('foo'), undef, 'empty query';
  is uri('?foo')->param('foo'), undef, 'query key w/o =value';
  is uri('?foo=')->param('foo'), '', 'query key w/o value';
  is uri('?=bar')->param('foo'), undef, 'query =value w/o key && request key ne value';
  is uri('?=bar')->param('bar'), undef, 'query =value w/o key && request key eq value';
  is uri('?=')->param('foo'), undef, 'query w/ = but w/o key or value';
  is uri('???')->param('??'), undef, 'multiple question marks';
  is uri('?food=bard&foo=bar')->param('foo'), 'bar', 'substring match';
};

subtest 'query_hash' => sub{
  is uri('?')->query_hash, hash{ end }, 'empty query';
  is uri('?foo')->query_hash, hash{ field 'foo' => array{ end }; end }, 'query key w/o =value';
  is uri('?foo')->query_hash, hash{ field 'foo' => array{ end }; end }, 'query key w/o =value';
  is uri('?foo=')->query_hash, hash{ field 'foo' => array{ item ''; end }; end }, 'query key w/o value';
  is uri('?=bar')->query_hash, hash{ end }, 'query =value w/o key';
  is uri('?=')->query_hash, hash{ end }, 'query w/ = but w/o key or value';
  is uri('???')->query_hash, hash{ field '??' => array{ end }; end }, 'multiple question marks';
};

subtest 'split_path' => sub{
  is uri('//foo/baz.png')->split_path, array{
    item '';
    item 'foo';
    item 'baz.png';
    end;
  }, 'double leading slashes';

  is uri('/foo/bar/')->split_path, array{
    item 'foo';
    item 'bar';
    end;
  }, 'trailing slash';

  is uri('/foo/bar//')->split_path, array{
    item 'foo';
    item 'bar';
    item '';
    end;
  }, 'double trailing slashes';

  is uri('/foo//bar')->split_path, array{
    item 'foo';
    item '';
    item 'bar';
    end;
  }, 'double internal slashes';
};

subtest 'overruns' => sub{
   # scheme: 32
   # path:   2048
   # query:  2048
   # frag:   128
   # usr:    128
   # pwd:    128
   # host:   512
   # port:   8
   # auth:   779 = 128 (usr) + 128 (pwd) + 512 (host) + 8 (port) + 3 (separator chars)
   ok dies{ uri(sprintf('%s://www.test.com', 'x' x 33)) }, 'scheme';
   ok dies{ uri(sprintf('http://%s', 'x' x 780)) }, 'auth';
   ok dies{ uri(sprintf('http://%s:foo@www.test.com', 'x' x 129)) }, 'usr';
   ok dies{ uri(sprintf('http://someone:%s@www.test.com', 'x' x 129)) }, 'pwd';
   ok dies{ uri(sprintf('http://%s', 'x' x 513)) }, 'host';
   ok dies{ uri('http://www.test.com:1234567890') }, 'port';
   ok dies{ uri(sprintf('http://www.test.com/%s', 'x' x 2049)) }, 'path';
   ok dies{ uri(sprintf('http://www.test.com/foo/?%s', 'x' x 2049)) }, 'query';
   ok dies{ uri(sprintf('http://www.test.com/foo#%s', 'x' x 129)) }, 'frag';
};

done_testing;
