package Venus::Test;

use 5.018;

use strict;
use warnings;

# IMPORTS

use Venus 'fault';

use Venus::Class 'attr', 'base', 'with';

# INHERITS

base 'Venus::Kind';

# INTEGRATES

with 'Venus::Role::Buildable';
with 'Venus::Role::Encaseable';

# REQUIRES

require Test::More;

# STATE

our @EXPORT = ('false', 'test', 'true');

# CONFIG

our $CONFIG = {
  block => {
    abstract => {
      find => {
        list => undef,
        name => 'abstract'
      },
      render => {
        index => 2,
        section => undef
      },
      required => true,
    },
    attribute => {
      find => {
        list => 'attribute',
        name => undef
      },
      render => {
        index => 6,
        section => 'attributes'
      },
      required => false,
    },
    attributes => {
      find => {
        list => undef,
        name => 'attributes'
      },
      render => {
        index => 6,
        section => undef
      },
      required => false,
    },
    authors => {
      find => {
        list => undef,
        name => 'authors'
      },
      render => {
        index => 17,
        section => undef
      },
      required => false,
    },
    description => {
      find => {
        list => undef,
        name => 'description'
      },
      render => {
        index => 5,
        section => undef
      },
      required => true,
    },
    encoding => {
      find => {
        list => undef,
        name => 'encoding'
      },
      render => {
        index => 0,
        section => undef
      },
      required => false,
    },
    error => {
      find => {
        list => 'error',
        name => undef
      },
      render => {
        index => 14,
        section => 'errors'
      },
      required => false,
    },
    example => {
      find => {
        list => 'example',
        name => undef
      },
      render => {
        index => undef,
        section => undef
      },
      required => false,
    },
    feature => {
      find => {
        list => 'feature',
        name => undef
      },
      render => {
        index => 13,
        section => 'features'
      },
      required => false,
    },
    function => {
      find => {
        list => 'function',
        name => undef
      },
      render => {
        index => 10,
        section => 'functions'
      },
      required => false,
    },
    includes => {
      find => {
        list => undef,
        name => 'includes'
      },
      render => {
        index => undef,
        section => undef
      },
      required => false,
    },
    inherits => {
      find => {
        list => undef,
        name => 'inherits'
      },
      render => {
        index => 7,
        section => undef
      },
      required => false,
    },
    integrates => {
      find => {
        list => undef,
        name => 'integrates'
      },
      render => {
        index => 8,
        section => undef
      },
      required => false,
    },
    layout => {
      find => {
        list => undef,
        name => 'layout'
      },
      render => {
        index => undef,
        section => undef
      },
      required => false,
    },
    libraries => {
      find => {
        list => undef,
        name => 'libraries'
      },
      render => {
        index => 9,
        section => undef
      },
      required => false,
    },
    license => {
      find => {
        list => undef,
        name => 'license'
      },
      render => {
        index => 18,
        section => undef
      },
      required => false,
    },
    message => {
      find => {
        list => 'message',
        name => undef
      },
      render => {
        index => 12,
        section => 'messages'
      },
      required => false,
    },
    metadata => {
      find => {
        list => 'metadata',
        name => undef
      },
      render => {
        index => undef,
        section => undef
      },
      required => false,
    },
    method => {
      find => {
        list => 'method',
        name => undef
      },
      render => {
        index => 11,
        section => 'methods'
      },
      required => false,
    },
    name => {
      find => {
        list => undef,
        name => 'name'
      },
      render => {
        index => 1,
        section => undef
      },
      required => true,
    },
    operator => {
      find => {
        list => 'operator',
        name => undef
      },
      render => {
        index => 15,
        section => 'operators'
      },
      required => false,
    },
    partials => {
      find => {
        list => undef,
        name => 'partials'
      },
      render => {
        index => 16,
        section => undef
      },
      required => false,
    },
    project => {
      find => {
        list => undef,
        name => 'project'
      },
      render => {
        index => 19,
        section => undef
      },
      required => false,
    },
    raise => {
      find => {
        list => 'raise',
        name => undef
      },
      render => {
        index => undef,
        section => undef
      },
      required => false,
    },
    signature => {
      find => {
        list => 'signature',
        name => undef
      },
      render => {
        index => undef,
        section => undef
      },
      required => false,
    },
    synopsis => {
      find => {
        list => undef,
        name => 'synopsis'
      },
      render => {
        index => 4,
        section => undef
      },
      required => true,
    },
    tagline => {
      find => {
        list => undef,
        name => 'tagline'
      },
      render => {
        index => undef,
        section => undef
      },
      required => true,
    },
    version => {
      find => {
        list => undef,
        name => 'version'
      },
      render => {
        index => 3,
        section => undef
      },
      required => false,
    },
  },
};

# ATTRIBUTES

attr 'file';

# BUILDERS

sub build_arg {
  my ($self, $data) = @_;

  return {
    file => $data,
  };
}

sub build_self {
  my ($self, $data) = @_;

  return $self if !$self->file;

  for my $name (grep {$CONFIG->{block}->{$_}->{required}} keys %{$CONFIG->{block}}) {
    if (!$self->text->count({name => $name, list => undef})) {
      $self->error_on_new({file => $self->file, block => $name})->throw;
    }
  }

  return $self;
}

# FUNCTIONS

sub test {

  return Venus::Test->new($_[0]);
}

# METHODS

sub auto {
  my ($self, $name, @args) = @_;

  return if !$name;

  my $NAME = uc "venus_test_auto_${name}";

  return $ENV{$NAME} if !@args;

  return ($ENV{$NAME}) = @args;
}

sub diag {
  my ($self, @args) = @_;

  return $self->handler('diag', $self->explain(@args));
}

sub done {
  my ($self) = @_;

  $self->render if $self->auto('render') && !$self->encased('rendered');

  return $self->handler('done_testing');
}

sub eval {
  my ($self, $perl) = @_;

  local $@;

  my @result = CORE::eval(join("\n\n", "no warnings q(redefine);", $perl));

  my $dollarat = $@;

  die $dollarat if $dollarat;

  return wantarray ? (@result) : $result[0];
}

sub explain {
  my ($self, @args) = @_;

  return join ' ',
    map {s/^\s+|\s+$//gr} map {$self->more('explain', $_)} grep {defined} @args;
}

sub fail {
  my ($self, $bool,  @args) = @_;

  my $result = $self->handler('ok', !$bool, @args);

  return $result;
}

sub for {
  my ($self, $type, @args) = @_;

  my $code = ref $args[-1] eq 'CODE' ? pop @args : undef;

  my $name = join ' ', grep {!ref $_} $type, @args;

  $self->subtest($name, sub {$self->execute($type, @args, $code)});

  return $self;
}

sub gate {
  my ($self, $note, $code) = @_;

  my $test = $self->renew;

  $test->recase('gate', [$note, $code]);

  return $test;
}

sub handler {
  my ($self, $name, @args) = @_;

  my $found = 0;
  my $level = 0;

  while (my @caller = caller($level)) {
    $level++;
    $found++ && last if $caller[1] =~ qr{@{[quotemeta($self->file)]}$};
  }

  local $Test::Builder::Level = $found ? $level + 1 : $Test::Builder::Level + $level;

  my @return;

  if (wantarray) {
    @return = Test::More->can($name)->(@args);
  }
  else {
    $return[0] = Test::More->can($name)->(@args);
  }

  # whitelist methods
  state $test_more_functions = {
    BAIL_OUT => 'BAIL_OUT',
    done_testing => 'done_testing',
    plan => 'plan',
    skip => 'skip',
  };
  $self->BAIL_OUT
    if (!@return || (@return == 1 && !$return[0]))
    && $self->auto('bailout')
    && !$test_more_functions->{$name};

  return wantarray ? @return : defined(wantarray) ? $return[0] : ();
}

sub in {
  my ($self, $this, $that, @args) = @_;

  return $self->pass(Venus::in($this, $that), @args);
}

sub is {
  my ($self, $data, @args) = @_;

  my $result = $self->handler('is', scalar($data), @args);

  return $result;
}

sub is_arrayref {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_arrayref($data), @args);
}

sub is_blessed {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_blessed($data), @args);
}

sub is_boolean {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_boolean($data), @args);
}

sub is_coderef {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_coderef($data), @args);
}

sub is_dirhandle {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_dirhandle($data), @args);
}

sub is_enum {
  my ($self, $data, $items, @args) = @_;

  return $self->pass(Venus::is_enum($data, Venus::list($items)), @args);
}

sub is_error {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_error($data), @args);
}

sub is_false {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_false($data), @args);
}

sub is_fault {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_fault($data), @args);
}

sub is_filehandle {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_filehandle($data), @args);
}

sub is_float {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_float($data), @args);
}

sub is_glob {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_glob($data), @args);
}

sub is_hashref {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_hashref($data), @args);
}

sub is_number {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_number($data), @args);
}

sub is_object {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_object($data), @args);
}

sub is_package {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_package($data), @args);
}

sub is_reference {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_reference($data), @args);
}

sub is_regexp {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_regexp($data), @args);
}

sub is_scalarref {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_scalarref($data), @args);
}

sub is_string {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_string($data), @args);
}

sub is_true {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_true($data), @args);
}

sub is_undef {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_undef($data), @args);
}

sub is_value {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_value($data), @args);
}

sub is_yesno {
  my ($self, $data, @args) = @_;

  return $self->pass(Venus::is_yesno($data), @args);
}

sub isnt {
  my ($self, $data, @args) = @_;

  my $result = $self->handler('isnt', scalar($data), @args);

  return $result;
}

sub isnt_arrayref {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_arrayref($data), @args);
}

sub isnt_blessed {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_blessed($data), @args);
}

sub isnt_boolean {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_boolean($data), @args);
}

sub isnt_coderef {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_coderef($data), @args);
}

sub isnt_dirhandle {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_dirhandle($data), @args);
}

sub isnt_enum {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_enum($data), @args);
}

sub isnt_error {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_error($data), @args);
}

sub isnt_false {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_false($data), @args);
}

sub isnt_fault {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_fault($data), @args);
}

sub isnt_filehandle {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_filehandle($data), @args);
}

sub isnt_float {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_float($data), @args);
}

sub isnt_glob {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_glob($data), @args);
}

sub isnt_hashref {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_hashref($data), @args);
}

sub isnt_number {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_number($data), @args);
}

sub isnt_object {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_object($data), @args);
}

sub isnt_package {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_package($data), @args);
}

sub isnt_reference {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_reference($data), @args);
}

sub isnt_regexp {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_regexp($data), @args);
}

sub isnt_scalarref {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_scalarref($data), @args);
}

sub isnt_string {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_string($data), @args);
}

sub isnt_true {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_true($data), @args);
}

sub isnt_undef {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_undef($data), @args);
}

sub isnt_value {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_value($data), @args);
}

sub isnt_yesno {
  my ($self, $data, @args) = @_;

  return $self->fail(Venus::is_yesno($data), @args);
}

sub lfile {
  my ($self) = @_;

  my $lfile = $self->space->format('lfile', 'lib/%s');

  return $lfile;
}

sub like {
  my ($self, $this, $that, @args) = @_;

  $that = qr/$that/ if ref $that ne 'Regexp';

  my $result = $self->handler('like', $this, $that, @args);

  return $result;
}

sub mktemp_dir {
  my ($self) = @_;

  my $path = $self->path;

  return $path->mktemp_dir;
}

sub mktemp_file {
  my ($self) = @_;

  my $path = $self->path;

  return $path->mktemp_file;
}

sub note {
  my ($self, @args) = @_;

  $self->handler('diag', $self->explain(@args));

  return;
}

sub only_if {
  my ($self, $what, @args) = @_;

  local $_ = $self;

  return $self->gate(
    (join ' ', 'only_if', grep {!ref} $what), sub {$self->$what(@args) ? true : false}
  );
}

sub os {
  my ($self) = @_;

  my $os;

  require Venus::Os;

  $os = $self->encased('os');

  return $os if $os;

  $os = Venus::Os->new;

  return $self->recase('os', $os);
}

sub os_is_bsd {
  my ($self) = @_;

  return $self->os->is_bsd;
}

sub os_is_cyg {
  my ($self) = @_;

  return $self->os->is_cyg;
}

sub os_is_dos {
  my ($self) = @_;

  return $self->os->is_dos;
}

sub os_is_lin {
  my ($self) = @_;

  return $self->os->is_lin;
}

sub os_is_mac {
  my ($self) = @_;

  return $self->os->is_mac;
}

sub os_is_non {
  my ($self) = @_;

  return $self->os->is_non;
}

sub os_is_sun {
  my ($self) = @_;

  return $self->os->is_sun;
}

sub os_is_vms {
  my ($self) = @_;

  return $self->os->is_vms;
}

sub os_is_win {
  my ($self) = @_;

  return $self->os->is_win;
}

sub os_isnt_bsd {
  my ($self) = @_;

  return $self->os->is_bsd ? false : true;
}

sub os_isnt_cyg {
  my ($self) = @_;

  return $self->os->is_cyg ? false : true;
}

sub os_isnt_dos {
  my ($self) = @_;

  return $self->os->is_dos ? false : true;
}

sub os_isnt_lin {
  my ($self) = @_;

  return $self->os->is_lin ? false : true;
}

sub os_isnt_mac {
  my ($self) = @_;

  return $self->os->is_mac ? false : true;
}

sub os_isnt_non {
  my ($self) = @_;

  return $self->os->is_non ? false : true;
}

sub os_isnt_sun {
  my ($self) = @_;

  return $self->os->is_sun ? false : true;
}

sub os_isnt_vms {
  my ($self) = @_;

  return $self->os->is_vms ? false : true;
}

sub os_isnt_win {
  my ($self) = @_;

  return $self->os->is_win ? false : true;
}

sub pass {
  my ($self, $bool,  @args) = @_;

  my $result = $self->handler('ok', $bool, @args);

  return $result;
}

sub patch {
  my ($self, $name, $code) = @_;

  my $space = $self->space;

  $space->patch($name, $code);

  my $patched = $space->{patched}->{$name};

  return $patched;
}

sub path {
  my ($self, $file) = @_;

  $file ||= $self->file;

  require Venus::Path;

  return Venus::Path->new($file);
}

sub pfile {
  my ($self) = @_;

  my $pfile = $self->space->format('pfile', 'lib/%s');

  return $pfile;
}

sub render {
  my ($self, $file) = @_;

  $file ||= $self->pfile;

  require Venus::Path;

  my $path = Venus::Path->new($file);

  $path->parent->mkdirs;

  my @layout = split /\r?\n/, join "\n\n", $self->collect('layout');

  my @output;

  for my $item (grep {length} @layout) {
    push @output, grep {length} $self->present(split /:\s*/, $item);
  }

  $path->write(join "\n", @output);

  $self->recase('rendered', true);

  return $path;
}

sub same {
  my ($self, $this, $that, @args) = @_;

  my $result = $self->handler('is_deeply', $this, $that, @args);

  return $result;
}

sub skip {
  my ($self, $note) = @_;

  return $self->handler('plan', 'skip_all', $note);
}

sub skip_if {
  my ($self, $what, @args) = @_;

  local $_ = $self;

  return $self->gate(
    (join ' ', 'skip_if', grep {!ref} $what), sub {$self->$what(@args) ? false : true}
  );
}

sub space {
  my ($self, $name) = @_;

  my $space;

  require Venus::Space;

  $space = Venus::Space->new($name) if $name;

  return $space if $name;

  $space = $self->encased('space');

  return $space if $space;

  $space = Venus::Space->new($self->collect('name'));

  return $self->recase('space', $space);
}

sub subtest {
  my ($self, $name, $code) = @_;

  my $gate = $self->encased('gate');

  my $callback = !$gate ? $code : sub {
    ($gate->[1] && $gate->[1]->()) ? $code->() : $self->skip($gate->[0]);
  };

  my $result = $self->handler('subtest', $name, $callback);

  $self->unpatch if $self->auto('unpatch');

  return $result;
}

sub text {
  my ($self) = @_;

  my $text;

  $text = $self->encased('text');

  return $text if $text;

  require Venus::Text::Pod;

  $text = Venus::Text::Pod->new($self->file);

  return $self->recase('text', $text);
}

sub tfile {
  my ($self) = @_;

  my $tfile = $self->space->format('tfile', 't/%s');

  return $tfile;
}

sub type {
  my ($self, $data, $expr, @args) = @_;

  require Venus::Type;

  my $type = Venus::Type->new;

  return $self->pass($type->check($expr)->eval($data), @args);
}

sub unlike {
  my ($self, $this, $that, @args) = @_;

  $that = qr/$that/ if ref $that ne 'Regexp';

  my $result = $self->handler('unlike', $this, $that, @args);

  return $result;
}

sub unpatch {
  my ($self, @args) = @_;

  my $space = $self->space;

  return $space->unpatch(@args);
}

# COMPATIBILITY

*data = *text; # deprecated since 4.15
*more = *handler; # deprecated since 4.15
*okay = *pass; # deprecated since 4.15

# ERRORS

sub error_on_missing {
  my ($self, $data) = @_;

  my $error = $self->error->sysinfo;

  my $message = 'Can\'t locate object method "{{method}}" via package "{{class}}"';

  $error->name('on.missing');
  $error->message($message);
  $error->offset(1);
  $error->stash($data);
  $error->reset;

  return $error;
}

sub error_on_new {
  my ($self, $data) = @_;

  my $error = $self->error->sysinfo;

  my $message = 'Test file "{{file}}" missing {{block}} block';

  $error->name('on.new');
  $error->message($message);
  $error->offset(1);
  $error->stash($data);
  $error->reset;

  return $error;
}

# AUTOLOAD

sub AUTOLOAD {
  my ($self, @args) = @_;

  our $AUTOLOAD;
  my $method = $AUTOLOAD =~ s/.*:://r;

  # Handle collect method
  if ($method eq 'collect' && @args) {
    my ($name) = shift @args;
    $method = "collect_data_for_${name}";
    return $self->$method(@args) if $self->can($method);
  }

  # Handle collect_data_for_* methods using config
  if (my ($block) = $method =~ /^collect_data_for_(\w+)$/) {
    if ($method eq 'collect_data_for_layout') {
      my ($find) = $self->data->find(undef, 'layout');

      my $data = $find ? $find->{data} : [
        'encoding',
        'name',
        'abstract',
        'version',
        'synopsis',
        'description',
        'attributes: attribute',
        'inherits',
        'integrates',
        'libraries',
        'functions: function',
        'methods: method',
        'messages: message',
        'features: feature',
        'errors: error',
        'operators: operator',
        'partials',
        'authors',
        'license',
        'project',
      ];

      return wantarray ? (@{$data}) : $data;
    }
    elsif ($method eq 'collect_data_for_version') {
      my ($find) = $self->data->find(undef, 'version');

      my $data = $find ? $find->{data} : [];

      require Venus::Space;

      if (!@{$data} && (my ($name) = $self->collect('name'))) {
        @{$data} = (Venus::Space->new($name)->version) || ();
      }

      return wantarray ? (@{$data}) : $data;
    }
    elsif (my $config = $CONFIG->{block}{$block}) {
      my $find_spec = $config->{find};
      my ($list, $name) = @{$find_spec}{qw(list name)};

      if ($block eq 'example' && @args == 2) {
        my ($number, $arg_name) = @args;
        my ($find) = $self->text->find("example-$number", $arg_name);
        my $data = $find ? $find->{data} : [];
        return wantarray ? (@{$data}) : $data;
      }

      my ($find) = defined($list)
        ? $self->text->find($list, @args ? $args[0] : undef)
        : $self->text->find(undef, $name);

      my $data = $find ? $find->{data} : [];

      # special-case: encoding
      if ($block eq 'encoding') {
        @{$data} = (map {map uc, split /\r?\n+/} @{$data});
      }

      # special-case: includes
      if ($block eq 'includes') {
        @{$data} = grep !/^#/, grep /\w/, map {split/\n/} @{$data};
      }

      return wantarray ? (@{$data}) : $data;
    }
  }

  # Handle execute method
  if ($method eq 'execute' && @args) {
    my ($name) = shift @args;
    $method = "execute_test_for_${name}";
    return $self->$method(@args) if $self->can($method);
  }

  # Handle execute_test_for_* methods using config
  if (my ($block) = $method =~ /^execute_test_for_(\w+)$/) {
    if ($method eq 'execute_test_for_attributes') {
      my ($code) = @args;
      my $data = $self->collect('attributes');
      my $result = $self->perform('attributes', $data);
      $result = $code->($data) if $code;
      $self->pass($result, '=attributes');
      my ($package) = $self->collect('name');
      for my $line (@{$data}) {
        next if !$line;
        my ($name, $is, $pre, $isa, $def) = map { split /,\s*/ } split /:\s*/,
          $line, 2;
        $self->pass($package->can($name), "$package has $name");
        $self->pass((($is eq 'ro' || $is eq 'rw')
            && ($pre eq 'opt' || $pre eq 'req')
            && $isa), $line);
      }
      return $result;
    }
    elsif ($method eq 'execute_test_for_example') {
      my ($number, $name, $code) = @args;
      my $data = $self->collect('example', $number, $name);
      my $text = join "\n\n", @{$data};
      my @includes;
      if ($text =~ /.*#\s*given:\s*synopsis/m) {
        my $line = $&;
        if ($line !~ /#.*#\s*given:\s*synopsis/) {
          push @includes, $self->collect('synopsis');
        }
      }
      for my $given ($text =~ /.*#\s*given:\s*example-((?:\d+)\s+(?:[\-\w]+))/gm) {
        my $line = $&;
        if ($line !~ /#.*#\s*given:\s*example-(?:\d+)\s+(?:[\-\w]+)/) {
          my ($number, $name) = split /\s+/, $given, 2;
          push @includes, $self->collect('example', $number, $name);
        }
      }
      $text =~ s/.*#\s*given:\s*.*\n\n*//g;
      $text = join "\n\n", @includes, $text;
      my $result = $self->perform('example', $number, $name, $data);
      $self->pass($result, "=example-$number $name");
      $result = $code->($self->try('eval', $text)) if $code;
      $self->pass($result, "=example-$number $name returns ok") if $code;
      if ($self->auto('scrub')) {
        state $name = $self->text->string(undef, 'name');
        for my $package ($text =~ /package\W+([\w:]+)/g) {
          next if $package eq 'main';
          next if $package eq __PACKAGE__;
          next if $package eq $name;
          $self->space($package)->scrub;
        }
      }
      return $result;
    }
    elsif ($method eq 'execute_test_for_raise') {
      my ($site, $class, $name, $code) = @args;
      my $data = do {
        my ($find) = $self->text->find($block, "$site $class $name");
        $find ? $find->{data} : [];
      };
      my $text = join "\n\n", @{$data};
      my @includes;
      if ($text =~ /.*#\s*given:\s*synopsis/m) {
        my $line = $&;
        if ($line !~ /#.*#\s*given:\s*synopsis/) {
          push @includes, $self->collect('synopsis');
        }
      }
      for my $given ($text =~ /.*#\s*given:\s*example-((?:\d+)\s+(?:[\-\w]+))/gm) {
        my $line = $&;
        if ($line !~ /#.*#\s*given:\s*example-(?:\d+)\s+(?:[\-\w]+)/) {
          my ($number, $name) = split /\s+/, $given, 2;
          push @includes, $self->collect('example', $number, $name);
        }
      }
      $text =~ s/.*#\s*given:\s*.*\n\n*//g;
      $text = join "\n\n", @includes, $text;
      my $handle = join ' ', grep defined, $site, $class, $name;
      my $result = $self->perform('raise', $handle, $data);
      $self->pass($result, "=$handle");
      $result = $code->($self->try('eval', $text)) if $code;
      $self->pass($result, "=error $site $class $name returns ok") if $code;
      if ($self->auto('scrub')) {
        state $name = $self->text->string(undef, 'name');
        for my $package ($text =~ /package\W+([\w:]+)/g) {
          next if $package eq 'main';
          next if $package eq __PACKAGE__;
          next if $package eq $name;
          $self->space($package)->scrub;
        }
      }
      return $result;
    }
    elsif ($method eq 'execute_test_for_synopsis') {
      my ($code) = @args;
      my $data = $self->collect('synopsis');
      my $text = join "\n\n", @{$data};
      my @includes;
      for my $given ($text =~ /.*#\s*given:\s*example-((?:\d+)\s+(?:[\-\w]+))/gm) {
        my $line = $&;
        if ($line !~ /#.*#\s*given:\s*example-(?:\d+)\s+(?:[\-\w]+)/) {
          my ($number, $name) = split /\s+/, $given, 2;
          push @includes, $self->collect('example', $number, $name);
        }
      }
      $text =~ s/.*#\s*given:\s*.*\n\n*//g;
      $text = join "\n\n", @includes, $text;
      my $result = $self->perform('synopsis', $data);
      $self->pass($result, "=synopsis");
      $result = $code->($self->try('eval', $text)) if $code;
      $self->pass($result, "=synopsis returns ok") if $code;
      if ($self->auto('scrub')) {
        state $name = $self->text->string(undef, 'name');
        for my $package ($text =~ /package\W+([\w:]+)/g) {
          next if $package eq 'main';
          next if $package eq __PACKAGE__;
          next if $package eq $name;
          $self->space($package)->scrub;
        }
      }
      return $result;
    }
    elsif ($method eq 'execute_test_for_name') {
      my ($code) = @args;
      my ($text) = $self->collect_data_for_name;
      my $result = length($text) ? true : false;
      $self->pass($result, '=name content');
      $self->pass(scalar(eval("require $text")), "require $text");
      return $result;
    }
    elsif (my $config = $CONFIG->{block}{$block}) {
      my $find_spec = $config->{find};
      my ($list, $name) = @{$find_spec}{qw(list name)};

      my $code = ref($args[-1]) eq 'CODE' ? pop(@args) : undef;
      my $collect = "collect_data_for_$block";
      my $data = $self->$collect(@args);
      my $perform = "perform_test_for_$block";

      my $result = $code
        ? $code->(defined $list ? ($block, $data) : $data)
        : $self->$perform(defined $list ? ($block, $data) : $data);

      $self->pass($result, "=$block");

      return $result;
    }
  }

  # Handle perform method
  if ($method eq 'perform' && @args) {
    my ($name) = shift @args;
    $method = "perform_test_for_${name}";
    return $self->$method(@args) if $self->can($method);
  }

  # Handle perform_test_for_* methods using config
  if (my ($block) = $method =~ /^perform_test_for_(\w+)$/) {
    if ($method eq 'perform_test_for_example') {
      my ($number, $name, $data) = @args;
      my $result = length(join "\n", @{$data}) ? true : false;
      $self->pass($result, "=example-$number $name content");
      return $result;
    }
    elsif ($method eq 'perform_test_for_name') {
      my ($data) = @args;
      my $text = join "\n", @{$data};
      my $result = length($text) ? true : false;
      $self->pass($result, '=name content');
      $self->pass(scalar(eval("require $text")), $self->explain('require', $text));
      return $result;
    }
    elsif (my $config = $CONFIG->{block}{$block}) {
      my $find_spec = $config->{find};
      my $list = $find_spec->{list};
      my ($name, $data) = defined $list ? @args : (undef, $args[0]);
      my $result = length(join "\n", @{$data}) ? true : false;
      $self->pass($result, defined $list ? "=$block $name content" : "=$block content");
      return $result;
    }
  }

  # Handle present method
  if ($method eq 'present' && @args) {
    my ($name) = shift @args;
    $method = "present_data_for_${name}";
    return $self->$method(@args) if $self->can($method);
  }

  # Handle present_data_for_* methods using config
  if (my ($block) = $method =~ /^present_data_for_(\w+)$/) {
    # special-case: present_data_for_abstract
    if ($method eq 'present_data_for_abstract') {
      my @data = $self->collect('abstract');
      return @data ? ($self->present_data_for_head1('abstract', @data)) : ();
    }
    # special-case: present_data_for_attribute
    elsif ($method eq 'present_data_for_attribute') {
      my ($name) = @args;
      return $self->present_data_for_attribute_type2($name);
    }
    # special-case: present_data_for_attribute_type1
    elsif ($method eq 'present_data_for_attribute_type1') {
      my ($name, $is, $pre, $isa, $def) = @args;
      my @output;
      $is = $is eq 'ro' ? 'read-only' : 'read-write';
      $pre = $pre eq 'req' ? 'required' : 'optional';
      push @output, "  $name($isa)\n";
      push @output, "This attribute is $is, accepts C<($isa)> values, ". (
        $def ? "is $pre, and defaults to $def." : "and is $pre."
      );
      return ($self->present_data_for_head2($name, @output));
    }
    # special-case: present_data_for_attribute_type2
    elsif ($method eq 'present_data_for_attribute_type2') {
      my ($name) = @args;
      my @output;
      my ($metadata) = $self->collect('metadata', $name);
      my ($signature) = $self->collect('signature', $name);
      push @output, ($signature, '') if $signature;
      my @data = $self->collect('attribute', $name);
      return () if !@data;
      push @output, join "\n\n", @data;
      if ($metadata) {
        local $@;
        my $perlcode = do{no warnings; eval $metadata};
        if ($metadata = ($perlcode && ref $perlcode) ? $perlcode : $metadata) {
          if (ref $metadata ne 'HASH') {
            my @debug_data = (map {split /:\s*/} split /\n+/, $metadata);
            $metadata = {map {split /:\s*/} split /\n+/, $metadata};
          }
          my @lifespan;
          if (my $introduced = $metadata->{introduced}) {
            push @lifespan, "I<Introduced C<$introduced>>";
          }
          if (my $deprecated = $metadata->{deprecated}) {
            push @lifespan, "I<Deprecated C<$deprecated>>";
          }
          if (my $since = $metadata->{since}) {
            @lifespan = ("I<Since C<$since>>");
          }
          push @output, "", join(", ", @lifespan) if @lifespan;
        }
      }
      my @results = $self->data->search({name => $name});
      for my $i (1..(int grep {($$_{list} || '') =~ /^example-\d+/} @results)) {
        push @output, join "\n\n", $self->present('example', $i, $name);
      }
      pop @output if $output[-1] eq '';
      return ($self->present_data_for_head2($name, @output));
    }
    # special-case: present_data_for_attributes
    elsif ($method eq 'present_data_for_attributes') {
      my $method = $self->data->count({list => undef, name => 'attributes'})
        ? 'attributes_type1'
        : 'attributes_type2';
      return $self->present($method, @args);
    }
    # special-case: present_data_for_attributes_type1
    elsif ($method eq 'present_data_for_attributes_type1') {
      my @output;
      my @data = $self->collect('attributes');
      return () if !@data;
      for my $line (split /\r?\n/, join "\n", @data) {
        push @output, $self->present('attribute_type1', (
          map {split /,\s*/} split /:\s*/, $line, 2
        ));
      }
      return () if !@output;
      if (@output) {
        unshift @output,
          $self->present_data_for_head1('attributes',
            $self->text->string('header', 'attributes') ||
          'This package has the following attributes:');
      }
      return join "\n", @output;
    }
    # special-case: present_data_for_attributes_type2
    elsif ($method eq 'present_data_for_attributes_type2') {
      my @output;
      for my $list ($self->data->search({list => 'attribute'})) {
        push @output, $self->present('attribute_type2', $list->{name});
      }
      if (@output) {
        unshift @output,
          $self->present_data_for_head1('attributes',
            $self->text->string('header', 'attributes') ||
          'This package has the following attributes:');
      }
      return join "\n", @output;
    }
    # special-case: present_data_for_authors
    elsif ($method eq 'present_data_for_authors') {
      my @data = $self->collect('authors');
      return @data ? ($self->present_data_for_head1('authors', join "\n\n", @data)) : ();
    }
    # special-case: present_data_for_description
    elsif ($method eq 'present_data_for_description') {
      my @data = $self->collect('description');
      return @data ? ($self->present_data_for_head1('description', join "\n\n", @data)) : ();
    }
    # special-case: present_data_for_encoding
    elsif ($method eq 'present_data_for_encoding') {
      my ($name) = $self->collect('encoding');
      return () if !$name;
      return join("\n", "", "=encoding \U$name", "", "=cut");
    }
    # special-case: present_data_for_error
    elsif ($method eq 'present_data_for_error') {
      my ($name) = @args;
      my @output;
      my @data = $self->collect('error', $name);
      return () if !@data;
      my @results = $self->data->search({name => $name});
      for my $i (1..(int grep {($$_{list} || '') =~ /^example-\d+/} @results)) {
        push @output, "B<example $i>", $self->collect('example', $i, $name);
      }
      return (
        $self->present_data_for_over($self->present_data_for_item(
          "error: C<$name>",
          join "\n\n", @data, @output
        ))
      );
    }
    # special-case: present_data_for_errors
    elsif ($method eq 'present_data_for_errors') {
      my @output;
      my $type = 'error';
      for my $name (
        sort map $$_{name},
          $self->data->search({list => $type})
      )
      {
        push @output, $self->present($type, $name);
      }
      if (@output) {
        unshift @output,
          $self->present_data_for_head1('errors',
            $self->text->string('header', 'errors') ||
          'This package may raise the following errors:');
      }
      return join "\n", @output;
    }
    # special-case: present_data_for_example
    elsif ($method eq 'present_data_for_example') {
      my ($number, $name) = @args;
      my @data = $self->collect('example', $number, $name);
      return @data
        ? (
        $self->present_data_for_over($self->present_data_for_item(
          "$name example $number", join "\n\n", @data)))
        : ();
    }
    # special-case: present_data_for_feature
    elsif ($method eq 'present_data_for_feature') {
      my ($name) = @args;
      my @output;
      my ($metadata) = $self->collect('metadata', $name);
      my ($signature) = $self->collect('signature', $name);
      push @output, ($signature, '') if $signature;
      my @data = $self->collect('feature', $name);
      return () if !@data;
      push @output, join "\n\n", @data;
      if ($metadata) {
        local $@;
        my $perlcode = do{no warnings; eval $metadata};
        if ($metadata = ($perlcode && ref $perlcode) ? $perlcode : $metadata) {
          if (ref $metadata ne 'HASH') {
            $metadata = {map {split /:\s*/} split /\n+/, $metadata};
          }
          my @lifespan;
          if (my $introduced = $metadata->{introduced}) {
            push @lifespan, "I<Introduced C<$introduced>>";
          }
          if (my $deprecated = $metadata->{deprecated}) {
            push @lifespan, "I<Deprecated C<$deprecated>>";
          }
          if (my $since = $metadata->{since}) {
            @lifespan = ("I<Since C<$since>>");
          }
          push @output, "", join(", ", @lifespan) if @lifespan;
        }
      }
      my @results = $self->data->search({name => $name});
      for my $i (1..(int grep {($$_{list} || '') =~ /^example-\d+/} @results)) {
        push @output, "B<example $i>", $self->collect('example', $i, $name);
      }
      pop @output if $output[-1] eq '';
      return (
        $self->present_data_for_over($self->present_data_for_item(
          $name, join "\n\n", grep {length} @output))
      );
    }
    # special-case: present_data_for_features
    elsif ($method eq 'present_data_for_features') {
      my @output;
      my $type = 'feature';
      for my $name (
        sort map $$_{name},
          $self->data->search({list => $type})
      )
      {
        push @output, $self->present($type, $name);
      }
      if (@output) {
        unshift @output,
          $self->present_data_for_head1('features',
            $self->text->string('header', 'features') ||
          'This package provides the following features:');
      }
      return join "\n", @output;
    }
    # special-case: present_data_for_function
    elsif ($method eq 'present_data_for_function') {
      my ($name) = @args;
      my @output;
      my ($metadata) = $self->collect('metadata', $name);
      my ($signature) = $self->collect('signature', $name);
      push @output, ($signature, '') if $signature;
      my @data = $self->collect('function', $name);
      return () if !@data;
      push @output, join "\n\n", @data;
      if ($metadata) {
        local $@;
        my $perlcode = do{no warnings; eval $metadata};
        if ($metadata = ($perlcode && ref $perlcode) ? $perlcode : $metadata) {
          if (ref $metadata ne 'HASH') {
            $metadata = {map {split /:\s*/} split /\n+/, $metadata};
          }
          my @lifespan;
          if (my $introduced = $metadata->{introduced}) {
            push @lifespan, "I<Introduced C<$introduced>>";
          }
          if (my $deprecated = $metadata->{deprecated}) {
            push @lifespan, "I<Deprecated C<$deprecated>>";
          }
          if (my $since = $metadata->{since}) {
            @lifespan = ("I<Since C<$since>>");
          }
          push @output, "", join(", ", @lifespan) if @lifespan;
        }
      }
      my @results = $self->data->search({name => $name});
      for my $i (1..(int grep {($$_{list} || '') =~ /^example-\d+/} @results)) {
        push @output, $self->present('example', $i, $name);
      }
      pop @output if $output[-1] eq '';
      return ($self->present_data_for_head2($name, @output));
    }
    # special-case: present_data_for_functions
    elsif ($method eq 'present_data_for_functions') {
      my @output;
      my $type = 'function';
      for my $name (
        sort map /:\s*(\w+)$/,
        grep /^$type/,
        split /\r?\n/,
        join "\n\n", $self->collect('includes')
      )
      {
        push @output, $self->present($type, $name);
      }
      if (@output) {
        unshift @output,
          $self->present_data_for_head1('functions',
            $self->text->string('header', 'functions') ||
          'This package provides the following functions:');
      }
      return join "\n", @output;
    }
    # special-case: present_data_for_head1
    elsif ($method eq 'present_data_for_head1') {
      my ($name, @data) = @args;
      return join("\n", "", "=head1 \U$name", "", grep(defined, @data), "", "=cut");
    }
    # special-case: present_data_for_head2
    elsif ($method eq 'present_data_for_head2') {
      my ($name, @data) = @args;
      return join("\n", "", "=head2 \L$name", "", grep(defined, @data), "", "=cut");
    }
    # special-case: present_data_for_includes
    elsif ($method eq 'present_data_for_includes') {
      return ();
    }
    # special-case: present_data_for_inherits
    elsif ($method eq 'present_data_for_inherits') {
      my @output = map +($self->present_data_for_link($_), ""), grep defined,
        split /\r?\n/, join "\n\n", $self->collect('inherits');
      return () if !@output;
      pop @output;
      return $self->present_data_for_head1('inherits',
        $self->text->string('header', 'inherits') ||
        "This package inherits behaviors from:",
        "",
        @output,
      );
    }
    # special-case: present_data_for_integrates
    elsif ($method eq 'present_data_for_integrates') {
      my @output = map +($self->present_data_for_link($_), ""), grep defined,
        split /\r?\n/, join "\n\n", $self->collect('integrates');
      return () if !@output;
      pop @output;
      return $self->present_data_for_head1('integrates',
        $self->text->string('header', 'integrates') ||
        "This package integrates behaviors from:",
        "",
        @output,
      );
    }
    # special-case: present_data_for_item
    elsif ($method eq 'present_data_for_item') {
      my ($name, $data) = @args;
      return ("=item $name\n", "$data\n");
    }
    # special-case: present_data_for_layout
    elsif ($method eq 'present_data_for_layout') {
      return ();
    }
    # special-case: present_data_for_libraries
    elsif ($method eq 'present_data_for_libraries') {
      my @output = map +($self->present_data_for_link($_), ""), grep defined,
        split /\r?\n/, join "\n\n", $self->collect('libraries');
      return '' if !@output;
      pop @output;
      return $self->present_data_for_head1('libraries',
        $self->text->string('header', 'libraries') ||
        "This package uses type constraints from:",
        "",
        @output,
      );
    }
    # special-case: present_data_for_license
    elsif ($method eq 'present_data_for_license') {
      my @data = $self->collect('license');
      return @data
        ? ($self->present_data_for_head1('license', join "\n\n", @data))
        : ();
    }
    # special-case: present_data_for_link
    elsif ($method eq 'present_data_for_link') {
      return ("L<@{[join('|', @args)]}>");
    }
    # special-case: present_data_for_message
    elsif ($method eq 'present_data_for_message') {
      my ($name) = @args;
      my @output;
      my ($signature) = $self->collect('signature', $name);
      push @output, ($signature, '') if $signature;
      my @data = $self->collect('message', $name);
      return () if !@data;
      my @results = $self->data->search({name => $name});
      for my $i (1..(int grep {($$_{list} || '') =~ /^example-\d+/} @results)) {
        push @output, "B<example $i>", join "\n\n",
          $self->collect('example', $i, $name);
      }
      return (
        $self->present_data_for_over($self->present_data_for_item(
          $name, join "\n\n", @data, @output))
      );
    }
    # special-case: present_data_for_messages
    elsif ($method eq 'present_data_for_messages') {
      my @output;
      my $type = 'message';
      for my $name (
        sort map /:\s*(\w+)$/,
        grep /^$type/,
        split /\r?\n/,
        join "\n\n", $self->collect('includes')
      )
      {
        push @output, $self->present($type, $name);
      }
      if (@output) {
        unshift @output,
          $self->present_data_for_head1('messages',
            $self->text->string('header', 'messages') ||
          'This package provides the following messages:');
      }
      return join "\n", @output;
    }
    # special-case: present_data_for_metadata
    elsif ($method eq 'present_data_for_metadata') {
      return ();
    }
    # special-case: present_data_for_method
    elsif ($method eq 'present_data_for_method') {
      my ($name) = @args;
      my @output;
      my ($metadata) = $self->collect('metadata', $name);
      my ($signature) = $self->collect('signature', $name);
      push @output, ($signature, '') if $signature;
      my @data = $self->collect('method', $name);
      return () if !@data;
      push @output, join "\n\n", @data;
      if ($metadata) {
        local $@;
        my $perlcode = do{no warnings; eval $metadata};
        if ($metadata = ($perlcode && ref $perlcode) ? $perlcode : $metadata) {
          if (ref $metadata ne 'HASH') {
            $metadata = {map {split /:\s*/} split /\n+/, $metadata};
          }
          my @lifespan;
          if (my $introduced = $metadata->{introduced}) {
            push @lifespan, "I<Introduced C<$introduced>>";
          }
          if (my $deprecated = $metadata->{deprecated}) {
            push @lifespan, "I<Deprecated C<$deprecated>>";
          }
          if (my $since = $metadata->{since}) {
            @lifespan = ("I<Since C<$since>>");
          }
          push @output, "", join(", ", @lifespan) if @lifespan;
        }
      }
      my @results = $self->data->search({name => $name});
      for my $i (1..(int grep {($$_{list} || '') =~ /^example-\d+/} @results)) {
        push @output, $self->present('example', $i, $name);
      }
      my @raises = $self->data->search({list => 'raise'});
      for my $raise (grep {$$_{name} =~ /^$name\s/} @raises) {
        my ($sub, $package, $id) = $raise->{name}
          =~ /(\w+)\s+([:\w]*)\s+([\w\.]+)/;
        next unless $sub && $package;
        push @output,
          $self->present_data_for_over(
            $self->present_data_for_item(
              join(" ", "B<may raise> L<$package>", ($id ? "C<$id>" : ())), join("\n\n", @{$raise->{data}})
            ),
          );
      }
      pop @output if $output[-1] eq '';
      return ($self->present_data_for_head2($name, @output));
    }
    # special-case: present_data_for_methods
    elsif ($method eq 'present_data_for_methods') {
      my @output;
      my $type = 'method';
      for my $name (
        sort map /:\s*(\w+)$/,
        grep /^$type/,
        split /\r?\n/,
        join "\n\n", $self->collect('includes')
      )
      {
        push @output, $self->present($type, $name);
      }
      if (@output) {
        unshift @output,
          $self->present_data_for_head1('methods',
            $self->text->string('header', 'methods') ||
          'This package provides the following methods:');
      }
      return join "\n", @output;
    }
    # special-case: present_data_for_name
    elsif ($method eq 'present_data_for_name') {
      my $name = join ' - ', map $self->collect($_), 'name', 'tagline';
      return $name ? ($self->present_data_for_head1('name', $name)) : ();
    }
    # special-case: present_data_for_operator
    elsif ($method eq 'present_data_for_operator') {
      my ($name) = @args;
      my @output;
      my @data = $self->collect('operator', $name);
      return () if !@data;
      my @results = $self->data->search({name => $name});
      for my $i (1..(int grep {($$_{list} || '') =~ /^example-\d+/} @results)) {
        push @output, "B<example $i>", join "\n\n",
          $self->collect('example', $i, $name);
      }
      return (
        $self->present_data_for_over($self->present_data_for_item(
          "operation: C<$name>",
          join "\n\n", @data, @output
        ))
      );
    }
    # special-case: present_data_for_operators
    elsif ($method eq 'present_data_for_operators') {
      my @output;
      my $type = 'operator';
      for my $name (
        sort map $$_{name},
          $self->data->search({list => $type})
      )
      {
        push @output, $self->present($type, $name);
      }
      if (@output) {
        unshift @output,
          $self->present_data_for_head1('operators',
            $self->text->string('header', 'operators') ||
          'This package overloads the following operators:');
      }
      return join "\n", @output;
    }
    # special-case: present_data_for_over
    elsif ($method eq 'present_data_for_over') {
      return join("\n", "", "=over 4", "", grep(defined, @args), "=back");
    }
    # special-case: present_data_for_partial
    elsif ($method eq 'present_data_for_partial') {
      my ($data) = @args;
      my ($file, $method, @args) = @{$data};
      $method = 'present' if lc($method) eq 'pdml';
      my $test = $self->new($file);
      my @output;
      $self->pass((-f $file && (@output = ($test->$method(@args)))),
        "$file: $method: @args");
      return join "\n", @output;
    }
    # special-case: present_data_for_partials
    elsif ($method eq 'present_data_for_partials') {
      my @output;
      push @output, $self->present('partial', $_)
        for map [split /\:\s*/], grep /\w/, grep !/^#/, split /\r?\n/, join "\n\n",
          $self->collect('partials');
      return join "\n", @output;
    }
    # special-case: present_data_for_project
    elsif ($method eq 'present_data_for_project') {
      my @data = $self->collect('project');
      return @data ? ($self->present_data_for_head1('project', join "\n\n", @data)) : ();
    }
    # special-case: present_data_for_signature
    elsif ($method eq 'present_data_for_signature') {
      return ();
    }
    # special-case: present_data_for_synopsis
    elsif ($method eq 'present_data_for_synopsis') {
      my @data = $self->collect('synopsis');
      return @data
        ? ($self->present_data_for_head1('synopsis', join "\n\n", @data))
        : ();
    }
    # special-case: present_data_for_tagline
    elsif ($method eq 'present_data_for_tagline') {
      my @data = $self->collect('tagline');
      return @data
        ? ($self->present_data_for_head1('tagline', join "\n\n", @data))
        : ();
    }
    # special-case: present_data_for_version
    elsif ($method eq 'present_data_for_version') {
      my @data = $self->collect('version');
      return @data
        ? ($self->present_data_for_head1('version', join "\n\n", @data))
        : ();
    }
    else {
      my ($name) = @args;
      my @output;
      for my $item (
        sort map $$_{name},
          $self->data->search({list => $name})
      )
      {
        my @iteration;
        my ($metadata) = $self->collect('metadata', $item);
        my ($signature) = $self->collect('signature', $item);
        push @iteration, ($signature, '') if $signature;
        my @data = do {
          my ($find) = $self->text->find($name, $item); @{$find ? $find->{data} : []};
        };
        next if !@data;
        push @iteration, join "\n\n", @data;
        if ($metadata) {
          local $@;
          my $perlcode = do{no warnings; eval $metadata};
          if ($metadata = ($perlcode && ref $perlcode) ? $perlcode : $metadata) {
            if (ref $metadata ne 'HASH') {
              $metadata = {map {split /:\s*/} split /\n+/, $metadata};
            }
            my @lifespan;
            if (my $introduced = $metadata->{introduced}) {
              push @lifespan, "I<Introduced C<$introduced>>";
            }
            if (my $deprecated = $metadata->{deprecated}) {
              push @lifespan, "I<Deprecated C<$deprecated>>";
            }
            if (my $since = $metadata->{since}) {
              @lifespan = ("I<Since C<$since>>");
            }
            push @output, "", join(", ", @lifespan) if @lifespan;
          }
        }
        my @results = $self->data->search({name => $item});
        for my $i (1..(int grep {($$_{list} || '') =~ /^example-\d+/} @results)) {
          push @iteration, "B<example $i>", $self->collect('example', $i, $item);
        }
        pop @iteration if @output && $output[-1] eq '';
        push @output, (
          $self->present_data_for_over($self->present_data_for_item(
            $item, join "\n\n", @iteration))
        );
      }
      if (@output) {
        unshift @output,
          $self->present_data_for_head1($block,
            $self->text->string('header', $block) ||
          $self->text->string(undef, $block));
      }
      return join "\n", @output;
    }
  }

  # handle deprecated "okay_can"
  if ($method eq 'okay_can') {
    my $result = $self->handler('can_ok', @args);

    return $result;
  }

  # handle deprecated "okay_isa"
  if ($method eq 'okay_isa') {
    my $result = $self->handler('isa_ok', @args);

    return $result;
  }

  # hidden but accessible Test::More functions
  state $test_more_functions = {
    BAIL_OUT => 'BAIL_OUT',
    can_ok => 'can_ok',
    cmp_ok => 'cmp_ok',
    done_testing => 'done_testing',
    eq_array => 'eq_array',
    eq_hash => 'eq_hash',
    eq_set => 'eq_set',
    is_deeply => 'is_deeply',
    isa_ok => 'isa_ok',
    new_ok => 'new_ok',
    ok => 'ok',
    plan => 'plan',
    require_ok => 'require_ok',
    skip => 'skip',
    use_ok => 'use_ok',
  };
  if ($test_more_functions->{$method}) {
    return $self->handler($method, @args);
  }

  # method not found
  my $class = ref($self) || $self;
  return $self->error_on_missing({class => $class, method => $method})->throw;
}

1;



=head1 NAME

Venus::Test - Test Class

=cut

=head1 ABSTRACT

Test Class for Perl 5

=cut

=head1 SYNOPSIS

  package main;

  use Venus::Test;

  my $test = Venus::Test->new('t/Venus_Test.t');

  # $test->for('name');

  # $test->for('tagline');

  # $test->for('abstract');

  # $test->for('synopsis');

  # $test->done;

=cut

=head1 DESCRIPTION

This package aims to provide a standard for documenting L<Venus> derived
software projects, a framework writing tests, test automation, and
documentation generation. This package will automatically exports C<true>,
C<false>, and L</test> keyword functions.

=cut

=head1 SPECIFICATION

This section describes the specification format used by L<Venus::Test> to
generate documentation and automate testing for Perl packages. The
specification uses specially formatted POD blocks that serve as both
human-readable documentation and executable test cases.

B<Note:> When code blocks are evaluated, "redefined" warnings are automatically
disabled.

=head2 Overview

A specification document consists of POD blocks that describe a package. The
blocks are organized into the following categories:

=over 4

=item * B<Required Blocks> - Must be present in every specification

=item * B<Package Structure Blocks> - Define inheritance and dependencies

=item * B<API Blocks> - Document attributes, methods, functions, etc.

=item * B<Supporting Blocks> - Signatures, examples, metadata, and exceptions

=item * B<Feature Blocks> - Special capabilities and operators

=item * B<Document Control Blocks> - Layout and partial inclusions

=item * B<Project Information Blocks> - Authors, license, version

=back

=head2 Quick Reference

  # [required]

  =name
  =abstract
  =tagline
  =synopsis
  =description

  # [optional]

  =encoding
  =includes
  =libraries
  =inherits
  =integrates

  # [optional; repeatable]

  =attribute $name
  =signature $name
  =metadata $name
  =example-$number $name
  =raise $name $error ($id optional)

  =function $name
  =signature $name
  =metadata $name
  =example-$number $name
  =raise $name $error ($id optional)

  =message $name
  =signature $name
  =metadata $name
  =example-$number $name

  =method $name
  =signature $name
  =metadata $name
  =example-$number $name
  =raise $name $error ($id optional)

  =routine $name
  =signature $name
  =metadata $name
  =example-$number $name
  =raise $name $error ($id optional)

  =feature $name
  =metadata $name
  =example-$number $name

  =error $name
  =example-$number $name

  =operator $name
  =example-$number $name

  # [optional]

  =layout
  =partials
  =authors
  =license
  =project
  =version

=head1 REQUIRED BLOCKS

These blocks must be present in every specification document.

=head2 name

  =name

  Example

  =cut

  $test->for('name');

The C<name> block should contain the package name. This is tested for
loadability.

=head2 abstract

  =abstract

  Example Test Documentation

  =cut

  $test->for('abstract');

The C<abstract> block should contain a subtitle describing the package. This is
tested for existence.

=head2 tagline

  =tagline

  Example Class

  =cut

  $test->for('tagline');

The C<tagline> block should contain a 2-5 word description of the package,
which will be prepended to the name as a full description of the package.

=head2 synopsis

  =synopsis

    use Example;

    my $example = Example->new;

    # bless(..., "Example")

  =cut

  $test->for('synopsis', sub {
    my ($tryable) = @_;
    $tryable->result;
  });

The C<synopsis> block should contain the normative usage of the package. This
is tested for existence. This block should be written in a way that allows it
to be evaled successfully and should return a value.

=head2 description

  =description

  This package provides an example class.

  =cut

  $test->for('description');

The C<description> block should contain a description of the package and its
behaviors.

=head1 PACKAGE BLOCKS

These optional blocks define the package's relationships and dependencies.

=head2 includes

  =includes

  function: eg

  method: prepare
  method: execute

  =cut

  $test->for('includes');

The C<includes> block should contain a list of C<function>, C<method>, and/or
C<routine> names in the format of C<$type: $name>. Empty (or commented out)
lines are ignored. Each function, method, and/or routine is tested to be
documented properly, i.e. has the requisite counterparts (e.g. signature and at
least one example block). Also, the package must recognize that each exists.

=head2 libraries

  =libraries

  Venus::Check

  =cut

  $test->for('libraries');

The C<libraries> block should contain a list of packages, each describing how
particular type names used within function and method signatures will be
validated. These packages are tested for loadability.

=head2 inherits

  =inherits

  Venus::Core::Class

  =cut

  $test->for('inherits');

The C<inherits> block should contain a list of parent packages. These packages
are tested for loadability.

=head2 integrates

  =integrates

  Venus::Role::Catchable
  Venus::Role::Throwable

  =cut

  $test->for('integrates');

The C<integrates> block should contain a list of packages that are involved in
the behavior of the main package (typically roles). These packages are not
automatically tested.

=head1 API BLOCKS

These blocks document the package's interface: attributes, methods, functions,
messages, and routines. Each API block follows a common pattern requiring a
description block, a signature block, and at least one example block.

=head2 Common Pattern

All API blocks (attribute, function, message, method, routine) follow this
structure:

  =$type $name               # Description of the $type
  =signature $name           # Type signature
  =metadata $name            # Optional metadata (since, deprecated, etc.)
  =example-1 $name           # First example (required)
  =example-2 $name           # Additional examples (optional)
  =raise $name $error        # Document exceptions (optional)
  =raise $name $error $id    # Exception with named error (optional)
  ...

The C<signature> block should contain a routine signature in the form of
C<$signature : $return_type>, where C<$signature> is a valid typed signature
and C<$return_type> is any valid L<Venus::Check> expression.

The C<example-$number> block should contain valid Perl code and return a value.
Examples can include a "magic" comment to incorporate other code:

=over 4

=item * C<# given: synopsis> - Include the synopsis code

=item * C<# given: example-$number $name> - Include another example's code

=back

=head2 attribute

  =attribute name

  The name attribute is read-write, optional, and holds a string.

  =signature name

    name(string $value) (string)

  =metadata name

  since: 1.0.0

  =example-1 name

    # given: synopsis

    my $name = $example->name;

    # "..."

  =cut

  $test->for('attribute', 'name');

  $test->for('example', 1, 'name', sub {
    my ($tryable) = @_;
    $tryable->result;
  });

The C<attribute> block should contain a description of the attribute and its
purpose. Each attribute is tested and must be recognized to exist.

=head2 method

  =method prepare

  The prepare method prepares for execution.

  =signature prepare

    prepare() (boolean)

  =example-1 prepare

    # given: synopsis

    my $prepare = $example->prepare;

    # "..."

  =cut

  $test->for('method', 'prepare');

  $test->for('example', 1, 'prepare', sub {
    my ($tryable) = @_;
    $tryable->result;
  });

The C<method> block should contain a description of the method and its purpose.
Each method is tested and must be recognized to exist.

=head2 function

  =function eg

  The eg function returns a new instance of Example.

  =signature eg

    eg() (Example)

  =example-1 eg

    # given: synopsis

    my $example = eg();

    # "..."

  =cut

  $test->for('function', 'eg');

  $test->for('example', 1, 'eg', sub {
    my ($tryable) = @_;
    $tryable->result;
  });

The C<function> block should contain a description of the function and its
purpose. Each function is tested and must be recognized to exist.

=head2 routine

  =routine process

  The process routine processes and returns data.

  =signature process

    process(any @args) (any)

  =example-1 process

    # given: synopsis

    my $result = $example->process;

    # "..."

  =cut

  $test->for('routine', 'process');

  $test->for('example', 1, 'process', sub {
    my ($tryable) = @_;
    $tryable->result;
  });

The C<routine> block documents a subroutine that can be called as either a
function or a method. It follows the same pattern as method and function
blocks.

=head2 message

  =message accept

  The accept message represents acceptance.

  =signature accept

    accept(any @args) (string)

  =example-1 accept

    # given: synopsis

    my $accept = $example->accept;

    # "..."

  =cut

  $test->for('message', 'accept');

  $test->for('example', 1, 'accept', sub {
    my ($tryable) = @_;
    $tryable->result;
  });

The C<message> block documents a method that returns a message string, typically
used for error messages or localization. It follows the same pattern as other
API blocks.

=head1 SUPPORTING BLOCKS

These blocks provide additional context for API documentation.

=head2 signature

  =signature prepare

    prepare() (boolean)

  =cut

  $test->for('signature', 'prepare');

The C<signature> block should contain a routine signature in the form of
C<$signature : $return_type>, where C<$signature> is a valid typed signature
and C<$return_type> is any valid L<Venus::Check> expression.

=head2 example

  =example-1 name

    # given: synopsis

    my $name = $example->name;

    # "..."

  =cut

  $test->for('example', 1, 'name', sub {
    my ($tryable) = @_;
    $tryable->result;
  });

The C<example-$number $name> block should contain valid Perl code and return a
value. The block may contain a "magic" comment in the form of C<given:
synopsis> or C<given: example-$number $name> which if present will include the
given code example(s) with the evaluation of the current block.

=head2 metadata

  =metadata prepare

  {since => "1.2.3"}

  =cut

  $test->for('metadata', 'prepare');

The C<metadata $name> block should contain a stringified hashref containing
Perl data structures used in the rendering of the package's documentation.
Metadata can also be specified as flat key/value pairs:

  =metadata prepare

  introduced: 1.2.3
  deprecated: 2.0.0

  =cut

=head2 raise

  =raise execute Venus::Error

    # given: synopsis

    $example->operation; # throw exception

    # Error

  =cut

  $test->for('raise', 'execute', 'Venus::Error', sub {
    my ($tryable) = @_;
    my $error = $tryable->error->result;
    $test->is_error($error);
  });

The C<raise $name $error> block documents an exception that may be thrown by
an API (attribute, function, method, or routine). The parameters are:

=over 4

=item * C<$name> - The name of the attribute, function, method, or routine that may throw the exception.

=item * C<$error> - The error class or package that may be caught (e.g., C<Venus::Error>, C<Example::Error>).

=item * C<$id> (optional) - An error name for further classification within the error class.

=back

The C<$error> represents the exception class that calling code can catch using
a try/catch mechanism. This links the API documentation to error handling
expectations.

An optional C<$id> can be appended to specify a named error. Venus::Error
objects support named errors for further classification:

  =raise execute Venus::Error on.unknown

    # given: synopsis

    $example->operation; # throw exception

    # Error (on.unknown)

  =cut

  $test->for('raise', 'execute', 'Venus::Error', 'on.unknown', sub {
    my ($tryable) = @_;
    my $error = $tryable->error->result;
    $test->is_error($error);
    $test->is($error->name, 'on.unknown');
  });

When C<$id> is provided, it indicates a specific named error within the error
class, allowing for more granular error documentation and handling.

=head1 FEATURE BLOCKS

These blocks document special capabilities, errors, and operator overloads.

=head2 feature

  =feature noop

  This package provides no particularly useful features.

  =example-1 noop

    # given: synopsis

    my $feature = $example->feature;

    # "..."

  =cut

  $test->for('feature');

  $test->for('example', 1, 'noop', sub {
    my ($tryable) = @_;
    $tryable->result;
  });

The C<feature $name> block should contain a description of the feature(s) the
package enables, and can include an C<example-$number $name> block to ensure
the feature described works as expected.

=head2 error

  =error error_on_unknown

  This package may raise an error_on_unknown error.

  =example-1 error_on_unknown

    # given: synopsis

    my $error = $example->catch('error', {
      with => 'error_on_unknown',
    });

    # "..."

  =cut

  $test->for('error', 'error_on_unknown');

  $test->for('example', 1, 'error_on_unknown', sub {
    my ($tryable) = @_;
    $tryable->result;
  });

The C<error $name> block should contain a description of the error the package
may raise, and can include an C<example-$number $name> block to ensure the
error is raised and caught.

=head2 operator

  =operator ("")

  This package overloads the C<""> operator.

  =example-1 ("")

    # given: synopsis

    my $string = "$example";

    # "..."

  =cut

  $test->for('operator', '("")');

  $test->for('example', 1, '("")', sub {
    my ($tryable) = @_;
    $tryable->result;
  });

The C<operator $name> block should contain a description of the overloaded
operation the package performs, and can include an C<example-$number $name>
block to ensure the operation is functioning properly.

=head1 CONTROL BLOCKS

These blocks control how documentation is rendered.

=head2 encoding

  =encoding

  utf8

  =cut

  $test->for('encoding');

The C<encoding> block should contain the appropriate
L<encoding|perlpod/encoding-encodingname>.

=head2 layout

  =layout

  encoding
  name
  synopsis
  description
  attributes: attribute
  authors
  license

  =cut

  $test->for('layout');

The C<layout> block should contain a list of blocks to render using L</render>,
in the order they should be rendered.

=head2 partials

  =partials

  t/path/to/other.t: present: authors
  t/path/to/other.t: present: license

  =cut

  $test->for('partials');

The C<partials> block should contain references to other marked-up test files
in the form of C<$file: $method: $section>, which will call the C<$method> on a
L<Venus::Test> instance for the C<$file> and include the results in-place as
part of the rendering of the current file.

=head1 PROJECT BLOCKS

These blocks provide metadata about the project.

=head2 authors

  =authors

  Awncorp, C<awncorp@cpan.org>

  =cut

  $test->for('authors');

The C<authors> block should contain text describing the authors of the package.

=head2 license

  =license

  No license granted.

  =cut

  $test->for('license');

The C<license> block should contain a link and/or description of the license
governing the package.

=head2 project

  =project

  https://github.com/awncorp/example

  =cut

  $test->for('project');

The C<project> block should contain a description and/or links for the
package's project.

=head2 version

  =version

  1.2.3

  =cut

  $test->for('version');

The C<version> block should contain a valid version number for the package.

=head1 TESTING

This framework provides automated subtests based on the package specification,
but also provides hooks for manual testing when automation is not sufficient.

=head2 Basic Testing

For simple blocks, testing verifies existence:

  $test->for('name');
  $test->for('abstract');
  $test->for('description');

=head2 Testing with Callbacks

Code examples can be evaluated and returned using a callback for further
testing:

  $test->for('synopsis', sub {
    my ($tryable) = @_;

    my $result = $tryable->result;

    # must return truthy to continue
    $result;
  });

=head2 Exception Testing

Because code examples are returned as L<Venus::Try> objects, capturing and
testing exceptions is straightforward:

  $test->for('synopsis', sub {
    my ($tryable) = @_;

    # catch exception thrown by the synopsis
    $tryable->catch('Path::Find::Error', sub {
      return $_[0];
    });

    # test the exception
    my $result = $tryable->result;
    ok $result->isa('Path::Find::Error'), 'exception caught';

    # must return truthy to continue
    $result;
  });

=head2 Testing Examples

The C<example> method evaluates a given example and returns the result as a
L<Venus::Try> object. The first argument is the example number:

  $test->for('example', 1, 'children', sub {
    my ($tryable) = @_;

    my $result = $tryable->result;

    # must return truthy to continue
    $result;
  });

=head2 Testing Features

The C<feature> method evaluates a documented feature and returns the result as
a L<Venus::Try> object:

  $test->for('feature', 'export-path-make', sub {
    my ($tryable) = @_;

    ok my $result = $tryable->result, 'result ok';

    # must return truthy to continue
    $result;
  });

=head2 Benefits

The test automation and documentation generation enabled through this framework
makes it easy to maintain source/test/documentation parity. This also increases
reusability and reduces the need for complicated state and test setup.

=cut

=cut

=head1 ATTRIBUTES

This package has the following attributes:

=cut

=head2 file

  file(string $data) (string)

The file attribute is read-write, accepts C<(string)> values, and is required.

I<Since C<4.15>>

=over 4

=item file example 1

  # given: synopsis

  package main;

  my $set_file = $test->file("t/Venus_Test.t");

  # "t/Venus_Test.t"

=back

=over 4

=item file example 2

  # given: synopsis

  # given: example-1 file

  package main;

  my $get_file = $test->file;

  # "t/Venus_Test.t"

=back

=cut

=head1 INHERITS

This package inherits behaviors from:

L<Venus::Kind>

=cut

=head1 INTEGRATES

This package integrates behaviors from:

L<Venus::Role::Buildable>

L<Venus::Role::Encaseable>

=cut

=head1 FUNCTIONS

This package provides the following functions:

=cut

=head2 test

  test(string $file) (Venus::Test)

The test function is exported automatically and returns a L<Venus::Test> object
for the test file given.

I<Since C<0.09>>

=over 4

=item test example 1

  package main;

  use Venus::Test;

  my $test = test 't/Venus_Test.t';

  # bless(..., "Venus::Test")

=back

=cut

=head1 METHODS

This package provides the following methods:

=cut

=head2 auto

  auto(string $name, any @args) (any)

The auto method gets or sets environment variables that control automatic
behaviors in the testing framework. When called with just a name, it returns
the current value of the corresponding environment variable. When called with
a name and value, it sets the environment variable. The environment variable
name is derived from the name parameter as C<VENUS_TEST_AUTO_${NAME}>.

Supported auto settings:

=over 4

=item * C<bailout> - When truthy, bails out of testing on the first error.

=item * C<render> - When truthy, automatically renders POD when L</done> is
called.

=item * C<scrub> - When truthy, uses L<Venus::Space/scrub> to clean up packages
created in example code for testing.

=item * C<unpatch> - When truthy, uses L<Venus::Space/unpatch> (via
L</unpatch>) to restore any existing monkey-patching on the package associated
with the test.

=back

I<Since C<4.15>>

=over 4

=item auto example 1

  # given: synopsis

  package main;

  my $auto = $test->auto('render');

  # undef

=back

=over 4

=item auto example 2

  # given: synopsis

  package main;

  my $auto = $test->auto('render', 1);

  # 1

=back

=over 4

=item auto example 3

  # given: synopsis

  package main;

  $test->auto('render', 1);

  my $auto = $test->auto('render');

  # 1

=back

=over 4

=item auto example 4

  # given: synopsis

  package main;

  $test->auto('render', 0);

  my $auto = $test->auto('render');

  # 0

=back

=over 4

=item auto example 5

  # given: synopsis

  package main;

  my $auto = $test->auto('bailout');

  # undef

=back

=over 4

=item auto example 6

  # given: synopsis

  package main;

  my $auto = $test->auto('bailout', 1);

  # 1

=back

=over 4

=item auto example 7

  # given: synopsis

  package main;

  $test->auto('bailout', 1);

  my $auto = $test->auto('bailout');

  # 1

=back

=over 4

=item auto example 8

  # given: synopsis

  package main;

  $test->auto('bailout', 0);

  my $auto = $test->auto('bailout');

  # 0

=back

=over 4

=item auto example 9

  # given: synopsis

  package main;

  my $auto = $test->auto('scrub');

  # undef

=back

=over 4

=item auto example 10

  # given: synopsis

  package main;

  my $auto = $test->auto('scrub', 1);

  # 1

=back

=over 4

=item auto example 11

  # given: synopsis

  package main;

  $test->auto('scrub', 1);

  my $auto = $test->auto('scrub');

  # 1

=back

=over 4

=item auto example 12

  # given: synopsis

  package main;

  $test->auto('scrub', 0);

  my $auto = $test->auto('scrub');

  # 0

=back

=over 4

=item auto example 13

  # given: synopsis

  package main;

  my $auto = $test->auto('unpatch');

  # undef

=back

=over 4

=item auto example 14

  # given: synopsis

  package main;

  my $auto = $test->auto('unpatch', 1);

  # 1

=back

=over 4

=item auto example 15

  # given: synopsis

  package main;

  $test->auto('unpatch', 1);

  my $auto = $test->auto('unpatch');

  # 1

=back

=over 4

=item auto example 16

  # given: synopsis

  package main;

  $test->auto('unpatch', 0);

  my $auto = $test->auto('unpatch');

  # 0

=back

=cut

=head2 diag

  diag(string @messages) (any)

The diag method prints diagnostic messages using L<Test::More/diag>.

I<Since C<4.15>>

=over 4

=item diag example 1

  # given: synopsis

  package main;

  my $diag = $test->diag('Test failed due to...');

  # ()

=back

=cut

=head2 done

  done() (any)

The done method dispatches to the L<Test::More/done_testing> operation and
returns the result.

I<Since C<4.15>>

=over 4

=item done example 1

  # given: synopsis

  package main;

  my $done = $test->done;

  # true

=back

=cut

=head2 eval

  eval(string $perl) (any)

The eval method evaluates Perl code and returns the result.

I<Since C<4.15>>

=over 4

=item eval example 1

  # given: synopsis

  package main;

  my $eval = $test->eval('1 + 1');

  # 2

=back

=cut

=head2 explain

  explain(any @args) (any)

The explain method dispatches to the L<Test::More/explain> operation and
returns the result.

I<Since C<4.15>>

=over 4

=item explain example 1

  # given: synopsis

  package main;

  my $explain = $test->explain(123.456);

  # "123.456"

=back

=cut

=head2 fail

  fail(any $data, string $description) (any)

The fail method dispatches to the L<Test::More/ok> operation expecting the
first argument to be falsy and returns the result.

I<Since C<4.15>>

=over 4

=item fail example 1

  # given: synopsis

  package main;

  my $fail = $test->fail(0, 'example-1 fail passed');

  # true

=back

=cut

=head2 for

  for(any @args) (Venus::Test)

The for method dispatches to the L</execute> method using the arguments
provided within a L<subtest|Test::More/subtest> and returns the invocant.

I<Since C<4.15>>

=over 4

=item for example 1

  # given: synopsis

  package main;

  my $for = $test->for('name');

  # bless(..., "Venus::Test")

=back

=over 4

=item for example 2

  # given: synopsis

  package main;

  my $for = $test->for('synopsis');

  # bless(..., "Venus::Test")

=back

=over 4

=item for example 3

  # given: synopsis

  package main;

  my $for = $test->for('synopsis', sub{
    my ($tryable) = @_;
    return $tryable->result;
  });

  # bless(..., "Venus::Test")

=back

=over 4

=item for example 4

  # given: synopsis

  package main;

  my $for = $test->for('example', 1, 'test', sub {
    my ($tryable) = @_;
    return $tryable->result;
  });

  # bless(..., "Venus::Test")

=back

=cut

=head2 gate

  gate(string $note, coderef $code) (Venus::Test)

The gate method creates a new L<Venus::Test> instance with a gate callback that
prevents subtests from running unless the callback returns a truthy value.

I<Since C<4.15>>

=over 4

=item gate example 1

  # given: synopsis

  package main;

  my $test2 = $test->gate('OS is linux', sub {
    $^O eq 'linux'
  });

  # bless(..., "Venus::Test")

=back

=cut

=head2 handler

  handler(any @args) (any)

The handler method dispatches to the L<Test::More> method specified by the
first argument and returns its result.

I<Since C<4.15>>

=over 4

=item handler example 1

  # given: synopsis

  package main;

  my $handler = $test->handler('ok', true);

  # true

=back

=cut

=head2 in

  in(arrayref | hashref | consumes[Venus::Role::Mappable] $collection, any $value) (boolean)

The in method checks if a value exists in a collection (arrayref, hashref, or
L<"mappable"|Venus::Role::Mappable> object) and returns true if the type and
value match.

I<Since C<4.15>>

=over 4

=item in example 1

  # given: synopsis

  package main;

  my $in = $test->in([1, 2, 3], 2);

  # true

=back

=cut

=head2 is

  is(any $data1, any $data2, string $description) (any)

The is method tests for equality using L<Test::More/is>.

I<Since C<4.15>>

=over 4

=item is example 1

  # given: synopsis

  package main;

  my $is = $test->is('hello', 'hello', 'strings match');

  # ()

=back

=cut

=head2 is_arrayref

  is_arrayref(any $data, string @args) (boolean)

The is_arrayref method tests whether the data is an arrayref using
L<Venus/is_arrayref>.

I<Since C<4.15>>

=over 4

=item is_arrayref example 1

  # given: synopsis

  package main;

  my $is_arrayref = $test->is_arrayref([1,2,3], 'valid arrayref');

  # true

=back

=cut

=head2 is_blessed

  is_blessed(any $data, string @args) (boolean)

The is_blessed method tests whether the data is blessed using
L<Venus/is_blessed>.

I<Since C<4.15>>

=over 4

=item is_blessed example 1

  # given: synopsis

  package main;

  my $is_blessed = $test->is_blessed(bless({}), 'valid blessed');

  # true

=back

=cut

=head2 is_boolean

  is_boolean(any $data, string @args) (boolean)

The is_boolean method tests whether the data is a boolean using
L<Venus/is_boolean>.

I<Since C<4.15>>

=over 4

=item is_boolean example 1

  # given: synopsis

  package main;

  require Venus;

  my $is_boolean = $test->is_boolean(true, 'valid boolean');

  # true

=back

=cut

=head2 is_coderef

  is_coderef(any $data, string @args) (boolean)

The is_coderef method tests whether the data is a coderef using
L<Venus/is_coderef>.

I<Since C<4.15>>

=over 4

=item is_coderef example 1

  # given: synopsis

  package main;

  my $is_coderef = $test->is_coderef(sub{}, 'valid coderef');

  # true

=back

=cut

=head2 is_dirhandle

  is_dirhandle(any $data, string @args) (boolean)

The is_dirhandle method tests whether the data is a directory handle using
L<Venus/is_dirhandle>.

I<Since C<4.15>>

=over 4

=item is_dirhandle example 1

  # given: synopsis

  package main;

  opendir(my $dh, '.');
  my $is_dirhandle = $test->is_dirhandle($dh, 'valid dirhandle');

  # true

=back

=cut

=head2 is_enum

  is_enum(any $data, arrayref | hashref $data, string @args) (boolean)

The is_enum method tests whether the data is an enum using L<Venus/is_enum>.

I<Since C<4.15>>

=over 4

=item is_enum example 1

  # given: synopsis

  package main;

  $test->is_enum('light', ['light', 'dark'], 'is in enum');

  # true

=back

=cut

=head2 is_error

  is_error(any $data, string @args) (boolean)

The is_error method tests whether the data is a Venus::Error object using
L<Venus/is_error>.

I<Since C<4.15>>

=over 4

=item is_error example 1

  # given: synopsis

  package main;

  use Venus::Error;

  my $is_error = $test->is_error(Venus::Error->new, 'valid error');

  # true

=back

=cut

=head2 is_false

  is_false(any $data, string @args) (boolean)

The is_false method tests whether the data is a false value using
L<Venus/is_false>.

I<Since C<4.15>>

=over 4

=item is_false example 1

  # given: synopsis

  package main;

  my $is_false = $test->is_false(0, 'valid false');

  # true

=back

=cut

=head2 is_fault

  is_fault(any $data, string @args) (boolean)

The is_fault method tests whether the data is a Venus::Fault object using
L<Venus/is_fault>.

I<Since C<4.15>>

=over 4

=item is_fault example 1

  # given: synopsis

  package main;

  use Venus::Fault;

  my $is_fault = $test->is_fault(Venus::Fault->new, 'valid fault');

  # true

=back

=cut

=head2 is_filehandle

  is_filehandle(any $data, string @args) (boolean)

The is_filehandle method tests whether the data is a file handle using
L<Venus/is_filehandle>.

I<Since C<4.15>>

=over 4

=item is_filehandle example 1

  # given: synopsis

  package main;

  open(my $fh, '<', 't/Venus_Test.t');
  my $is_filehandle = $test->is_filehandle($fh, 'valid filehandle');

  # true

=back

=cut

=head2 is_float

  is_float(any $data, string @args) (boolean)

The is_float method tests whether the data is a float using L<Venus/is_float>.

I<Since C<4.15>>

=over 4

=item is_float example 1

  # given: synopsis

  package main;

  my $is_float = $test->is_float(1.5, 'valid float');

  # true

=back

=cut

=head2 is_glob

  is_glob(any $data, string @args) (boolean)

The is_glob method tests whether the data is a glob reference using
L<Venus/is_glob>.

I<Since C<4.15>>

=over 4

=item is_glob example 1

  # given: synopsis

  package main;

  my $is_glob = $test->is_glob(\*STDOUT, 'valid glob');

  # true

=back

=cut

=head2 is_hashref

  is_hashref(any $data, string @args) (boolean)

The is_hashref method tests whether the data is a hashref using
L<Venus/is_hashref>.

I<Since C<4.15>>

=over 4

=item is_hashref example 1

  # given: synopsis

  package main;

  my $is_hashref = $test->is_hashref({a=>1}, 'valid hashref');

  # true

=back

=cut

=head2 is_number

  is_number(any $data, string @args) (boolean)

The is_number method tests whether the data is a number using
L<Venus/is_number>.

I<Since C<4.15>>

=over 4

=item is_number example 1

  # given: synopsis

  package main;

  my $is_number = $test->is_number(123, 'valid number');

  # true

=back

=cut

=head2 is_object

  is_object(any $data, string @args) (boolean)

The is_object method tests whether the data is an object using
L<Venus/is_object>.

I<Since C<4.15>>

=over 4

=item is_object example 1

  # given: synopsis

  package main;

  my $is_object = $test->is_object(bless({}), 'valid object');

  # true

=back

=cut

=head2 is_package

  is_package(any $data, string @args) (boolean)

The is_package method tests whether the data is a package name using
L<Venus/is_package>.

I<Since C<4.15>>

=over 4

=item is_package example 1

  # given: synopsis

  package main;

  my $is_package = $test->is_package('Venus::Test', 'valid package');

  # true

=back

=cut

=head2 is_reference

  is_reference(any $data, string @args) (boolean)

The is_reference method tests whether the data is a reference using
L<Venus/is_reference>.

I<Since C<4.15>>

=over 4

=item is_reference example 1

  # given: synopsis

  package main;

  my $is_reference = $test->is_reference([], 'valid reference');

  # true

=back

=cut

=head2 is_regexp

  is_regexp(any $data, string @args) (boolean)

The is_regexp method tests whether the data is a regexp using
L<Venus/is_regexp>.

I<Since C<4.15>>

=over 4

=item is_regexp example 1

  # given: synopsis

  package main;

  my $is_regexp = $test->is_regexp(qr/test/, 'valid regexp');

  # true

=back

=cut

=head2 is_scalarref

  is_scalarref(any $data, string @args) (boolean)

The is_scalarref method tests whether the data is a scalar reference using
L<Venus/is_scalarref>.

I<Since C<4.15>>

=over 4

=item is_scalarref example 1

  # given: synopsis

  package main;

  my $scalar = 'hello';
  my $is_scalarref = $test->is_scalarref(\$scalar, 'valid scalarref');

  # true

=back

=cut

=head2 is_string

  is_string(any $data, string @args) (boolean)

The is_string method tests whether the data is a string using
L<Venus/is_string>.

I<Since C<4.15>>

=over 4

=item is_string example 1

  # given: synopsis

  package main;

  my $is_string = $test->is_string('hello', 'valid string');

  # true

=back

=cut

=head2 is_true

  is_true(any $data, string @args) (boolean)

The is_true method tests whether the data is a true value using L<Venus/is_true>.

I<Since C<4.15>>

=over 4

=item is_true example 1

  # given: synopsis

  package main;

  my $is_true = $test->is_true(1, 'valid true');

  # true

=back

=cut

=head2 is_undef

  is_undef(any $data, string @args) (boolean)

The is_undef method tests whether the data is undef using L<Venus/is_undef>.

I<Since C<4.15>>

=over 4

=item is_undef example 1

  # given: synopsis

  package main;

  my $is_undef = $test->is_undef(undef, 'valid undef');

  # true

=back

=cut

=head2 is_value

  is_value(any $data, string @args) (boolean)

The is_value method tests whether the data is a defined value using
L<Venus/is_value>.

I<Since C<4.15>>

=over 4

=item is_value example 1

  # given: synopsis

  package main;

  my $is_value = $test->is_value('hello', 'valid value');

  # true

=back

=cut

=head2 is_yesno

  is_yesno(any $data, string @args) (boolean)

The is_yesno method tests whether the data is a yes/no value using
L<Venus/is_yesno>.

I<Since C<4.15>>

=over 4

=item is_yesno example 1

  # given: synopsis

  package main;

  my $is_yesno = $test->is_yesno(1, 'valid yesno');

  # true

=back

=cut

=head2 isnt

  isnt(any $data1, any $data2, string $description) (any)

The isnt method tests for inequality using L<Test::More/isnt>.

I<Since C<4.15>>

=over 4

=item isnt example 1

  # given: synopsis

  package main;

  my $isnt = $test->isnt('hello', 'world', 'strings differ');

  # ()

=back

=cut

=head2 isnt_arrayref

  isnt_arrayref(any $data, string @args) (boolean)

The isnt_arrayref method tests whether the data is not an arrayref.

I<Since C<4.15>>

=over 4

=item isnt_arrayref example 1

  # given: synopsis

  package main;

  my $isnt_arrayref = $test->isnt_arrayref({}, 'not an arrayref');

  # true

=back

=cut

=head2 isnt_blessed

  isnt_blessed(any $data, string @args) (boolean)

The isnt_blessed method tests whether the data is not a blessed object.

I<Since C<4.15>>

=over 4

=item isnt_blessed example 1

  # given: synopsis

  package main;

  my $isnt_blessed = $test->isnt_blessed('string', 'not blessed');

  # true

=back

=cut

=head2 isnt_boolean

  isnt_boolean(any $data, string @args) (boolean)

The isnt_boolean method tests whether the data is not a boolean.

I<Since C<4.15>>

=over 4

=item isnt_boolean example 1

  # given: synopsis

  package main;

  my $isnt_boolean = $test->isnt_boolean('string', 'not boolean');

  # true

=back

=cut

=head2 isnt_coderef

  isnt_coderef(any $data, string @args) (boolean)

The isnt_coderef method tests whether the data is not a coderef.

I<Since C<4.15>>

=over 4

=item isnt_coderef example 1

  # given: synopsis

  package main;

  my $isnt_coderef = $test->isnt_coderef('string', 'not coderef');

  # true

=back

=cut

=head2 isnt_dirhandle

  isnt_dirhandle(any $data, string @args) (boolean)

The isnt_dirhandle method tests whether the data is not a directory handle.

I<Since C<4.15>>

=over 4

=item isnt_dirhandle example 1

  # given: synopsis

  package main;

  my $isnt_dirhandle = $test->isnt_dirhandle('string', 'not dirhandle');

  # true

=back

=cut

=head2 isnt_enum

  isnt_enum(any $data, arrayref | hashref $data, string @args) (boolean)

The isnt_enum method tests whether the data is not an enum.

I<Since C<4.15>>

=over 4

=item isnt_enum example 1

  # given: synopsis

  package main;

  my $isnt_enum = $test->isnt_enum('light', [], 'not in enum');

  # true

=back

=cut

=head2 isnt_error

  isnt_error(any $data, string @args) (boolean)

The isnt_error method tests whether the data is not a Venus::Error object.

I<Since C<4.15>>

=over 4

=item isnt_error example 1

  # given: synopsis

  package main;

  my $isnt_error = $test->isnt_error('string', 'not error');

  # true

=back

=cut

=head2 isnt_false

  isnt_false(any $data, string @args) (boolean)

The isnt_false method tests whether the data is not a false value.

I<Since C<4.15>>

=over 4

=item isnt_false example 1

  # given: synopsis

  package main;

  my $isnt_false = $test->isnt_false(1, 'not false');

  # true

=back

=cut

=head2 isnt_fault

  isnt_fault(any $data, string @args) (boolean)

The isnt_fault method tests whether the data is not a Venus::Fault object.

I<Since C<4.15>>

=over 4

=item isnt_fault example 1

  # given: synopsis

  package main;

  my $isnt_fault = $test->isnt_fault('string', 'not fault');

  # true

=back

=cut

=head2 isnt_filehandle

  isnt_filehandle(any $data, string @args) (boolean)

The isnt_filehandle method tests whether the data is not a file handle.

I<Since C<4.15>>

=over 4

=item isnt_filehandle example 1

  # given: synopsis

  package main;

  my $isnt_filehandle = $test->isnt_filehandle('string', 'not filehandle');

  # true

=back

=cut

=head2 isnt_float

  isnt_float(any $data, string @args) (boolean)

The isnt_float method tests whether the data is not a float.

I<Since C<4.15>>

=over 4

=item isnt_float example 1

  # given: synopsis

  package main;

  my $isnt_float = $test->isnt_float(123, 'not float');

  # true

=back

=cut

=head2 isnt_glob

  isnt_glob(any $data, string @args) (boolean)

The isnt_glob method tests whether the data is not a glob reference.

I<Since C<4.15>>

=over 4

=item isnt_glob example 1

  # given: synopsis

  package main;

  my $isnt_glob = $test->isnt_glob('string', 'not glob');

  # true

=back

=cut

=head2 isnt_hashref

  isnt_hashref(any $data, string @args) (boolean)

The isnt_hashref method tests whether the data is not a hashref.

I<Since C<4.15>>

=over 4

=item isnt_hashref example 1

  # given: synopsis

  package main;

  my $isnt_hashref = $test->isnt_hashref([], 'not a hashref');

  # true

=back

=cut

=head2 isnt_number

  isnt_number(any $data, string @args) (boolean)

The isnt_number method tests whether the data is not a number.

I<Since C<4.15>>

=over 4

=item isnt_number example 1

  # given: synopsis

  package main;

  my $isnt_number = $test->isnt_number('string', 'not number');

  # true

=back

=cut

=head2 isnt_object

  isnt_object(any $data, string @args) (boolean)

The isnt_object method tests whether the data is not an object.

I<Since C<4.15>>

=over 4

=item isnt_object example 1

  # given: synopsis

  package main;

  my $isnt_object = $test->isnt_object('string', 'not object');

  # true

=back

=cut

=head2 isnt_package

  isnt_package(any $data, string @args) (boolean)

The isnt_package method tests whether the data is not a package name.

I<Since C<4.15>>

=over 4

=item isnt_package example 1

  # given: synopsis

  package main;

  my $isnt_package = $test->isnt_package([], 'not package');

  # true

=back

=cut

=head2 isnt_reference

  isnt_reference(any $data, string @args) (boolean)

The isnt_reference method tests whether the data is not a reference.

I<Since C<4.15>>

=over 4

=item isnt_reference example 1

  # given: synopsis

  package main;

  my $isnt_reference = $test->isnt_reference('string', 'not reference');

  # true

=back

=cut

=head2 isnt_regexp

  isnt_regexp(any $data, string @args) (boolean)

The isnt_regexp method tests whether the data is not a regexp.

I<Since C<4.15>>

=over 4

=item isnt_regexp example 1

  # given: synopsis

  package main;

  my $isnt_regexp = $test->isnt_regexp('string', 'not regexp');

  # true

=back

=cut

=head2 isnt_scalarref

  isnt_scalarref(any $data, string @args) (boolean)

The isnt_scalarref method tests whether the data is not a scalar reference.

I<Since C<4.15>>

=over 4

=item isnt_scalarref example 1

  # given: synopsis

  package main;

  my $isnt_scalarref = $test->isnt_scalarref('string', 'not scalarref');

  # true

=back

=cut

=head2 isnt_string

  isnt_string(any $data, string @args) (boolean)

The isnt_string method tests whether the data is not a string.

I<Since C<4.15>>

=over 4

=item isnt_string example 1

  # given: synopsis

  package main;

  my $isnt_string = $test->isnt_string([], 'not string');

  # true

=back

=cut

=head2 isnt_true

  isnt_true(any $data, string @args) (boolean)

The isnt_true method tests whether the data is not a true value.

I<Since C<4.15>>

=over 4

=item isnt_true example 1

  # given: synopsis

  package main;

  my $isnt_true = $test->isnt_true(0, 'not true');

  # true

=back

=cut

=head2 isnt_undef

  isnt_undef(any $data, string @args) (boolean)

The isnt_undef method tests whether the data is not undef.

I<Since C<4.15>>

=over 4

=item isnt_undef example 1

  # given: synopsis

  package main;

  my $isnt_undef = $test->isnt_undef('string', 'not undef');

  # true

=back

=cut

=head2 isnt_value

  isnt_value(any $data, string @args) (boolean)

The isnt_value method tests whether the data is not a defined value.

I<Since C<4.15>>

=over 4

=item isnt_value example 1

  # given: synopsis

  package main;

  my $isnt_value = $test->isnt_value(undef, 'not value');

  # true

=back

=cut

=head2 isnt_yesno

  isnt_yesno(any $data, string @args) (boolean)

The isnt_yesno method tests whether the data is not a yes/no value.

I<Since C<4.15>>

=over 4

=item isnt_yesno example 1

  # given: synopsis

  package main;

  my $isnt_yesno = $test->isnt_yesno('string', 'not yesno');

  # true

=back

=cut

=head2 lfile

  lfile() (Venus::Path)

The lfile method returns the path to a lib file for the package being tested.

I<Since C<4.15>>

=over 4

=item lfile example 1

  # given: synopsis

  package main;

  my $lfile = $test->lfile;

  # "lib/Venus/Test.pm"

=back

=cut

=head2 like

  like(string $data, string | Venus::Regexp $match, string $description) (any)

The like method dispatches to the L<Test::More/like> operation and returns the
result.

I<Since C<4.15>>

=over 4

=item like example 1

  # given: synopsis

  package main;

  my $like = $test->like('hello world', 'world', 'example-1 like passed');

  # true

=back

=over 4

=item like example 2

  # given: synopsis

  package main;

  my $like = $test->like('hello world', qr/world/, 'example-1 like passed');

  # true

=back

=cut

=head2 mktemp_dir

  mktemp_dir() (Venus::Path)

The mktemp_dir method creates and returns a temporary directory as a
L<Venus::Path> object.

I<Since C<4.15>>

=over 4

=item mktemp_dir example 1

  # given: synopsis

  package main;

  my $mktemp_dir = $test->mktemp_dir;

  # bless(..., "Venus::Path")

=back

=cut

=head2 mktemp_file

  mktemp_file() (Venus::Path)

The mktemp_file method creates and returns a temporary file as a L<Venus::Path>
object.

I<Since C<4.15>>

=over 4

=item mktemp_file example 1

  # given: synopsis

  package main;

  my $mktemp_file = $test->mktemp_file;

  # bless(..., "Venus::Path")

=back

=cut

=head2 new

  new(any @args) (Venus::Test)

The new method constructs an instance of the package.

I<Since C<4.15>>

=over 4

=item new example 1

  package main;

  use Venus::Test;

  my $new = Venus::Test->new;

  # bless(..., "Venus::Test")

=back

=over 4

=item new example 2

  package main;

  use Venus::Test;

  my $new = Venus::Test->new('t/Venus_Test.t');

  # bless(..., "Venus::Test")

=back

=over 4

=item new example 3

  package main;

  use Venus::Test;

  my $new = Venus::Test->new(file => 't/Venus_Test.t');

  # bless(..., "Venus::Test")

=back

=over 4

=item B<may raise> L<Venus::Test::Error> C<on.new>

  package main;

  use Venus::Test;

  my $test = Venus::Test->new('t/data/no-name.t');

  # Error! (on.new)

=back

=cut

=head2 note

  note(string @messages) (any)

The note method prints debugging messages using L<Test::More/diag> and
L<Test::More/explain>.

I<Since C<4.15>>

=over 4

=item note example 1

  # given: synopsis

  package main;

  my $note = $test->note('Example note...');

  # ()

=back

=cut

=head2 only_if

  only_if(string | coderef $code) (Venus::Test)

The only_if method creates a gate that only runs subtests if the callback
returns a truthy value.

I<Since C<4.15>>

=over 4

=item only_if example 1

  # given: synopsis

  package main;

  my $gate = $test->only_if('os_is_mac');

  # bless(..., "Venus::Test")

=back

=cut

=head2 os

  os() (Venus::Os)

The os method returns a L<Venus::Os> object.

I<Since C<4.15>>

=over 4

=item os example 1

  # given: synopsis

  package main;

  my $os = $test->os;

  # bless(..., "Venus::Os")

=back

=cut

=head2 os_is_bsd

  os_is_bsd() (boolean)

The os_is_bsd method returns true if the operating system is BSD.

I<Since C<4.15>>

=over 4

=item os_is_bsd example 1

  # given: synopsis

  package main;

  my $os_is_bsd = $test->os_is_bsd;

  # true

=back

=cut

=head2 os_is_cyg

  os_is_cyg() (boolean)

The os_is_cyg method returns true if the operating system is Cygwin.

I<Since C<4.15>>

=over 4

=item os_is_cyg example 1

  # given: synopsis

  package main;

  my $os_is_cyg = $test->os_is_cyg;

  # true

=back

=cut

=head2 os_is_dos

  os_is_dos() (boolean)

The os_is_dos method returns true if the operating system is DOS.

I<Since C<4.15>>

=over 4

=item os_is_dos example 1

  # given: synopsis

  package main;

  my $os_is_dos = $test->os_is_dos;

  # true

=back

=cut

=head2 os_is_lin

  os_is_lin() (boolean)

The os_is_lin method returns true if the operating system is Linux.

I<Since C<4.15>>

=over 4

=item os_is_lin example 1

  # given: synopsis

  package main;

  my $os_is_lin = $test->os_is_lin;

  # true

=back

=cut

=head2 os_is_mac

  os_is_mac() (boolean)

The os_is_mac method returns true if the operating system is macOS.

I<Since C<4.15>>

=over 4

=item os_is_mac example 1

  # given: synopsis

  package main;

  my $os_is_mac = $test->os_is_mac;

  # true

=back

=cut

=head2 os_is_non

  os_is_non() (boolean)

The os_is_non method returns true if the operating system is non-Unix.

I<Since C<4.15>>

=over 4

=item os_is_non example 1

  # given: synopsis

  package main;

  my $os_is_non = $test->os_is_non;

  # true

=back

=cut

=head2 os_is_sun

  os_is_sun() (boolean)

The os_is_sun method returns true if the operating system is Solaris.

I<Since C<4.15>>

=over 4

=item os_is_sun example 1

  # given: synopsis

  package main;

  my $os_is_sun = $test->os_is_sun;

  # true

=back

=cut

=head2 os_is_vms

  os_is_vms() (boolean)

The os_is_vms method returns true if the operating system is VMS.

I<Since C<4.15>>

=over 4

=item os_is_vms example 1

  # given: synopsis

  package main;

  my $os_is_vms = $test->os_is_vms;

  # true

=back

=cut

=head2 os_is_win

  os_is_win() (boolean)

The os_is_win method returns true if the operating system is Windows.

I<Since C<4.15>>

=over 4

=item os_is_win example 1

  # given: synopsis

  package main;

  my $os_is_win = $test->os_is_win;

  # true

=back

=cut

=head2 os_isnt_bsd

  os_isnt_bsd() (boolean)

The os_isnt_bsd method returns true if the operating system is not BSD.

I<Since C<4.15>>

=over 4

=item os_isnt_bsd example 1

  # given: synopsis

  package main;

  my $os_isnt_bsd = $test->os_isnt_bsd;

  # true

=back

=cut

=head2 os_isnt_cyg

  os_isnt_cyg() (boolean)

The os_isnt_cyg method returns true if the operating system is not Cygwin.

I<Since C<4.15>>

=over 4

=item os_isnt_cyg example 1

  # given: synopsis

  package main;

  my $os_isnt_cyg = $test->os_isnt_cyg;

  # true

=back

=cut

=head2 os_isnt_dos

  os_isnt_dos() (boolean)

The os_isnt_dos method returns true if the operating system is not DOS.

I<Since C<4.15>>

=over 4

=item os_isnt_dos example 1

  # given: synopsis

  package main;

  my $os_isnt_dos = $test->os_isnt_dos;

  # true

=back

=cut

=head2 os_isnt_lin

  os_isnt_lin() (boolean)

The os_isnt_lin method returns true if the operating system is not Linux.

I<Since C<4.15>>

=over 4

=item os_isnt_lin example 1

  # given: synopsis

  package main;

  my $os_isnt_lin = $test->os_isnt_lin;

  # true

=back

=cut

=head2 os_isnt_mac

  os_isnt_mac() (boolean)

The os_isnt_mac method returns true if the operating system is not macOS.

I<Since C<4.15>>

=over 4

=item os_isnt_mac example 1

  # given: synopsis

  package main;

  my $os_isnt_mac = $test->os_isnt_mac;

  # true

=back

=cut

=head2 os_isnt_non

  os_isnt_non() (boolean)

The os_isnt_non method returns true if the operating system is not non-Unix.

I<Since C<4.15>>

=over 4

=item os_isnt_non example 1

  # given: synopsis

  package main;

  my $os_isnt_non = $test->os_isnt_non;

  # true

=back

=cut

=head2 os_isnt_sun

  os_isnt_sun() (boolean)

The os_isnt_sun method returns true if the operating system is not Solaris.

I<Since C<4.15>>

=over 4

=item os_isnt_sun example 1

  # given: synopsis

  package main;

  my $os_isnt_sun = $test->os_isnt_sun;

  # true

=back

=cut

=head2 os_isnt_vms

  os_isnt_vms() (boolean)

The os_isnt_vms method returns true if the operating system is not VMS.

I<Since C<4.15>>

=over 4

=item os_isnt_vms example 1

  # given: synopsis

  package main;

  my $os_isnt_vms = $test->os_isnt_vms;

  # true

=back

=cut

=head2 os_isnt_win

  os_isnt_win() (boolean)

The os_isnt_win method returns true if the operating system is not Windows.

I<Since C<4.15>>

=over 4

=item os_isnt_win example 1

  # given: synopsis

  package main;

  my $os_isnt_win = $test->os_isnt_win;

  # true

=back

=cut

=head2 pass

  pass(any $data, string $description) (any)

The pass method dispatches to the L<Test::More/ok> operation expecting the
first argument to be truthy and returns the result.

I<Since C<4.15>>

=over 4

=item pass example 1

  # given: synopsis

  package main;

  my $fail = $test->pass(1, 'example-1 pass passed');

  # true

=back

=cut

=head2 patch

  patch(string $name, coderef $code) (coderef)

The patch method monkey-patches the named subroutine and returns the original
coderef.

I<Since C<4.15>>

=over 4

=item patch example 1

  # given: synopsis

  package main;

  my $orig = $test->patch('pass', sub {
    return 'patched';
  });

  # sub {...}

  $test->unpatch;

  # bless(..., "Venus::Space")

  $orig

  # sub {...}

=back

=cut

=head2 path

  path(string $path) (Venus::Path)

The path method returns a L<Venus::Path> object for the given path. Defaults to
the test file.

I<Since C<4.15>>

=over 4

=item path example 1

  # given: synopsis

  package main;

  my $path = $test->path('t/Venus_Test.t');

  # bless(..., "Venus::Path")

=back

=cut

=head2 pfile

  pfile() (Venus::Path)

The pfile method returns the path to a pod file for the package being tested.

I<Since C<4.15>>

=over 4

=item pfile example 1

  # given: synopsis

  package main;

  my $pfile = $test->pfile;

  # "lib/Venus/Test.pod"

=back

=cut

=head2 render

  render(string $file) (Venus::Path)

The render method reads the test specification and generates L<perlpod>
documentation and returns a L<Venus::Path> object for the filename provided.

I<Since C<4.15>>

=over 4

=item render example 1

  # given: synopsis

  package main;

  my $path = $test->render('t/path/pod/test');

  # bless(..., "Venus::Path")

=back

=cut

=head2 same

  same(any $data1, any $data2, string $description) (any)

The same method dispatches to the L<Test::More/is_deeply> operation and returns
the result.

I<Since C<4.15>>

=over 4

=item same example 1

  # given: synopsis

  package main;

  my $same = $test->same({1..4}, {1..4}, 'example-1 same passed');

  # true

=back

=cut

=head2 skip

  skip(string $reason) (any)

The skip method dispatches to the L<Test::More/skip> operation with the
C<plan_all> option and returns the result.

I<Since C<4.15>>

=over 4

=item skip example 1

  # given: synopsis

  package main;

  my $skip = $test->skip('Unsupported');

  # true

=back

=cut

=head2 skip_if

  skip_if(string | coderef $code) (Venus::Test)

The skip_if method creates a gate that only runs subtests if the callback
returns a falsy value.

I<Since C<4.15>>

=over 4

=item skip_if example 1

  # given: synopsis

  package main;

  my $gate = $test->skip_if('os_is_mac');

  # bless(..., "Venus::Test")

=back

=cut

=head2 space

  space(string $package) (Venus::Space)

The space method returns a L<Venus::Space> object for the package being tested,
or for the package name provided.

I<Since C<4.15>>

=over 4

=item space example 1

  # given: synopsis

  package main;

  my $space = $test->space;

  # bless(..., "Venus::Space")

=back

=over 4

=item space example 2

  # given: synopsis

  package main;

  my $space = $test->space('Venus::Path');

  # bless(..., "Venus::Space")

=back

=cut

=head2 subtest

  subtest(string $name, coderef $code) (any)

The subtest method runs a subtest using L<Test::More/subtest>. Enclosed tests
maybe be made conditional using a L</gate>, e.g., L</only_if> and L</skip_if>.

I<Since C<4.15>>

=over 4

=item subtest example 1

  # given: synopsis

  package main;

  my $subtest = $test->subtest('test something', sub {
    $test->pass('it works');
  });

  # ()

=back

=cut

=head2 text

  text() (Venus::Text::Pod)

The text method returns a L<Venus::Text::Pod> object using L</file> for parsing
the test specification.

I<Since C<4.15>>

=over 4

=item text example 1

  # given: synopsis

  package main;

  my $text = $test->text;

  # bless(..., "Venus::Text::Pod")

=back

=cut

=head2 tfile

  tfile() (Venus::Path)

The tfile method returns the path to a test file for the package being tested.

I<Since C<4.15>>

=over 4

=item tfile example 1

  # given: synopsis

  package main;

  my $tfile = $test->tfile;

  # "t/Venus_Test.t"

=back

=cut

=head2 type

  type(any $data, string $expression, string @args) (boolean)

The type method performs type assertion using L<Venus::Type> and tests if the
data matches the type expression.

I<Since C<4.15>>

=over 4

=item type example 1

  # given: synopsis

  package main;

  my $type = $test->type([1,2,3], 'arrayref', 'valid arrayref');

  # true

=back

=cut

=head2 unlike

  unlike(string $data, regexp $regex, string $description) (any)

The unlike method tests that a string doesn't match a regex using
L<Test::More/unlike>.

I<Since C<4.15>>

=over 4

=item unlike example 1

  # given: synopsis

  package main;

  my $unlike = $test->unlike('hello', qr/world/, 'does not match');

  # ()

=back

=cut

=head2 unpatch

  unpatch(string @names) (Venus::Space)

The unpatch method undoes patches by name, or undoes all patches if no names
are provided.

I<Since C<4.15>>

=over 4

=item unpatch example 1

  # given: synopsis

  package main;

  $test->patch('pass', sub {'patched'});

  # sub {...}

  my $unpatch = $test->unpatch('pass');

  # bless(..., "Venus::Space")

=back

=cut

=head1 FEATURES

This package provides the following features:

=cut

=over 4

=item collect

  collect(string $name, any @args) (any)

The collect method dispatches to the C<collect_data_for_${name}> method
indictated by the first argument and returns the result. Returns an arrayref in
scalar context, and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # given: synopsis

  package main;

  my ($collect) = $test->collect('name');

  # "Venus::Test"

B<example 2>

  # given: synopsis

  package main;

  my $collect = $test->collect('name');

  # ["Venus::Test"]

=back

=over 4

=item collect_data_for_abstract

  collect_data_for_abstract() (arrayref)

The collect_data_for_abstract method uses L</data> to fetch data for the C<abstract>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =abstract
  #
  # Example Test Documentation
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_abstract = $test->collect_data_for_abstract;

  # ["Example Test Documentation"]

B<example 2>

  # =abstract
  #
  # Example Test Documentation
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_abstract) = $test->collect_data_for_abstract;

  # "Example Test Documentation"

=back

=over 4

=item collect_data_for_attribute

  collect_data_for_attribute(string $name) (arrayref)

The collect_data_for_attribute method uses L</data> to fetch data for the
C<attribute $name> section and returns the data. Returns an arrayref in scalar
context, and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =attribute name
  #
  # The name attribute is read-write, optional, and holds a string.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $name = $example->name;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_attribute = $test->collect_data_for_attribute('name');

  # ["The name attribute is read-write, optional, and holds a string."]

B<example 2>

  # =attribute name
  #
  # The name attribute is read-write, optional, and holds a string.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $name = $example->name;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_attribute) = $test->collect_data_for_attribute('name');

  # "The name attribute is read-write, optional, and holds a string."

=back

=over 4

=item collect_data_for_authors

  collect_data_for_authors() (arrayref)

The collect_data_for_authors method uses L</data> to fetch data for the
C<authors> section and returns the data. Returns an arrayref in scalar context,
and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =authors
  #
  # Awncorp, C<awncorp@cpan.org>
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_authors = $test->collect_data_for_authors;

  # ["Awncorp, C<awncorp@cpan.org>"]

B<example 2>

  # =authors
  #
  # Awncorp, C<awncorp@cpan.org>
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_authors) = $test->collect_data_for_authors;

  # "Awncorp, C<awncorp@cpan.org>"

=back

=over 4

=item collect_data_for_description

  collect_data_for_description() (arrayref)

The collect_data_for_description method uses L</data> to fetch data for the C<description>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =description
  #
  # This package provides an example class.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_description = $test->collect_data_for_description;

  # ["This package provides an example class."]

B<example 2>

  # =description
  #
  # This package provides an example class.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_description) = $test->collect_data_for_description;

  # "This package provides an example class."

=back

=over 4

=item collect_data_for_encoding

  collect_data_for_encoding() (arrayref)

The collect_data_for_encoding method uses L</data> to fetch data for the C<encoding>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =encoding
  #
  # utf8
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_encoding = $test->collect_data_for_encoding;

  # ["UTF8"]

B<example 2>

  # =encoding
  #
  # utf8
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_encoding) = $test->collect_data_for_encoding;

  # "UTF8"

=back

=over 4

=item collect_data_for_error

  collect_data_for_error(string $name) (arrayref)

The collect_data_for_error method uses L</data> to fetch data for the C<error
$name> section and returns the data. Returns an arrayref in scalar context, and
a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =error error_on_unknown
  #
  # This package may raise an error_on_unknown error.
  #
  # =cut
  #
  # =example-1 error_on_unknown
  #
  #   # given: synopsis
  #
  #   my $error = $example->catch('error', {
  #     with => 'error_on_unknown',
  #   });
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_error = $test->collect_data_for_error('error_on_unknown');

  # ["This package may raise an error_on_unknown error."]

B<example 2>

  # =error error_on_unknown
  #
  # This package may raise an error_on_unknown error.
  #
  # =cut
  #
  # =example-1 error_on_unknown
  #
  #   # given: synopsis
  #
  #   my $error = $example->catch('error', {
  #     with => 'error_on_unknown',
  #   });
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_error) = $test->collect_data_for_error('error_on_unknown');

  # "This package may raise an error_on_unknown error."

=back

=over 4

=item collect_data_for_example

  collect_data_for_example(number $numberm string $name) (arrayref)

The collect_data_for_example method uses L</data> to fetch data for the
C<example-$number $name> section and returns the data. Returns an arrayref in
scalar context, and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =attribute name
  #
  # The name attribute is read-write, optional, and holds a string.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $name = $example->name;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_example = $test->collect_data_for_example(1, 'name');

  # ['  # given: synopsis', '  my $name = $example->name;', '  # "..."']

B<example 2>

  # =attribute name
  #
  # The name attribute is read-write, optional, and holds a string.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $name = $example->name;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my @collect_data_for_example = $test->collect_data_for_example(1, 'name');

  # ('  # given: synopsis', '  my $name = $example->name;', '  # "..."')

=back

=over 4

=item collect_data_for_feature

  collect_data_for_feature(string $name) (arrayref)

The collect_data_for_feature method uses L</data> to fetch data for the
C<feature $name> section and returns the data. Returns an arrayref in scalar
context, and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =feature noop
  #
  # This package is no particularly useful features.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_feature = $test->collect_data_for_feature('noop');

  # ["This package is no particularly useful features."]

B<example 2>

  # =feature noop
  #
  # This package is no particularly useful features.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_feature) = $test->collect_data_for_feature('noop');

  # "This package is no particularly useful features."

=back

=over 4

=item collect_data_for_function

  collect_data_for_function(string $name) (arrayref)

The collect_data_for_function method uses L</data> to fetch data for the
C<function $name> section and returns the data. Returns an arrayref in scalar
context, and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =function eg
  #
  # The eg function returns a new instance of Example.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $example = eg();
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_function = $test->collect_data_for_function('eg');

  # ["The eg function returns a new instance of Example."]

B<example 2>

  # =function eg
  #
  # The eg function returns a new instance of Example.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $example = eg();
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_function) = $test->collect_data_for_function('eg');

  # "The eg function returns a new instance of Example."

=back

=over 4

=item collect_data_for_includes

  collect_data_for_includes() (arrayref)

The collect_data_for_includes method uses L</data> to fetch data for the
C<includes> section and returns the data. Returns an arrayref in scalar
context, and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =includes
  #
  # function: eg
  #
  # method: prepare
  # method: execute
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_includes = $test->collect_data_for_includes;

  # ["function: eg", "method: prepare", "method: execute"]

B<example 2>

  # =includes
  #
  # function: eg
  #
  # method: prepare
  # method: execute
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my @collect_data_for_includes = $test->collect_data_for_includes;

  # ("function: eg", "method: prepare", "method: execute")

=back

=over 4

=item collect_data_for_inherits

  collect_data_for_inherits() (arrayref)

The collect_data_for_inherits method uses L</data> to fetch data for the C<inherits>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =inherits
  #
  # Venus::Core::Class
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_inherits = $test->collect_data_for_inherits;

  # ["Venus::Core::Class"]

B<example 2>

  # =inherits
  #
  # Venus::Core::Class
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_inherits) = $test->collect_data_for_inherits;

  # "Venus::Core::Class"

=back

=over 4

=item collect_data_for_integrates

  collect_data_for_integrates() (arrayref)

The collect_data_for_integrates method uses L</data> to fetch data for the C<integrates>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =integrates
  #
  # Venus::Role::Catchable
  # Venus::Role::Throwable
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_integrates = $test->collect_data_for_integrates;

  # ["Venus::Role::Catchable\nVenus::Role::Throwable"]

B<example 2>

  # =integrates
  #
  # Venus::Role::Catchable
  # Venus::Role::Throwable
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_integrates) = $test->collect_data_for_integrates;

  # "Venus::Role::Catchable\nVenus::Role::Throwable"

=back

=over 4

=item collect_data_for_layout

  collect_data_for_layout() (arrayref)

The collect_data_for_layout method uses L</data> to fetch data for the C<layout>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =layout
  #
  # encoding
  # name
  # synopsis
  # description
  # attributes: attribute
  # authors
  # license
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_layout = $test->collect_data_for_layout;

  # ["encoding\nname\nsynopsis\ndescription\nattributes: attribute\nauthors\nlicense"]

B<example 2>

  # =layout
  #
  # encoding
  # name
  # synopsis
  # description
  # attributes: attribute
  # authors
  # license
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_layout) = $test->collect_data_for_layout;

  # "encoding\nname\nsynopsis\ndescription\nattributes: attribute\nauthors\nlicense"

=back

=over 4

=item collect_data_for_libraries

  collect_data_for_libraries() (arrayref)

The collect_data_for_libraries method uses L</data> to fetch data for the C<libraries>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =libraries
  #
  # Venus::Check
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_libraries = $test->collect_data_for_libraries;

  # ["Venus::Check"]

B<example 2>

  # =libraries
  #
  # Venus::Check
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_libraries) = $test->collect_data_for_libraries;

  # "Venus::Check"

=back

=over 4

=item collect_data_for_license

  collect_data_for_license() (arrayref)

The collect_data_for_license method uses L</data> to fetch data for the C<license>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =license
  #
  # No license granted.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_license = $test->collect_data_for_license;

  # ["No license granted."]

B<example 2>

  # =license
  #
  # No license granted.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_license) = $test->collect_data_for_license;

  # "No license granted."

=back

=over 4

=item collect_data_for_message

  collect_data_for_message(string $name) (arrayref)

The collect_data_for_message method uses L</data> to fetch data for the
C<message $name> section and returns the data. Returns an arrayref in scalar
context, and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =message accept
  #
  # The accept message represents acceptance.
  #
  # =cut
  #
  # =example-1 accept
  #
  #   # given: synopsis
  #
  #   my $accept = $example->accept;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_message = $test->collect_data_for_message('accept');

  # ["The accept message represents acceptance."]

B<example 2>

  # =message accept
  #
  # The accept message represents acceptance.
  #
  # =cut
  #
  # =example-1 accept
  #
  #   # given: synopsis
  #
  #   my $accept = $example->accept;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_message) = $test->collect_data_for_message('accept');

  # "The accept message represents acceptance."

=back

=over 4

=item collect_data_for_metadata

  collect_data_for_metadata(string $name) (arrayref)

The collect_data_for_metadata method uses L</data> to fetch data for the C<metadata $name>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =method prepare
  #
  # The prepare method prepares for execution.
  #
  # =cut
  #
  # =metadata prepare
  #
  # {since => 1.2.3}
  #
  # =cut
  #
  # =example-1 prepare
  #
  #   # given: synopsis
  #
  #   my $prepare = $example->prepare;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_metadata = $test->collect_data_for_metadata('prepare');

  # ["{since => 1.2.3}"]

B<example 2>

  # =method prepare
  #
  # The prepare method prepares for execution.
  #
  # =cut
  #
  # =metadata prepare
  #
  # {since => 1.2.3}
  #
  # =cut
  #
  # =example-1 prepare
  #
  #   # given: synopsis
  #
  #   my $prepare = $example->prepare;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_metadata) = $test->collect_data_for_metadata('prepare');

  # "{since => 1.2.3}"

=back

=over 4

=item collect_data_for_method

  collect_data_for_method(string $name) (arrayref)

The collect_data_for_method method uses L</data> to fetch data for the C<method $name>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =method execute
  #
  # The execute method executes the logic.
  #
  # =cut
  #
  # =metadata execute
  #
  # {since => 1.2.3}
  #
  # =cut
  #
  # =example-1 execute
  #
  #   # given: synopsis
  #
  #   my $execute = $example->execute;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_method = $test->collect_data_for_method('execute');

  # ["The execute method executes the logic."]

B<example 2>

  # =method execute
  #
  # The execute method executes the logic.
  #
  # =cut
  #
  # =metadata execute
  #
  # {since => 1.2.3}
  #
  # =cut
  #
  # =example-1 execute
  #
  #   # given: synopsis
  #
  #   my $execute = $example->execute;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_method) = $test->collect_data_for_method('execute');

  # "The execute method executes the logic."

=back

=over 4

=item collect_data_for_name

  collect_data_for_name() (arrayref)

The collect_data_for_name method uses L</data> to fetch data for the C<name>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =name

  # Example

  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_name = $test->collect_data_for_name;

  # ["Example"]

B<example 2>

  # =name

  # Example

  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_name) = $test->collect_data_for_name;

  # "Example"

=back

=over 4

=item collect_data_for_operator

  collect_data_for_operator(string $name) (arrayref)

The collect_data_for_operator method uses L</data> to fetch data for the C<operator $name>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =operator ("")
  #
  # This package overloads the C<""> operator.
  #
  # =cut
  #
  # =example-1 ("")
  #
  #   # given: synopsis
  #
  #   my $string = "$example";
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_operator = $test->collect_data_for_operator('("")');

  # ['This package overloads the C<""> operator.']

B<example 2>

  # =operator ("")
  #
  # This package overloads the C<""> operator.
  #
  # =cut
  #
  # =example-1 ("")
  #
  #   # given: synopsis
  #
  #   my $string = "$example";
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_operator) = $test->collect_data_for_operator('("")');

  # 'This package overloads the C<""> operator.'

=back

=over 4

=item collect_data_for_partials

  collect_data_for_partials() (arrayref)

The collect_data_for_partials method uses L</data> to fetch data for the C<partials>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =partials
  #
  # t/path/to/other.t: present: authors
  # t/path/to/other.t: present: license
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_partials = $test->collect_data_for_partials;

  # ["t/path/to/other.t: present: authors\nt/path/to/other.t: present: license"]

B<example 2>

  # =partials
  #
  # t/path/to/other.t: present: authors
  # t/path/to/other.t: present: license
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_partials) = $test->collect_data_for_partials;

  # "t/path/to/other.t: present: authors\nt/path/to/other.t: present: license"

=back

=over 4

=item collect_data_for_project

  collect_data_for_project() (arrayref)

The collect_data_for_project method uses L</data> to fetch data for the C<project>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =project
  #
  # https://github.com/awncorp/example
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_project = $test->collect_data_for_project;

  # ["https://github.com/awncorp/example"]

B<example 2>

  # =project
  #
  # https://github.com/awncorp/example
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_project) = $test->collect_data_for_project;

  # "https://github.com/awncorp/example"

=back

=over 4

=item collect_data_for_signature

  collect_data_for_signature(string $name) (arrayref)

The collect_data_for_signature method uses L</data> to fetch data for the C<signature $name>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =method execute
  #
  # The execute method executes the logic.
  #
  # =cut
  #
  # =signature execute
  #
  #   execute() (boolean)
  #
  # =cut
  #
  # =metadata execute
  #
  # {since => 1.2.3}
  #
  # =cut
  #
  # =example-1 execute
  #
  #   # given: synopsis
  #
  #   my $execute = $example->execute;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_signature = $test->collect_data_for_signature('execute');

  # ["  execute() (boolean)"]

B<example 2>

  # =method execute
  #
  # The execute method executes the logic.
  #
  # =cut
  #
  # =signature execute
  #
  #   execute() (boolean)
  #
  # =cut
  #
  # =metadata execute
  #
  # {since => 1.2.3}
  #
  # =cut
  #
  # =example-1 execute
  #
  #   # given: synopsis
  #
  #   my $execute = $example->execute;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_signature) = $test->collect_data_for_signature('execute');

  # "  execute() (boolean)"

=back

=over 4

=item collect_data_for_synopsis

  collect_data_for_synopsis() (arrayref)

The collect_data_for_synopsis method uses L</data> to fetch data for the C<synopsis>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =synopsis
  #
  #   use Example;
  #
  #   my $example = Example->new;
  #
  #   # bless(..., "Example")
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_synopsis = $test->collect_data_for_synopsis;

  # ['  use Example;', '  my $example = Example->new;', '  # bless(..., "Example")']

B<example 2>

  # =synopsis
  #
  #   use Example;
  #
  #   my $example = Example->new;
  #
  #   # bless(..., "Example")
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my @collect_data_for_synopsis = $test->collect_data_for_synopsis;

  # ('  use Example;', '  my $example = Example->new;', '  # bless(..., "Example")')

=back

=over 4

=item collect_data_for_tagline

  collect_data_for_tagline() (arrayref)

The collect_data_for_tagline method uses L</data> to fetch data for the C<tagline>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =tagline
  #
  # Example Class
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_tagline = $test->collect_data_for_tagline;

  # ["Example Class"]

B<example 2>

  # =tagline
  #
  # Example Class
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_tagline) = $test->collect_data_for_tagline;

  # "Example Class"

=back

=over 4

=item collect_data_for_version

  collect_data_for_version() (arrayref)

The collect_data_for_version method uses L</data> to fetch data for the C<version>
section and returns the data. Returns an arrayref in scalar context, and a list
in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =version
  #
  # 1.2.3
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $collect_data_for_version = $test->collect_data_for_version;

  # ["1.2.3"]

B<example 2>

  # =version
  #
  # 1.2.3
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my ($collect_data_for_version) = $test->collect_data_for_version;

  # "1.2.3"

=back

=over 4

=item data

  data() (Venus::Text::Pod)

The data method returns a L<Venus::Text::Pod> object using L</file> for parsing
the test specification.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # given: synopsis

  package main;

  my $data = $test->data;

  # bless(..., "Venus::Text::Pod")

=back

=over 4

=item execute

  execute(string $name, any @args) (boolean)

The execute method dispatches to the C<execute_data_for_${name}> method
indictated by the first argument and returns the result. Returns an arrayref in
scalar context, and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # given: synopsis

  package main;

  my $execute = $test->execute('name');

  # true

B<example 2>

  # given: synopsis

  package main;

  my $execute = $test->execute('name', sub {
    my ($data) = @_;

    my $result = $data->[0] eq 'Venus::Test' ? true : false;

    $self->pass($result, 'name set as Venus::Test');

    return $result;
  });

  # true

=back

=over 4

=item execute_test_for_abstract

  execute_test_for_abstract() (arrayref)

The execute_test_for_abstract method tests a documentation block for the C<abstract> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =abstract
  #
  # Example Test Documentation
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_abstract = $test->execute_test_for_abstract;

  # true

=back

=over 4

=item execute_test_for_attribute

  execute_test_for_attribute(string $name) (arrayref)

The execute_test_for_attribute method tests a documentation block for the C<attribute $name> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =attribute name
  #
  # The name attribute is read-write, optional, and holds a string.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $name = $example->name;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_attribute = $test->execute_test_for_attribute('name');

  # true

=back

=over 4

=item execute_test_for_authors

  execute_test_for_authors() (arrayref)

The execute_test_for_authors method tests a documentation block for the C<authors> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =authors
  #
  # Awncorp, C<awncorp@cpan.org>
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_authors = $test->execute_test_for_authors;

  # true

=back

=over 4

=item execute_test_for_description

  execute_test_for_description() (arrayref)

The execute_test_for_description method tests a documentation block for the C<description> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =description
  #
  # This package provides an example class.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_description = $test->execute_test_for_description;

  # true

=back

=over 4

=item execute_test_for_encoding

  execute_test_for_encoding() (arrayref)

The execute_test_for_encoding method tests a documentation block for the C<encoding> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =encoding
  #
  # utf8
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_encoding = $test->execute_test_for_encoding;

  # true

=back

=over 4

=item execute_test_for_error

  execute_test_for_error(string $name) (arrayref)

The execute_test_for_error method tests a documentation block for the C<error $name> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =error error_on_unknown
  #
  # This package may raise an error_on_unknown error.
  #
  # =cut
  #
  # =example-1 error_on_unknown
  #
  #   # given: synopsis
  #
  #   my $error = $example->catch('error', {
  #     with => 'error_on_unknown',
  #   });
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_error = $test->execute_test_for_error('error_on_unknown');

  # true

=back

=over 4

=item execute_test_for_example

  execute_test_for_example(number $numberm string $name) (arrayref)

The execute_test_for_example method tests a documentation block for the C<example-$number $name> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =attribute name
  #
  # The name attribute is read-write, optional, and holds a string.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $name = $example->name;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_example = $test->execute_test_for_example(1, 'name');

  # true

=back

=over 4

=item execute_test_for_feature

  execute_test_for_feature(string $name) (arrayref)

The execute_test_for_feature method tests a documentation block for the C<feature $name> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =feature noop
  #
  # This package is no particularly useful features.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_feature = $test->execute_test_for_feature('noop');

  # true

=back

=over 4

=item execute_test_for_function

  execute_test_for_function(string $name) (arrayref)

The execute_test_for_function method tests a documentation block for the C<function $name> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =function eg
  #
  # The eg function returns a new instance of Example.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $example = eg();
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_function = $test->execute_test_for_function('eg');

  # true

=back

=over 4

=item execute_test_for_includes

  execute_test_for_includes() (arrayref)

The execute_test_for_includes method tests a documentation block for the C<includes> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =includes
  #
  # function: eg
  #
  # method: prepare
  # method: execute
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_includes = $test->execute_test_for_includes;

  # true

=back

=over 4

=item execute_test_for_inherits

  execute_test_for_inherits() (arrayref)

The execute_test_for_inherits method tests a documentation block for the C<inherits> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =inherits
  #
  # Venus::Core::Class
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_inherits = $test->execute_test_for_inherits;

  # true

=back

=over 4

=item execute_test_for_integrates

  execute_test_for_integrates() (arrayref)

The execute_test_for_integrates method tests a documentation block for the C<integrates> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =integrates
  #
  # Venus::Role::Catchable
  # Venus::Role::Throwable
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_integrates = $test->execute_test_for_integrates;

  # true

=back

=over 4

=item execute_test_for_layout

  execute_test_for_layout() (arrayref)

The execute_test_for_layout method tests a documentation block for the C<layout> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =layout
  #
  # encoding
  # name
  # synopsis
  # description
  # attributes: attribute
  # authors
  # license
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_layout = $test->execute_test_for_layout;

  # true

=back

=over 4

=item execute_test_for_libraries

  execute_test_for_libraries() (arrayref)

The execute_test_for_libraries method tests a documentation block for the C<libraries> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =libraries
  #
  # Venus::Check
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_libraries = $test->execute_test_for_libraries;

  # true

=back

=over 4

=item execute_test_for_license

  execute_test_for_license() (arrayref)

The execute_test_for_license method tests a documentation block for the C<license> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =license
  #
  # No license granted.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_license = $test->execute_test_for_license;

  # true

=back

=over 4

=item execute_test_for_message

  execute_test_for_message(string $name) (arrayref)

The execute_test_for_message method tests a documentation block for the C<message $name> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =message accept
  #
  # The accept message represents acceptance.
  #
  # =cut
  #
  # =example-1 accept
  #
  #   # given: synopsis
  #
  #   my $accept = $example->accept;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_message = $test->execute_test_for_message('accept');

  # true

=back

=over 4

=item execute_test_for_metadata

  execute_test_for_metadata(string $name) (arrayref)

The execute_test_for_metadata method tests a documentation block for the C<metadata $name> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =method prepare
  #
  # The prepare method prepares for execution.
  #
  # =cut
  #
  # =metadata prepare
  #
  # {since => 1.2.3}
  #
  # =cut
  #
  # =example-1 prepare
  #
  #   # given: synopsis
  #
  #   my $prepare = $example->prepare;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_metadata = $test->execute_test_for_metadata('prepare');

  # true

=back

=over 4

=item execute_test_for_method

  execute_test_for_method(string $name) (arrayref)

The execute_test_for_method method tests a documentation block for the C<method $name> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =method execute
  #
  # The execute method executes the logic.
  #
  # =cut
  #
  # =metadata execute
  #
  # {since => 1.2.3}
  #
  # =cut
  #
  # =example-1 execute
  #
  #   # given: synopsis
  #
  #   my $execute = $example->execute;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_method = $test->execute_test_for_method('execute');

  # true

=back

=over 4

=item execute_test_for_name

  execute_test_for_name() (arrayref)

The execute_test_for_name method tests a documentation block for the C<name> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =name

  # Example

  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_name = $test->execute_test_for_name;

  # true

=back

=over 4

=item execute_test_for_operator

  execute_test_for_operator(string $name) (arrayref)

The execute_test_for_operator method tests a documentation block for the C<operator $name> section and returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =operator ("")
  #
  # This package overloads the C<""> operator.
  #
  # =cut
  #
  # =example-1 ("")
  #
  #   # given: synopsis
  #
  #   my $string = "$example";
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_operator = $test->execute_test_for_operator('("")');

  # true

=back

=over 4

=item execute_test_for_raise

  execute_test_for_raise(string $name, string $class, string $id) (arrayref)

The execute_test_for_raise method tests a documentation block for the C<raise
$name $error $id> section and returns the result.

I<Introduced C<4.15>>

B<example 1>

  # =raise execute Venus::Error on.unknown
  #
  #   # given: synopsis
  #
  #   $example->operation; # throw exception
  #
  #   # Error (on.unknown)
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $execute_test_for_raise = $test->execute_test_for_raise('execute', 'Venus::Error', 'on.unknown');

  # true

=back

=over 4

=item more

  more(any @args) (any)

The more method dispatches to the L<Test::More> method specified by the first
argument and returns its result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # given: synopsis

  package main;

  my $more = $test->more('ok', true);

  # true

=back

=over 4

=item okay

  okay(any $data, string $description) (any)

The okay method dispatches to the L<Test::More/ok> operation and returns the
result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # given: synopsis

  package main;

  my $okay = $test->okay(1, 'example-1 okay passed');

  # true

B<example 2>

  # given: synopsis

  package main;

  my $okay = $test->okay(!0, 'example-1 okay passed');

  # true

=back

=over 4

=item okay_can

  okay_can(string $name, string @args) (any)

The okay_can method dispatches to the L<Test::More/can_ok> operation and
returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # given: synopsis

  package main;

  my $okay_can = $test->okay_can('Venus::Test', 'diag');

  # true

=back

=over 4

=item okay_isa

  okay_isa(string $name, string $base) (any)

The okay_isa method dispatches to the L<Test::More/isa_ok> operation and
returns the result.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # given: synopsis

  package main;

  my $okay_isa = $test->okay_isa('Venus::Test', 'Venus::Kind');

  # true

=back

=over 4

=item perform

  perform(string $name, any @args) (boolean)

The perform method dispatches to the C<perform_data_for_${name}> method
indictated by the first argument and returns the result. Returns an arrayref in
scalar context, and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # given: synopsis

  package main;

  my $data = $test->collect('name');

  my $perform = $test->perform('name', $data);

  # true

=back

=over 4

=item perform_test_for_abstract

  perform_test_for_abstract(arrayref $data) (boolean)

The perform_data_for_abstract method performs an overridable test for the C<abstract> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_abstract {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=abstract content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_abstract;

  my $perform_test_for_abstract = $test->perform_test_for_abstract(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_attribute

  perform_test_for_attribute(string $name, arrayref $data) (boolean)

The perform_data_for_attribute method performs an overridable test for the C<attribute $name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_attribute {
    my ($self, $name, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=attribute $name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_attribute('name');

  my $perform_test_for_attribute = $test->perform_test_for_attribute(
    'name', $data,
  );

  # true

=back

=over 4

=item perform_test_for_authors

  perform_test_for_authors(arrayref $data) (boolean)

The perform_data_for_authors method performs an overridable test for the C<authors> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_authors {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=authors content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_authors;

  my $perform_test_for_authors = $test->perform_test_for_authors(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_description

  perform_test_for_description(arrayref $data) (boolean)

The perform_data_for_description method performs an overridable test for the C<description> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_description {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=description content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_description;

  my $perform_test_for_description = $test->perform_test_for_description(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_encoding

  perform_test_for_encoding(arrayref $data) (boolean)

The perform_data_for_encoding method performs an overridable test for the C<encoding> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_encoding {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=encoding content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_encoding;

  my $perform_test_for_encoding = $test->perform_test_for_encoding(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_error

  perform_test_for_error(arrayref $data) (boolean)

The perform_data_for_error method performs an overridable test for the C<error $name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_error {
    my ($self, $name, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=error $name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_error('error_on_unknown');

  my $perform_test_for_error = $test->perform_test_for_error(
    'error_on_unknown', $data,
  );

  # true

=back

=over 4

=item perform_test_for_example

  perform_test_for_example(arrayref $data) (boolean)

The perform_data_for_example method performs an overridable test for the C<example-$number $name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_example {
    my ($self, $number, $name, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=example-$number $name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_example(1, 'execute');

  my $perform_test_for_example = $test->perform_test_for_example(
    1, 'execute', $data,
  );

  # true

=back

=over 4

=item perform_test_for_feature

  perform_test_for_feature(arrayref $data) (boolean)

The perform_data_for_feature method performs an overridable test for the C<feature $name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_feature {
    my ($self, $name, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=feature $name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_feature('noop');

  my $perform_test_for_feature = $test->perform_test_for_feature(
    'noop', $data,
  );

  # true

=back

=over 4

=item perform_test_for_function

  perform_test_for_function(arrayref $data) (boolean)

The perform_data_for_function method performs an overridable test for the C<function $name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_function {
    my ($self, $name, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=function $name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_function('eg');

  my $perform_test_for_function = $test->perform_test_for_function(
    'eg', $data,
  );

  # true

=back

=over 4

=item perform_test_for_includes

  perform_test_for_includes(arrayref $data) (boolean)

The perform_data_for_includes method performs an overridable test for the C<includes> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_includes {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=includes content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_includes;

  my $perform_test_for_includes = $test->perform_test_for_includes(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_inherits

  perform_test_for_inherits(arrayref $data) (boolean)

The perform_data_for_inherits method performs an overridable test for the C<inherits> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_inherits {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=inherits content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_inherits;

  my $perform_test_for_inherits = $test->perform_test_for_inherits(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_integrates

  perform_test_for_integrates(arrayref $data) (boolean)

The perform_data_for_integrates method performs an overridable test for the C<integrates> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_integrates {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=integrates content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_integrates;

  my $perform_test_for_integrates = $test->perform_test_for_integrates(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_layout

  perform_test_for_layout(arrayref $data) (boolean)

The perform_data_for_layout method performs an overridable test for the C<layout> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_layout {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=layout content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_layout;

  my $perform_test_for_layout = $test->perform_test_for_layout(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_libraries

  perform_test_for_libraries(arrayref $data) (boolean)

The perform_data_for_libraries method performs an overridable test for the C<libraries> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_libraries {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=libraries content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_libraries;

  my $perform_test_for_libraries = $test->perform_test_for_libraries(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_license

  perform_test_for_license(arrayref $data) (boolean)

The perform_data_for_license method performs an overridable test for the C<license> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_license {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=license content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_license;

  my $perform_test_for_license = $test->perform_test_for_license(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_message

  perform_test_for_message(arrayref $data) (boolean)

The perform_data_for_message method performs an overridable test for the C<message $name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_message {
    my ($self, $name, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=message $name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_message('accept');

  my $perform_test_for_message = $test->perform_test_for_message(
    'accept', $data,
  );

  # true

=back

=over 4

=item perform_test_for_metadata

  perform_test_for_metadata(arrayref $data) (boolean)

The perform_data_for_metadata method performs an overridable test for the C<metadata $name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_metadata {
    my ($self, $name, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=metadata $name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_metadata('execute');

  my $perform_test_for_metadata = $test->perform_test_for_metadata(
    'execute', $data,
  );

  # true

=back

=over 4

=item perform_test_for_method

  perform_test_for_method(arrayref $data) (boolean)

The perform_data_for_method method performs an overridable test for the C<method $name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_method {
    my ($self, $name, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=method $name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_method('execute');

  my $perform_test_for_method = $test->perform_test_for_method(
    'execute', $data,
  );

  # true

=back

=over 4

=item perform_test_for_name

  perform_test_for_name(arrayref $data) (boolean)

The perform_data_for_name method performs an overridable test for the C<name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_name {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_name;

  my $perform_test_for_name = $test->perform_test_for_name(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_operator

  perform_test_for_operator(arrayref $data) (boolean)

The perform_data_for_operator method performs an overridable test for the C<operator $name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_operator {
    my ($self, $name, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=operator $name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_operator('("")');

  my $perform_test_for_operator = $test->perform_test_for_operator(
    '("")', $data,
  );

  # true

=back

=over 4

=item perform_test_for_partials

  perform_test_for_partials(arrayref $data) (boolean)

The perform_data_for_partials method performs an overridable test for the C<partials> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_partials {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=partials content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_partials;

  my $perform_test_for_partials = $test->perform_test_for_partials(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_project

  perform_test_for_project(arrayref $data) (boolean)

The perform_data_for_project method performs an overridable test for the C<project> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_project {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=project content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_project;

  my $perform_test_for_project = $test->perform_test_for_project(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_signature

  perform_test_for_signature(arrayref $data) (boolean)

The perform_data_for_signature method performs an overridable test for the C<signature $name> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_signature {
    my ($self, $name, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=signature $name content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_signature('execute');

  my $perform_test_for_signature = $test->perform_test_for_signature(
    'execute', $data,
  );

  # true

=back

=over 4

=item perform_test_for_synopsis

  perform_test_for_synopsis(arrayref $data) (boolean)

The perform_data_for_synopsis method performs an overridable test for the C<synopsis> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_synopsis {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=synopsis content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_synopsis;

  my $perform_test_for_synopsis = $test->perform_test_for_synopsis(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_tagline

  perform_test_for_tagline(arrayref $data) (boolean)

The perform_data_for_tagline method performs an overridable test for the C<tagline> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_tagline {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=tagline content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_tagline;

  my $perform_test_for_tagline = $test->perform_test_for_tagline(
    $data,
  );

  # true

=back

=over 4

=item perform_test_for_version

  perform_test_for_version(arrayref $data) (boolean)

The perform_data_for_version method performs an overridable test for the C<version> section and returns truthy or falsy.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  package Example::Test;

  use Venus::Class 'base';

  base 'Venus::Test';

  sub perform_test_for_version {
    my ($self, $data) = @_;

    my $result = length(join "\n", @{$data}) ? true : false;

    $self->pass($result, "=version content");

    return $result;
  }

  package main;

  my $test = Example::Test->new('t/path/pod/example');

  my $data = $test->collect_data_for_version;

  my $perform_test_for_version = $test->perform_test_for_version(
    $data,
  );

  # true

=back

=over 4

=item present

  present(string $name, any @args) (string)

The present method dispatches to the C<present_data_for_${name}> method
indictated by the first argument and returns the result. Returns an arrayref in
scalar context, and a list in list context.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # given: synopsis

  package main;

  my $present = $test->present('name');

  # =head1 NAME
  #
  # Venus::Test - Test Class
  #
  # =cut

=back

=over 4

=item present_data_for_abstract

  present_data_for_abstract() (arrayref)

The present_data_for_abstract method builds a documentation block for the C<abstract> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =abstract
  #
  # Example Test Documentation
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_abstract = $test->present_data_for_abstract;

  # =head1 ABSTRACT
  #
  # Example Test Documentation
  #
  # =cut

=back

=over 4

=item present_data_for_attribute

  present_data_for_attribute(string $name) (arrayref)

The present_data_for_attribute method builds a documentation block for the C<attribute $name> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =attribute name
  #
  # The name attribute is read-write, optional, and holds a string.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $name = $example->name;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_attribute = $test->present_data_for_attribute('name');

  # =head2 name
  #
  # The name attribute is read-write, optional, and holds a string.
  #
  # =over 4
  #
  # =item name example 1
  #
  #   # given: synopsis
  #
  #   my $name = $example->name;
  #
  #   # "..."
  #
  # =back
  #
  # =cut

=back

=over 4

=item present_data_for_authors

  present_data_for_authors() (arrayref)

The present_data_for_authors method builds a documentation block for the C<authors> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =authors
  #
  # Awncorp, C<awncorp@cpan.org>
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_authors = $test->present_data_for_authors;

  # =head1 AUTHORS
  #
  # Awncorp, C<awncorp@cpan.org>
  #
  # =cut

=back

=over 4

=item present_data_for_description

  present_data_for_description() (arrayref)

The present_data_for_description method builds a documentation block for the C<description> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =description
  #
  # This package provides an example class.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_description = $test->present_data_for_description;

  # =head1 DESCRIPTION
  #
  # This package provides an example class.
  #
  # =cut

=back

=over 4

=item present_data_for_encoding

  present_data_for_encoding() (arrayref)

The present_data_for_encoding method builds a documentation block for the C<encoding> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =encoding
  #
  # utf8
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_encoding = $test->present_data_for_encoding;

  # =encoding UTF8
  #
  # =cut

=back

=over 4

=item present_data_for_error

  present_data_for_error(string $name) (arrayref)

The present_data_for_error method builds a documentation block for the C<error $name> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =error error_on_unknown
  #
  # This package may raise an error_on_unknown error.
  #
  # =cut
  #
  # =example-1 error_on_unknown
  #
  #   # given: synopsis
  #
  #   my $error = $example->catch('error', {
  #     with => 'error_on_unknown',
  #   });
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_error = $test->present_data_for_error('error_on_unknown');

  # =over 4
  #
  # =item error: C<error_on_unknown>
  #
  # This package may raise an error_on_unknown error.
  #
  # B<example 1>
  #
  #   # given: synopsis
  #
  #   my $error = $example->catch('error', {
  #     with => 'error_on_unknown',
  #   });
  #
  #   # "..."
  #
  # =back

=back

=over 4

=item present_data_for_example

  present_data_for_example(number $numberm string $name) (arrayref)

The present_data_for_example method builds a documentation block for the C<example-$number $name> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =attribute name
  #
  # The name attribute is read-write, optional, and holds a string.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $name = $example->name;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_example = $test->present_data_for_example(1, 'name');

  # =over 4
  #
  # =item name example 1
  #
  #   # given: synopsis
  #
  #   my $name = $example->name;
  #
  #   # "..."
  #
  # =back

=back

=over 4

=item present_data_for_feature

  present_data_for_feature(string $name) (arrayref)

The present_data_for_feature method builds a documentation block for the C<feature $name> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =feature noop
  #
  # This package is no particularly useful features.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_feature = $test->present_data_for_feature('noop');

  # =over 4
  #
  # =item noop
  #
  # This package is no particularly useful features.
  #
  # =back

=back

=over 4

=item present_data_for_function

  present_data_for_function(string $name) (arrayref)

The present_data_for_function method builds a documentation block for the C<function $name> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =function eg
  #
  # The eg function returns a new instance of Example.
  #
  # =cut
  #
  # =example-1 name
  #
  #   # given: synopsis
  #
  #   my $example = eg();
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_function = $test->present_data_for_function('eg');

  # =head2 eg
  #
  # The eg function returns a new instance of Example.
  #
  # =cut

=back

=over 4

=item present_data_for_includes

  present_data_for_includes() (arrayref)

The present_data_for_includes method builds a documentation block for the C<includes> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =includes
  #
  # function: eg
  #
  # method: prepare
  # method: execute
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_includes = $test->present_data_for_includes;

  # undef

=back

=over 4

=item present_data_for_inherits

  present_data_for_inherits() (arrayref)

The present_data_for_inherits method builds a documentation block for the C<inherits> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =inherits
  #
  # Venus::Core::Class
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_inherits = $test->present_data_for_inherits;

  # =head1 INHERITS
  #
  # This package inherits behaviors from:
  #
  # L<Venus::Core::Class>
  #
  # =cut

=back

=over 4

=item present_data_for_integrates

  present_data_for_integrates() (arrayref)

The present_data_for_integrates method builds a documentation block for the C<integrates> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =integrates
  #
  # Venus::Role::Catchable
  # Venus::Role::Throwable
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_integrates = $test->present_data_for_integrates;

  # =head1 INTEGRATES
  #
  # This package integrates behaviors from:
  #
  # L<Venus::Role::Catchable>
  #
  # L<Venus::Role::Throwable>
  #
  # =cut

=back

=over 4

=item present_data_for_layout

  present_data_for_layout() (arrayref)

The present_data_for_layout method builds a documentation block for the C<layout> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =layout
  #
  # encoding
  # name
  # synopsis
  # description
  # attributes: attribute
  # authors
  # license
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_layout = $test->present_data_for_layout;

  # undef

=back

=over 4

=item present_data_for_libraries

  present_data_for_libraries() (arrayref)

The present_data_for_libraries method builds a documentation block for the C<libraries> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =libraries
  #
  # Venus::Check
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_libraries = $test->present_data_for_libraries;

  # =head1 LIBRARIES
  #
  # This package uses type constraints from:
  #
  # L<Venus::Check>
  #
  # =cut

=back

=over 4

=item present_data_for_license

  present_data_for_license() (arrayref)

The present_data_for_license method builds a documentation block for the C<license> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =license
  #
  # No license granted.
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_license = $test->present_data_for_license;

  # =head1 LICENSE
  #
  # No license granted.
  #
  # =cut

=back

=over 4

=item present_data_for_message

  present_data_for_message(string $name) (arrayref)

The present_data_for_message method builds a documentation block for the C<message $name> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =message accept
  #
  # The accept message represents acceptance.
  #
  # =cut
  #
  # =example-1 accept
  #
  #   # given: synopsis
  #
  #   my $accept = $example->accept;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_message = $test->present_data_for_message('accept');

  # =over 4
  #
  # =item accept
  #
  # The accept message represents acceptance.
  #
  # B<example 1>
  #
  #   # given: synopsis
  #
  #   my $accept = $example->accept;
  #
  #   # "..."
  #
  # =back

=back

=over 4

=item present_data_for_metadata

  present_data_for_metadata(string $name) (arrayref)

The present_data_for_metadata method builds a documentation block for the C<metadata $name> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =method prepare
  #
  # The prepare method prepares for execution.
  #
  # =cut
  #
  # =metadata prepare
  #
  # {since => 1.2.3}
  #
  # =cut
  #
  # =example-1 prepare
  #
  #   # given: synopsis
  #
  #   my $prepare = $example->prepare;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_metadata = $test->present_data_for_metadata('prepare');

  # undef

=back

=over 4

=item present_data_for_method

  present_data_for_method(string $name) (arrayref)

The present_data_for_method method builds a documentation block for the C<method $name> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =method execute
  #
  # The execute method executes the logic.
  #
  # =cut
  #
  # =metadata execute
  #
  # {since => 1.2.3}
  #
  # =cut
  #
  # =example-1 execute
  #
  #   # given: synopsis
  #
  #   my $execute = $example->execute;
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_method = $test->present_data_for_method('execute');

  # =head2 execute
  #
  #   execute() (boolean)
  #
  # The execute method executes the logic.
  #
  # I<Since C<1.2.3>>
  #
  # =over 4
  #
  # =item execute example 1
  #
  #   # given: synopsis
  #
  #   my $execute = $example->execute;
  #
  #   # "..."
  #
  # =back
  #
  # =cut

=back

=over 4

=item present_data_for_name

  present_data_for_name() (arrayref)

The present_data_for_name method builds a documentation block for the C<name> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =name

  # Example

  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_name = $test->present_data_for_name;

  # =head1 NAME
  #
  # Example - Example Class
  #
  # =cut

=back

=over 4

=item present_data_for_operator

  present_data_for_operator(string $name) (arrayref)

The present_data_for_operator method builds a documentation block for the C<operator $name> section and returns it as a string.

I<Introduced C<3.55>>, I<Deprecated C<4.15>>

B<example 1>

  # =operator ("")
  #
  # This package overloads the C<""> operator.
  #
  # =cut
  #
  # =example-1 ("")
  #
  #   # given: synopsis
  #
  #   my $string = "$example";
  #
  #   # "..."
  #
  # =cut

  package main;

  use Venus::Test 'test';

  my $test = test 't/path/pod/example';

  my $present_data_for_operator = $test->present_data_for_operator('("")');

  # =over 4
  #
  # =item operation: C<("")>
  #
  # This package overloads the C<""> operator.
  #
  # B<example 1>
  #
  #   # given: synopsis
  #
  #   my $string = "$example";
  #
  #   # "..."
  #
  # =back

=back

=head1 AUTHORS

Awncorp, C<awncorp@cpan.org>

=cut

=head1 LICENSE

Copyright (C) 2022, Awncorp, C<awncorp@cpan.org>.

This program is free software, you can redistribute it and/or modify it under
the terms of the Apache license version 2.0.

=cut