/*
 *  File: SG_Bitmap.h
 *  Project: ShmanssyGUI
 *
 *  Written by: Ivan Leben
 *  Licence: GNU GPL
 *
 */

#include "ShmanssyGUI.h"

#ifndef SG_BITMAP_H
#define SG_BITMAP_H

#define COLOR_R 0
#define COLOR_G 1
#define COLOR_B 2
#define COLOR_A 3

#define SG_GRAYSCALE_8			0
#define SG_GRAYSCALE_ALPHA_16	1
#define SG_RGB_16				2
#define SG_RGB_ALPHA_24			3
#define SG_RGB_24				4
#define SG_RGB_ALPHA_32			5
extern int SG_formatBpp[6];
bool SG_IsAlphaFormat(int format);

#define TILE_VERTICAL 0
#define TILE_HORIZONTAL 1
#define TILE_BOTH 2

#define PI 3.14159265359

typedef unsigned char BYTE;

class SG_Color
{
public:
	unsigned char r,g,b,a;
	
	SG_Color(){r=0; g=0; b=0; a=255;}
	SG_Color(int R, int G, int B) {r=R; g=G; b=B; a=255;}
	SG_Color(int R, int G, int B, int A) {r=R; g=G; b=B; a=A;}
	void FromRGB(int R, int G, int B) {r=R; g=G; b=B; a=255;}
	void FromRGBA(int R, int G, int B, int A) {r=R; g=G; b=B; a=A;}
	void FromPixel(BYTE *data, int format);
	void ToPixel(BYTE *data, int format);
	void BlendWith(SG_Color &aColor);
	unsigned int ToInt() {unsigned char retval[4] = {a,r,g,b}; return *((int*)retval);}
	
	void operator= (SG_Color &color) {FromRGBA(color.r, color.g, color.b, color.a);}
	bool operator== (SG_Color &color) {return (color.r==r && color.g==g && color.b==b && color.a==a);}
};

class SG_Point
{
public:
    int x, y;
    SG_Point() {x=0; y=0;}
    SG_Point(int x0, int y0) {x=x0; y=y0;}
    
	void Set(int x0, int y0) {x=x0; y=y0;}
    void Offset(int offX, int offY) {x+=offX; y+=offY;}

    void operator= (SG_Point pt){x=pt.x; y=pt.y;}
};

class SG_Rect
{

public:
    int x,y,w,h;
    SG_Rect() {x=0; y=0; w=0; h=0;}
    SG_Rect(int x0, int y0, int w0, int h0) {x=x0; y=y0; w=w0; h=h0;}
    
	bool operator== (SG_Rect rc) {return (rc.x == x && rc.y==y && rc.w==w && rc.h==h);}
    void operator= (SG_Rect rc) {x=rc.x; y=rc.y; w=rc.w; h=rc.h;}
	
    int Right() {return x+w-1;}
    int Bottom() {return y+h-1;}
    SG_Point Center() {SG_Point ptRet(x+w/2, y+h/2); return ptRet;}

    void Set(int x0, int y0, int w0, int h0) {x=x0; y=y0; w=w0; h=h0;}
    void Offset(int offX, int offY) {x+=offX; y+=offY;}
    void Inflate(int l, int t, int r, int b) {x-=l; y-=t; w+=l+r; h+=t+b;}
    
    bool ContainsPt(int X, int Y) {return X>=x && X<=Right() && Y>=y && Y<=Bottom();}
};

#define TOPLEFT		0
#define TOP			1
#define TOPRIGHT	2
#define LEFT		3
#define CENTER		4
#define RIGHT		5
#define BOTTOMLEFT	6
#define BOTTOM		7
#define BOTTOMRIGHT	8

class SG_Texture
{
public:
	bool initialized;
	GLuint handle;
	SG_Point size;

	SG_Texture();
	~SG_Texture();
	
	void Create(int width, int height);
	void CreateFromPixels(BYTE *pixels, int width, int height, int pitch, int format);
	void Draw(SG_Rect &rcSrc, SG_Rect &rcDst);
	void DrawRotated(SG_Rect &rcSrc, SG_Rect &rcDst, float angle);
};

class SG_Bitmap
{
private:
	int tileLeft, tileRight, tileTop, tileBottom;
						
public:
	SG_Rect rcTileSrc[9]; SG_Rect rcTileDst[9];
	SG_Point size;
	int format;
	SG_Color maskColor;	
	bool manualColor;
	bool alphaBlend;	
	SDL_Surface *surface;
	bool isTextureUpdated;
	SG_Texture texture;	
	
	SG_Bitmap();
	~SG_Bitmap();

	BYTE *Pixels() {return ( surface==NULL ? NULL : (BYTE*)surface->pixels );}
	
	SG_Texture* Texture() {return &texture;}
	void EnableAlphaBlending(bool isEnabled) {alphaBlend = isEnabled; SDL_SetAlpha(surface, SDL_SRCALPHA, 255);}
	bool IsAlphaBlendingEnabled() {return alphaBlend;}
	void EnableManualBackColor(bool isEnabled) {manualColor = isEnabled;}
	bool IsManualBackColorEnabled() {return manualColor;}

	void Create(int width, int height, int format);
	void FromFile(char *fileName);
	void FromPixels(BYTE *pixels, int width, int height, int newFormat);
	void FromBitmap(SG_Bitmap &bmp);	
	void SaveToFile(char *fileName);	
	void ConvertFormatTo(int newFormat);
	void SetPixel(int x, int y, SG_Color color);
	SG_Color GetPixel(int x, int y);
	void UpdateTexture();
		
	void SetAlphaLevel(BYTE alpha);	
	void MaskAlphaLevel(SG_Color &color);
	void MaskAlphaLevel(BYTE r, BYTE g, BYTE b);
	SG_Color& MaskColor() {return maskColor;}
		
	void SetTileBounds(int left, int right, int top, int bottom);
	SG_Rect TileBounds();	
	void SetupTileRectangles(SG_Rect &rcDst);	
};

void SG_SetClipRect(SG_Rect *rcClip);
void SG_EnableAlphaBlending(bool isEnabled);
int SG_IsAlphaBlendingEnabled();

SDL_Surface* SG_CreateSurface(Uint32 flags, int width, int height, int depth);
void SG_SetColorKey(SDL_Surface *surface, SG_Color &color);
void SG_FreeSurface(SDL_Surface **surf);
void SG_BlitSurface(SDL_Surface *surfSrc, SG_Rect &rcSrc, SDL_Surface *surfDst, SG_Rect &rcDst);
void SG_BlitSurfaceWithAlpha(SDL_Surface *surfSrc, SG_Rect &rcSrc, SDL_Surface *surfDst, SG_Rect &rcDst, int formatSrc, int formatDst);
void SG_FillRectWithAlpha(SDL_Surface *surfDst, SG_Rect &rc, SG_Color &color, int formatDst);

extern SDL_Color SG_grayscalePalette[256];
SDL_Cursor* SG_LoadTextCursor(char *text[], int w, int h, int hotX, int hotY);
SDL_Cursor* SG_LoadMSBCursor(char *file, int hotX, int hotY);

#endif
