/* Array.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Array_
#define _H_Array_ 1

#include "Container.h"


/** Array
	The Array class.
*/
extern Class ArrayClass;


/** TArray
	This represents an instance of the Array class.

	See ={CONTAINER_HEAD}.
*/
struct TArray
{
	CONTAINER_HEAD;
	long  cap;      /* allocated capacity <= GC_MAX_ITEMS */
	Any  *objs;     /* the items, array is GcMemMalloc'ed */
	Any   type;     /* item type/validator, maybe 0 */
};


/** ArrayEMPTY
	This is a predefined frozen empty Array.

	C{const []} in programs simply points to this object.
*/
extern Array ArrayEMPTY;


/** ArrayIs
	Whether or not an object is an Array.
*/
#define ArrayIs(o) (ClassOf(o) == ArrayClass)


/** ArrayFlag
	The following flags are used by Array objects.

	Do not access these flags directly.
	Use the accessor macros provided instead.
*/
#define ArrayFlagPROTECTED ContainerFlagPROTECTED /* protected by program level */
#define ArrayFlagFROZEN    ContainerFlagFROZEN    /* immutable? */
#define ArrayFlagFIXED     ContainerFlagFIXED     /* resizable? */
#define ArrayFlagFORLAST   Flag10                 /* to indicate *c in for a, b, *c */


/** ArrayIsFrozen ArrayFreeze ArrayUnfreeze ArrayIsFixed
	ArrayIsFrozen checks whether or not the Array is immutable.

	ArrayFreeze must only be used after you have created a new Array and
	don't want the Array to be modified by others.

	ArrayUnfreeze must only be used on an Array you have created.
	Normally, to modify the Array then freeze it again.

	ArrayIsFixed checks whether or not the Array is resizable.

	You should have no reason to set/unset the ArrayFlagFIXED in C level.
	That is why the ArrayFix and ArrayUnfix macros are only defined within
	C{Array.c}.
*/
#define ArrayIsFrozen(self)  ContainerIsFrozen (self)
#define ArrayFreeze(self)    ContainerFreeze   (self)
#define ArrayUnfreeze(self)  ContainerUnfreeze (self)
#define ArrayIsFixed(self)   ContainerFixed    (self)


/** ArrayIsForlast ArraySetForlast
	This is strictly for the interpreter/compiler.

	The ArrayFlagFORLAST flag is used to indicate that the Array is a collection
	of the last items v{z} in a C{for x, y, *z} syntax.

	Once set, the flag must never be unset.
*/
#define ArrayIsForlast(self)  FlagTest (self, ArrayFlagFORLAST)
#define ArraySetForlast(self) FlagSet  (self, ArrayFlagFORLAST)


/** ArrayNew
	Various Array constructors.

	In any case v{len} must be within 0..={GC_MAX_ITEMS}.

	ArrayNew create a new regular GC'ed Array with a capacity enough to hold
	a specified number of items.

	ArrayRoot creates a root Array object, i.e. it is always marked by the GC.

	ArrayStatic creates a static Array object, i.e. it is never marked by the GC,
	thus you must make sure that all items are also ={GcIsStatic} objects.
	No checks are made.

	ArrayNewArg and ArrayNewEx create a new Array with the specified items.
*/
extern Array ArrayNew     (long len);
extern Array ArrayRoot    (long len);
extern Array ArrayStatic  (long len);
extern Array ArrayNewArgv (int argc, Any *argv);
extern Array ArrayNewEx   (long len, ...);


/** ArrayFlatten
	Return a new Array by flattening the arguments.

	This is used to call functions/methods with flattened arguments.
	See the C{Inlines.h} file for example.
*/
extern Array ArrayFlatten (int argc, Any *argv);


/** ArrayMakeSpace
	Make sure the Array has enough space for v{more} items, grow if necessary.

	Return 1 on success, return 0 and throw EMemory on failure.
*/
extern int ArrayMakeSpace (Array self, long more);


/** ArrayContains
	Whether an object is within the Array.
*/
#define ArrayContains(self, x) (array__in (self, x) == True)


/** ArrayLike
	Whether two arrays contain the same items.
*/
#define ArrayLike(self, x) (array__like (self, x) == True)


/** ArrayPush
	Push an item.

	Return self on success, throws on error.
*/
extern Array ArrayPush (Array self, Any x);


/** ArrayOf
	Whether all items are of a specific type.
*/
extern Array ArrayOf (Array self, Any type);


/** ArrayIndex ArrayRindex
	ArrayIndex returns the index of v{x} optionally starting from offset v{i} going
	forward. Return -1 if not found.

	ArrayRindex returns the index of v{x} optionally starting from offset v{i}
	going backward.

	Return -1 if not found.
*/
extern long ArrayIndex  (Array self, Any x, long i);
extern long ArrayRindex (Array self, Any x, long i);


/** ArrayQuick
	Quick and dirty functions that do not throw.

	ArrayQuickDelete removes v{x}.
	The equality test is done by comparing pointers, not via __eq.

	ArrayQuickPush pushes an item.
	Make sure the Array is relatively small.

	ArrayQuickPop pops the last item, if any, without shrinking the capacity.

	ArrayQuickSlice gets a slice.
	Return 0 if index out of bound.

	ArrayGETITEM gets an item.
	Return 0 if index out of bound.

	ArrayDELFIRST removes the first item without shrinking the capacity.

	ArrayCLEAR sets the length to zero without shrinking the capacity.
*/
extern void  ArrayQuickDelete (Array self, Any x);
extern void  ArrayQuickPush   (Array self, Any x);
extern void  ArrayQuickPop    (Array self);
extern Array ArrayQuickSlice  (Array self, long index, long len);

#define ArrayGETITEM(self, i) \
	(((i) < 0 || (i) >= ((Array) self)->len) ? 0 : ((Array) self)->objs[i])

#define ArrayDELFIRST(self) \
BEGIN \
	if ((self)->len > 0) \
		OBJECT_COPY ((self)->objs, (self)->objs + 1, --((self)->len)); \
END

#define ArrayCLEAR(self) \
BEGIN \
	(self)->len = 0; \
END


/** ArrayCopy
	Return a shallow copy.
*/
extern Array ArrayCopy (Array self);


/***/
/*--------------------------------------------------------------------------*/

/* called internally by Class.c to initialize the Array class */
extern void ArraySetup ();
extern void ArrayInit  ();

/* for efficiency */
extern Boolean array__in   (Array self, Any x);
extern Boolean array__like (Array self, Any x);


#endif /*_H_Array_*/
