/* Big.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Big_
#define _H_Big_ 1


/** Big
	This is the final Big class.
*/
extern Class BigClass;


/** BigConstant
	These are constants used by the Big class.

	BigDIGIT is the C type for the digits.

	BigBASE is the digit base.

	BigFIG is the number of ASCII characters in one digit.
	The relation between BigDIGIT and BigBASE is C{BigBASE = 10 ^ BigFIG}.

	BigMAXEXPO is the maximum absolute value of the exponent.
	It is such that we can always do +/- without overflow.

	BigMINPREC and BigMAXPREC are the minimum and maximum precision.
	Multiply by BigFIG to get the number of significant digits.
	The value must fit in an Int.

	BigMAXLEN is the practical maximum allocatable number of digits.
	It must be such that it is always possible to convert a Long to a Big
	and vice versa. It must also fit in an Int.
*/
#define BigDIGIT    unsigned long
#define BigBASE     10000
#define BigFIG      4
#define BigFORMAT   "%04lu"
#define BigFIGZERO  "0000"
#define BigMAXEXPO  (INT_MAX / SIZEOF_LONG / BigFIG / 16)
#define BigMINPREC  2
#define BigMAXPREC  (INT_MAX / SIZEOF_LONG / 8)
#define BigMAXLEN   (INT_MAX / SIZEOF_LONG / 4)


/** BigPrecision
	This is the current precision.

	@warn
	It must be a value between ={BigMINPREC} and ={BigMAXPREC}.
	So be careful.
*/
extern long BigPrecision;


/** TBig
	This represents an instance of the Big class.

	@code
	value = 0.xxxxxxxxxxxxxxxx * BASE ^ expo
	          <-----len------>
	          MSB          LSB digits
*/
struct TBig
{
	OBJ_HEAD;
	BigDIGIT *digits; /* the digits, MSB first */
	long      len;    /* actual number of digits */
	long      expo;   /* the exponent */
	long      cap;    /* allocated capacity of digits */
};


/** BigFlag
	Flags used by Big objects.

	Do not access these flags directly.
	Use the accessor macros provided instead.
*/
#define BigFlagNEG  Flag0  /* negative? = LongFlagNEG */
#define BigFlagTMP  Flag9  /* on local C stack? = LongFlagTMP */


/** BigPredefined
	Predefined system Big objects.

	See ={BigStatic} to create your own.
*/
extern Big Big0;    /*  0.0  */
extern Big Big1;    /*  1.0  */
extern Big Big2;    /*  2.0  */
extern Big Big3;    /*  3.0  */
extern Big Big4;    /*  4.0  */
extern Big Big5;    /*  5.0  */
extern Big Big6;    /*  6.0  */
extern Big Big7;    /*  7.0  */
extern Big Big8;    /*  8.0  */
extern Big Big9;    /*  9.0  */
extern Big Big10;   /* 10.0  */
extern Big Big16;   /* 16.0  */
extern Big Big_1;   /* -1.0  */
extern Big Big_2;   /* -2.0  */
extern Big Big_4;   /* -4.0  */
extern Big Big_8;   /* -8.0  */
extern Big Big1_2;  /*  0.5  */
extern Big Big_1_2; /* -0.5  */
extern Big Big1_4;  /*  0.25 */
extern Big Big_1_4; /* -0.25 */


/** BigNew
	Create a new Big object from various C types and other objects.

	Note that except for BigNew, all always succeed.

	@warn
	You may not pass a C double NaN nor infinity to BigNewd and BigNewld.
	Remember that NaN is an error in Qu and infinities are represented with
	={InfPositive} and ={InfNegative} which are Inf objects. Not Float nor Big objects.
*/
#define    BigNewh(v)       BigNewl  ((long) v)
#define    BigNewuh(v)      BigNewul ((unsigned long) v)
#define    BigNewi(v)       BigNewl  ((long) v)
#define    BigNewui(v)      BigNewul ((unsigned long) v)
extern Big BigNewl          (long v);
extern Big BigNewul         (unsigned long v);
extern Big BigNewd          (double v);
#ifdef HAVE_LONG_LONG
extern Big BigNewll         (long long v);
extern Big BigNewull        (unsigned long long v);
#endif
#ifdef HAVE_LONG_DOUBLE
extern Big BigNewld         (long double v);
#endif

#define    BigNewInt(x)     BigNewl (IntTol (x))
extern Big BigNewLong       (Long x);
extern Big BigNewRational   (Rational x);
#define    BigNewFloat(x)   BigNewd (FloatTod (x))
#define    BigNewBig(x)     (x)
#define    BigNewComplex(x) ComplexToBig (x)
#define    BigNewInteger(x) IntegerToBig (x)
#define    BigNewDecimal(x) DecimalToBig (x)
#define    BigNewReal(x)    RealToBig (x)
#define    BigNewNumber(x)  NumberToBig (x)
#define    BigNewBoolean(x) (((x) == True) ? Big1 : Big0)
extern Big BigNewString     (String x);
extern Big BigNew           (Any x);


/** BigNewdx
	Create a new Big/Real object from a C double by checking whether or not the
	value is NaN or infinity.

	If the value is NaN then EValue is thrown and NULL is returned.

	If the value is infinity then EInf is thrown and NULL is returned unless the
	={NumberContext} allows infinities to be returned, in which case one of
	={InfPositive} or ={InfNegative} is returned based on the sign.
*/
extern Real BigNewdx  (double v);
#ifdef HAVE_LONG_DOUBLE
extern Real BigNewldx (long double v);
#endif


/** BigStatic
	Create a new system Big object, i.e. not in GC.
*/
extern Big BigStatic   (const char *s);
extern Big BigToStatic (Big x);


/** BigMake
	Make a Big value.

	BigMake creates a Big by specifying the digits and exponent.
	v{digits} must be MSB (most significant digit) first.
	v{len} is the number of digits.
	If it is zero then Big0 is returned.

	BigMake1 creates a Big with one digit and a specified exponent.
	This is useful for things such as 1E-40.
	If v{digit} is zero then Big0 is returned.
*/
extern Big BigMake  (BigDIGIT *digits, long len, int expo);
extern Big BigMake1 (BigDIGIT  digit, int expo);


/** BigTo
	Convert a Big to various C types and other objects.

	For C type conversions, v{errno} is set to 0 on success, to ERANGE
	on failure and return a maximum value (e.g. LONG_MAX, ULONG_MAX)
	for a positive number and a minimum value (e.g. LONG_MIN, ULONG_MIN)
	for a negative number.

	For object conversions NULL is returned and an exception thrown on failure.
	Only BigToInt and BigToFloat can fail. The others always succeed.
*/
#define                   BigToh(x)  Number_ltoh   (BigTol  (x))
#define                   BigTouh(x) Number_ultouh (BigToul (x))
#define                   BigToi(x)  Number_ltoi   (BigTol  (x))
#define                   BigToui(x) Number_ultoui (BigToul (x))
#define                   BigToli(x) Number_ltoli  (BigTol  (x))
extern long               BigTol     (Big x);
extern unsigned long      BigToul    (Big x);
extern float              BigTof     (Big x);
extern double             BigTod     (Big x);
#ifdef HAVE_LONG_LONG
extern long long          BigToll    (Big x);
extern unsigned long long BigToull   (Big x);
#endif
#ifdef HAVE_LONG_DOUBLE
extern long double        BigTold    (Big x);
#endif

extern Int      BigToInt        (Big x);
extern Long     BigToLong       (Big x);
extern Rational BigToRational   (Big x);
extern Float    BigToFloat      (Big x);
#define         BigToBig(x)     (x)
#define         BigToComplex(x) ComplexNew1 ((Real) (x))
extern Integer  BigToInteger    (Big x);
#define         BigToDecimal(x) ((Decimal) (x))
#define         BigToReal(x)    ((Real) (x))
#define         BigToNumber(x)  ((Number) (x))
extern Array    BigToArray      (Big x);
extern String   BigToString     (Big x);
#define         BigToBoolean(x) (BigIsZero (x) ? False : True)


/** BigSign
	BigSign returns -1, 0, or 1 if the number is negative, 0, or positive respectively.
*/
extern int BigSign (Big x);


/** BigIs
	Identity functions.

	BigIs tests whether or not an object is a Big.

	BigIsNeg returns 1 if the number is negative.
	Returns 0 otherwise.

	BigIsZero returns 1 if the number is zero.
	Returns 0 otherwise.

	BigIsOne returns 1 if the number is -1 or 1.
	Returns 0 otherwise.

	BigIs1 returns 1 if the number is 1.
	Returns 0 otherwise.

	BigIsWhole returns 1 if there is no fractional part.
	Returns 0 otherwise.

	BigIsOdd returns 1 if the number is whole and odd.
	Returns 0 otherwise.

	BigIsFrac returns 1 if 0 < |x| < 1.
	Returns 0 otherwise.

	@note
	If we ever decide to let Big subclassed then BigIs will have to
	say C{ObjectInstOf (x, BigClass)} instead.
*/
#define BigIs(x)      (ClassOf (x) == BigClass)
#define BigIsNeg(x)   FlagTest ((x), BigFlagNEG)
#define BigIsZero(x)  ((x)->len == 0)
extern int BigIsOne   (Big x);
extern int BigIs1     (Big x);
extern int BigIsWhole (Big x);
extern int BigIsFrac  (Big x);
extern int BigIsOdd   (Big x);


/** BigUnary
	Unary functions.

	BigPos returns the number itself, obviously.

	BigAbs returns the absolute value.
	The result is a Big and always succeed.

	BigNeg returns a negated value.
	The result is a Big and always succeed.

	BigIncr returns the Big incremented by 1.
	On success, the result is a Big or ={InfPositive}.
	EBig is thrown if the ={NumberContext} does not allow infinities.

	BigDecr returns the Big decremented by 1.
	On success, the result is a Big or ={InfPositive}.
	EBig is thrown if the ={NumberContext} does not allow infinities.
*/
#define     BigPos(x)   (x)
extern  Big BigAbs      (Big x);
extern  Big BigNeg      (Big x);
#define     BigIncr(x)  BigPlusBig (x, Big1)
#define     BigDecr(x)  BigPlusBig (x, Big_1)


/** BigEq
	Whether the value of a Big is the same as the value of another Number object.
	All return 0 or 1.
*/
extern int BigEqInt           (Big x, Int y);
extern int BigEqLong          (Big x, Long y);
extern int BigEqRational      (Big x, Rational y);
extern int BigEqFloat         (Big x, Float y);
extern int BigEqBig           (Big x, Big y);
#define    BigEqComplex(x, y) ComplexEqBig (y, x)
#define    BigEqInf(x, y)     (0)
extern int BigEqInteger       (Big x, Integer y);
extern int BigEqDecimal       (Big x, Decimal y);
extern int BigEqReal          (Big x, Real y);
extern int BigEqNumber        (Big x, Number y);


/** BigLike
	Whether the absolute value of a Big is the same as the absolute value of another
	Number object.
	All return 0 or 1.
*/
#define    BigLikeInt(x, y)      (BigCmpAbsInt (x, y) == 0)
#define    BigLikeLong(x, y)     (BigCmpAbsLong (x, y) == 0)
#define    BigLikeRational(x, y) (BigCmpAbsRational (x, y) == 0)
#define    BigLikeFloat(x, y)    (BigCmpAbsFloat (x, y) == 0)
#define    BigLikeBig(x, y)      (BigCmpAbsBig (x, y) == 0)
#define    BigLikeComplex(x, y)  ComplexLikeBig (y, x)
#define    BigLikeInf(x, y)      (0)
extern int BigLikeInteger        (Big x, Integer y);
extern int BigLikeDecimal        (Big x, Decimal y);
extern int BigLikeReal           (Big x, Real y);
extern int BigLikeNumber         (Big x, Number y);


/** BigCmp
	Compare a Big with another Number object.
	All return -1, 0, or 1 in the usual manner.
*/
extern int BigCmpInt           (Big x, Int y);
extern int BigCmpLong          (Big x, Long y);
extern int BigCmpRational      (Big x, Rational y);
extern int BigCmpFloat         (Big x, Float y);
extern int BigCmpBig           (Big x, Big y);
#define    BigCmpComplex(x, y) (-ComplexCmpBig (y, x))
#define    BigCmpInf(x, y)     (-InfCmpBig (y, x))
extern int BigCmpInteger       (Big x, Integer y);
extern int BigCmpDecimal       (Big x, Decimal y);
extern int BigCmpReal          (Big x, Real y);
extern int BigCmpNumber        (Big x, Number y);



/** BigCmpAbs
	Compare the absolute value of a Big with the absolute value of another
	Number object.
	All return -1, 0, or 1 in the usual manner.
*/
extern int BigCmpAbsInt           (Big x, Int y);
extern int BigCmpAbsLong          (Big x, Long y);
extern int BigCmpAbsRational      (Big x, Rational y);
extern int BigCmpAbsFloat         (Big x, Float y);
extern int BigCmpAbsBig           (Big x, Big y);
#define    BigCmpAbsComplex(x, y) (-ComplexCmpAbsBig (y, x))
#define    BigCmpAbsInf(x, y)     (-1)
extern int BigCmpAbsInteger       (Big x, Integer y);
extern int BigCmpAbsDecimal       (Big x, Decimal y);
extern int BigCmpAbsReal          (Big x, Real y);
extern int BigCmpAbsNumber        (Big x, Number y);


/** BigPlus
	Add a Big with another Number object.
*/
extern Real   BigPlusInt           (Big x, Int y);
extern Real   BigPlusLong          (Big x, Long y);
extern Real   BigPlusRational      (Big x, Rational y);
extern Real   BigPlusFloat         (Big x, Float y);
extern Real   BigPlusBig           (Big x, Big y);
#define       BigPlusComplex(x, y) ComplexPlusBig (y, x)
#define       BigPlusInf(x, y)     InfPlusBig (y, x)
extern Real   BigPlusInteger       (Big x, Integer y);
extern Real   BigPlusDecimal       (Big x, Decimal y);
extern Real   BigPlusReal          (Big x, Real y);
extern Number BigPlusNumber        (Big x, Number y);


/** BigMinus
	Subtract a Big with another Number object.
*/
extern Real   BigMinusInt           (Big x, Int y);
extern Real   BigMinusLong          (Big x, Long y);
extern Real   BigMinusRational      (Big x, Rational y);
extern Real   BigMinusFloat         (Big x, Float y);
extern Real   BigMinusBig           (Big x, Big y);
#define       BigMinusComplex(x, y) ComplexRminusBig (y, x)
#define       BigMinusInf(x, y)     InfPlusBig (InfNeg (y), x)
extern Real   BigMinusInteger       (Big x, Integer y);
extern Real   BigMinusDecimal       (Big x, Decimal y);
extern Real   BigMinusReal          (Big x, Real y);
extern Number BigMinusNumber        (Big x, Number y);


/** BigRminus
	Subtract another Number object with a Big.
	This is only for efficiency.
*/
extern Real BigRminusInt      (Big x, Int y);
extern Real BigRminusLong     (Big x, Long y);
extern Real BigRminusRational (Big x, Rational y);
extern Real BigRminusFloat    (Big x, Float y);


/** BigMul
	Multiply a Big with another Number object.
*/
extern Real   BigMulInt           (Big x, Int y);
extern Real   BigMulLong          (Big x, Long y);
extern Real   BigMulRational      (Big x, Rational y);
extern Real   BigMulFloat         (Big x, Float y);
extern Real   BigMulBig           (Big x, Big y);
#define       BigMulComplex(x, y) ComplexMulBig (y, x)
#define       BigMulInf(x, y)     InfMulBig (y, x)
extern Real   BigMulInteger       (Big x, Integer y);
extern Real   BigMulDecimal       (Big x, Decimal y);
extern Real   BigMulReal          (Big x, Real y);
extern Number BigMulNumber        (Big x, Number y);


/** BigDiv
	Divide a Big with another Number object.
*/
extern Real   BigDivInt           (Big x, Int y);
extern Real   BigDivLong          (Big x, Long y);
extern Real   BigDivRational      (Big x, Rational y);
extern Real   BigDivFloat         (Big x, Float y);
extern Real   BigDivBig           (Big x, Big y);
#define       BigDivComplex(x, y) ComplexRdivBig (y, x)
extern Big    BigDivInf           (Big x, Inf y);
extern Real   BigDivInteger       (Big x, Integer y);
extern Real   BigDivDecimal       (Big x, Decimal y);
extern Real   BigDivReal          (Big x, Real y);
extern Number BigDivNumber        (Big x, Number y);


/** BigRdiv
	Divide another Number object with a Big.
	This is only for efficiency.
*/
extern Real BigRdivInt      (Big x, Int y);
extern Real BigRdivLong     (Big x, Long y);
extern Real BigRdivRational (Big x, Rational y);
extern Real BigRdivFloat    (Big x, Float y);


/** BigIdiv
	Integer division of a Big with another Number object.
*/
extern Integer BigIdivInt           (Big x, Int y);
extern Integer BigIdivLong          (Big x, Long y);
extern Integer BigIdivRational      (Big x, Rational y);
extern Integer BigIdivFloat         (Big x, Float y);
extern Integer BigIdivBig           (Big x, Big y);
#define        BigIdivComplex(x, y) ComplexRidivBig (y, x)
extern Integer BigIdivInf           (Big x, Inf y);
extern Integer BigIdivInteger       (Big x, Integer y);
extern Integer BigIdivDecimal       (Big x, Decimal y);
extern Integer BigIdivReal          (Big x, Real y);
extern Integer BigIdivNumber        (Big x, Number y);


/** BigRidiv
	Integer division of another Number object with a Big.
	This is only for efficiency.
*/
extern Integer BigRidivInt      (Big x, Int y);
extern Integer BigRidivLong     (Big x, Long y);
extern Integer BigRidivRational (Big x, Rational y);
extern Integer BigRidivFloat    (Big x, Float y);


/** BigRem
	Remainder of dividing a Big with another Number object.
*/
extern Real   BigRemInt           (Big x, Int y);
extern Real   BigRemLong          (Big x, Long y);
extern Real   BigRemRational      (Big x, Rational y);
extern Real   BigRemFloat         (Big x, Float y);
extern Real   BigRemBig           (Big x, Big y);
#define       BigRemComplex(x, y) ComplexRremBig (y, x)
extern Big    BigRemInf           (Big x, Inf y);
extern Real   BigRemInteger       (Big x, Integer y);
extern Real   BigRemDecimal       (Big x, Decimal y);
extern Real   BigRemReal          (Big x, Real y);
extern Number BigRemNumber        (Big x, Number y);


/** BigRrem
	Remainder of dividing another Number object with a  Big.
	This is only for efficiency.
*/
extern Real BigRremInt      (Big x, Int y);
extern Real BigRremLong     (Big x, Long y);
extern Real BigRremRational (Big x, Rational y);
extern Real BigRremFloat    (Big x, Float y);


/** BigPow
	Raise a Big to the power of another Number object.
*/
extern Real   BigPowInt           (Big x, Int y);
extern Real   BigPowLong          (Big x, Long y);
extern Number BigPowRational      (Big x, Rational y);
extern Number BigPowFloat         (Big x, Float y);
extern Number BigPowBig           (Big x, Big y);
#define       BigPowComplex(x, y) ComplexRpowBig (y, x)
extern Real   BigPowInf           (Big x, Inf y);
extern Real   BigPowInteger       (Big x, Integer y);
extern Number BigPowDecimal       (Big x, Decimal y);
extern Number BigPowReal          (Big x, Real y);
extern Number BigPowNumber        (Big x, Number y);


/** BigRpow
	Raise another Number object to the power of a Big.
	This is only for efficiency.
*/
extern Number BigRpowInt      (Big x, Int y);
extern Number BigRpowLong     (Big x, Long y);
extern Number BigRpowRational (Big x, Rational y);
extern Number BigRpowFloat    (Big x, Float y);


/** BigRound
	Rounding functions.

	BigCeil rounds the number to the nearest integer towards positive infinity.
	EBig is thrown if the number is too big.

	BigFloor rounds the number to the nearest integer towards negative infinity.
	EBig is thrown if the number is too big.

	BigTrunc rounds the number towards zero, that is, it removes the
	fraction part. This function always succeed and returns an Int or Long.

	BigCeilEx, BigFloorEx, and BigTruncEx do the same but return a Big instead
	of an Integer.

	BigRound rounds a Big to the specified number of fraction digits.
*/
extern Real BigCeil   (Big x);
extern Real BigFloor  (Big x);
extern Big  BigTrunc  (Big x);
extern Big  BigRound  (Big x, long prec);


/** BigFrac
	Return the fraction part.

	The precision used is the precision of v{x} thus there is no precision
	lost here.
*/
extern Big BigFrac (Big x);


/** BigPrint
	Print a Big to v{stdout}.

	Note that this might throw an exception because writing to v{stdout} may
	fail for various reasons.
*/
extern Null BigPrint (Big x);


/** BigFormat
	Format a Big.

	See ={Format} for details.
*/
extern String BigFormat (Big x, format_s *f);


/** BigScan
	Scan an Big number.

	v{s} is the start of the string.

	On success, a Big object is returned and v{s} is updated to the next cursor
	position after the number.

	On failure, NULL is returned and errno is set.

	BigScanString does the same but throws ESyntax on failure.
*/
extern Big BigScan       (char **s, int neg);
extern Big BigScanString (String s);


/** BigIntegerFactor
	Return the Integer which must be multiplied to the Big to get an Integer.

	@xmp
		BigIntegerFactor (1.2 ) = 10
		BigIntegerFactor (1.23) = 100
*/
extern Integer BigIntegerFactor (Big x);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Big class */
extern void BigSetup ();
extern void BigInit  ();

/* this needs to be set by the Math module */
typedef Number (*bigpowbig_f) (Big self, Big x);
extern bigpowbig_f BigPowBig_math;


#endif /*_H_Big_*/
