/* Buffer.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Buffer_
#define _H_Buffer_ 1

#include "String.h"


/** Buffer
	The final Buffer class.
*/
extern Class BufferClass;


/** TBuffer
	This represents an instance of the Buffer class.

	See ={STRING_HEAD}
*/
struct TBuffer
{
	STRING_HEAD;
	long  cap;    /* current allocated space */
	long  limit;  /* maximum size allowed */
	Any   scope;  /* protector scope */
};


/** BufferIs
	Whether or not an object is a Buffer.
*/
#define BufferIs(x) (ClassOf (x) == BufferClass)


/** Stringy
	Stringy is used to denote that a function/method can accept a Buffer and String
	argument. It is typed as String for convenience.

	@warn
	Be careful. If you are blocking then you must protect the Buffer using
	={BufferProtect}.

*/
typedef String Stringy;
#define StringyIs(x) (StringIs (x) || BufferIs (x))


/** BufferFlag
	Flags used by Buffer objects.

	Never access these flags directly.
	Use the accessor macros provided instead.
*/
#define BufferFlagPROTECTED ContainerFlagPROTECTED
#define BufferFlagFROZEN    ContainerFlagFROZEN


/** BufferProtect
	Temporarily protect the Buffer from being modified.
*/
#define BufferPROTECTBEGIN(self, saved_scope) \
BEGIN \
	saved_scope = (self)->scope; \
	(self)->scope = SysModule; \
	FlagSet (self, BufferFlagPROTECTED); \
END

#define BufferPROTECTEND(self, saved_scope) \
BEGIN \
	(self)->scope = saved_scope; \
	if (!saved_scope) \
		FlagUnset (self, BufferFlagPROTECTED); \
END

#define __BufferProtectBegin(self) \
BEGIN \
	Buffer tHiS = self; \
	Any    sCoPe = tHiS->scope; \
	tHiS->scope = SysModule; \
	FlagSet (tHiS, BufferFlagPROTECTED)

#define __BufferProtectEnd \
	tHiS->scope = sCoPe; \
	if (!sCoPe) \
		FlagUnset (tHiS, BufferFlagPROTECTED); \
END


/** BufferIsFrozen
	Whether the Buffer may be modified.
*/
#define BufferIsFrozen(self) \
	FlagTest  (self, BufferFlagFROZEN | BufferFlagPROTECTED)


/** BufferNew
	Various Buffer constructors.

	BufferAlloc create a new empty Buffer with v{cap} capacity.

	BufferNew create a new Buffer from a C string.

	BufferNewz simply calls C{BufferNew (z, strlen (z))}.
*/
extern Buffer BufferAlloc (long cap);
extern Buffer BufferNew   (const char *z, long len);
extern Buffer BufferNewz  (const char *z);


/** BufferClear
	Clear content.
*/
extern Buffer BufferClear (Buffer self);


/** BufferExchangeTmp
	Exchange the field v{str} and adjust other fields.

	@warn
	This is a delicate function.
	Use with care.
*/
extern void BufferExchangeTmp       (Buffer self, Buffer tmp);
extern void BufferExchangeTmpString (Buffer self, String tmp);


/** BufferTmpToString
	Magically turn the Buffer into a String, that is, the v{klass} field is set
	to StringClass.

	After a call to this function you must no longer use the Buffer methods.
	You must use the String methods instead.

	@warn
	This is a delicate function.
	Use with care.
*/
extern String BufferTmpToString (Buffer self);


/** BufferMakeSpace
	Ensure there is enough room for v{more} bytes.

	Return 1 on success, If not then EOverflow is thrown and 0 is returned.

*/
extern int BufferMakeSpace (Buffer self, long more);


/** BufferPut
	Put string into the Buffer.

	All functions return 1 on success, throw EMemory and return 0 on failure.
*/
extern int BufferPut  (Buffer self, const char *s, long len);
extern int BufferPutz (Buffer self, const char *s);
extern int BufferPuts (Buffer self, Stringy s);
extern int BufferPutc (Buffer self, char c);


/** BufferInsert
	Insert string to Buffer.

	@warn
	Make sure C{i <= 0 && i <= self->len}.
	No checks are made.
*/
extern int BufferInsert (Buffer self, long i, const char *s, long n);


/** BufferDelete
	Delete v{n} bytes starting from v{i}.

	BufferConsume is C{BufferDelete (self, 0, n)}.

	Both return the actual number of bytes deleted.

	@warn
	Make sure C{0 <= i < self->len}.
	No checks are made.
*/
extern long BufferDelete  (Buffer self, long i, long n);
extern long BufferConsume (Buffer self, long n);



/** BufferChop
	Delete v{n} bytes from the head or tail.

	If v{n} is negative then delete from head.
	Delete from tail otherwise.
*/
extern void BufferChop (Buffer self, long n);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Buffer class */
extern void BufferSetup ();
extern void BufferInit  ();

#endif /*_H_Buffer_*/
