/* Class.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Class_
#define _H_Class_ 1


/** Class
	The final Class class.
*/
extern Class ClassClass;


/** TClass
	This represents an instance of the Class object, a k{class}.

	v{accel} is used for fast C classes attribute lookup. It is automatically
	built by ={ClassEndDecl}. You must never set it yourself.

	v{init} is set by ={ClassMethod} to speed-up things a bit, i.e. if set
	then we don't need to look it up.

	v{get} and v{set} are used for C classes and must be set by the C classes
	explicitly. They are the get/set attribute functions called at run-time.
*/

typedef Any (*attrget_f) (Any self, String name, int restrict);
typedef Any (*attrset_f) (Any self, String name, Any val, int restrict);

typedef struct claccel_s claccel_s;
struct TClass
{
	OBJ_HEAD;
	String     name;   /* Class name - must be the first field */
	Module     parent; /* Module where declared */
	long       id;     /* Class ID - one of CLASS_ID_XXX*/
	Class      super;  /* Super class, 0 if none */
	size_t     size;   /* Instance: size of instance struct */
	free_f     free;   /* Instance: free memory */
	mark_f     mark;   /* Instance: mark referenced objects, may be NULL */
	claccel_s *accel;  /* Initially NULL, see below */
	Array      mixin;  /* Mixin classes, initially NULL */
	attrget_f  get;    /* Get instance attribute, never NULL */
	attrset_f  set;    /* Set instance attribute, never NULL */
	Sub        init;   /* Sub __init, borrowed from attr, may be NULL */
};

struct claccel_s
{
	sub1_f fas;       /* sub __as       (x)       -> Any      */
	sub0_f fclear;    /* sub __clear              -> self     */
	sub1_f fcmp;      /* sub __cmp      (x)       -> Int      */
	sub0_f fcopy;     /* sub __copy               -> Any      */
	sub0_f fdecr;     /* sub __decr               -> Any      */
	sub1_f fdel;      /* sub __del      (x)       -> Int      */
	sub1_f fdiv;      /* sub __div      (x)       -> Any      */
	sub0_f feach;     /* sub __each               -> Iterator */
	sub1_f feq;       /* sub __eq       (x)       -> Boolean  */
	sub1_f fformat;   /* sub __format   (f)       -> String   */
	sub1_f fgetitem;  /* sub __getitem  (i)       -> Any      */
	sub2_f fgetslice; /* sub __getslice (i, l)    -> Any      */
	sub0_f fhash;     /* sub __hash               -> Int      */
	sub1_f fiand;     /* sub __iand     (x)       -> Any      */
	sub1_f fidiv;     /* sub __idiv     (x)       -> Integer  */
	sub1_f fin;       /* sub __in       (x)       -> Boolean  */
	sub0_f fincr;     /* sub __incr               -> Any      */
	subn_f finit;     /* sub __init     (*arg)    -> Null     */
	sub0_f finot;     /* sub __inot               -> Any      */
	sub1_f fior;      /* sub __ior      (x)       -> Any      */
	sub1_f fixor;     /* sub __ixor     (x)       -> Any      */
	sub0_f flen;      /* sub __len                -> Int      */
	sub1_f flike;     /* sub __like     (x)       -> Boolean  */
	sub1_f flshift;   /* sub __lshift   (x)       -> Any      */
	sub1_f fminus;    /* sub __minus    (x)       -> Any      */
	sub1_f fmul;      /* sub __mul      (x)       -> Any      */
	sub0_f fneg;      /* sub __neg                -> Any      */
	sub1_f fof;       /* sub __of       (x)       -> Any      */
	sub1_f fplus;     /* sub __plus     (x)       -> Any      */
	sub0_f fpop;      /* sub __pop                -> Any      */
	sub0_f fpos;      /* sub __pos                -> Any      */
	sub1_f fpow;      /* sub __pow      (x)       -> Any      */
	sub0_f fprint;    /* sub __print              -> Any      */
	sub1_f fpush;     /* sub __push     (x)       -> self     */
	sub1_f frem;      /* sub __rem      (x)       -> Any      */
	sub1_f frshift;   /* sub __rshift   (x)       -> Any      */
	sub2_f fsetitem;  /* sub __setitem  (i, v)    -> Any      */
	sub3_f fsetslice; /* sub __setslice (i, l, v) -> Any      */
	sub0_f fstring;   /* sub __string             -> String   */
	sub0_f ftrue;     /* sub __true               -> Boolean  */
};



/** ClassFlag
	These are the flags used by Class objects.

	Some values are shared by the ={Sub} and ={Var} classes.
	See ={Flag}.

	Do not access these flags directly.
	Use the accessor macros provided instead.
	If none is defined then it means you must not touch the flag.
*/
#define ClassFlagDOC      SymbolFlagDOC      /* document compilation? */
#define ClassFlagEXTERN   SymbolFlagEXTERN   /* C class */
#define ClassFlagPRIVATE  SymbolFlagPRIVATE  /* is module private? */
#define ClassFlagLOCAL    SymbolFlagLOCAL    /* local to block? - must be w/ ClassFlagPRIVATE */
#define ClassFlagFINAL    SymbolFlagFINAL    /* is __final? */
#define ClassFlagABSTRACT SymbolFlagABSTRACT /* is __abstract? */
#define ClassFlagDYNAMIC  Flag10             /* can create attributes at run-time? */
#define ClassFlagDESTROY  Flag11             /* defines __destroy? */
#define ClassFlagAUTO     Flag12             /* __auto create attribute? */
#define ClassFlagMIXIN    Flag13             /* is __mixin? */
#define ClassFlagNOCATCH  Flag14             /* is non-catchable exception class? */
#define ClassFlagPURE     Flag15             /* don't inherit from Object */


/** ClassIsExtern
	Whether or not this is a C level class.

	You may not set/unset this flag.
*/
#define ClassIsExtern(self) FlagTest (self, ClassFlagEXTERN)


/** ClassIsPrivate
	Whether or not this Class should only be known to within the module where it
	is declared.

	This flag must be coupled with ClassFlagLOCAL because we only allow a class
	to be declared in module level.
*/
#define ClassIsPrivate(self) FlagTest (self, ClassFlagPRIVATE)


/** ClassIsFinal
	Whether or not this Class is final.
*/
#define ClassIsFinal(self) FlagTest (self, ClassFlagFINAL)


/** ClassIsAbstract
	Whether or not this Class is an abstract.
*/
#define ClassIsAbstract(self) FlagTest (self, ClassFlagABSTRACT)


/** ClassIsMixin
	Whether or not this Class is a mixin class.
*/
#define ClassIsMixin(self) FlagTest (self, ClassFlagMIXIN)


/** ClassIsDynamic ClassIsAuto
	ClassIsDynamic checks whether or not this Class allows creation of attributes
	at run-time.

	ClassIsAuto checks whether the Class allows automatic creation of attributes
	at run-time, e.g. C{x.foo = 1} automatically creates the attribute v{foo}
	if it has not been declared.
*/
#define ClassIsDynamic(self) FlagTest (self, ClassFlagDYNAMIC)
#define ClassIsAuto(self)    FlagTest (self, ClassFlagAUTO)


/** ClassCatchable
	Whether the exception Class can not be catch-ed (at program level).
*/
#define ClassCatchable(self) (!FlagTest (self, ClassFlagNOCATCH))


/** ClassHasDestroy
	Whether the Class defines the __destroy instance method.

	The ClassFlagDESTROY flag is set by ClassEndDecl by checking whether or not
	the class declares the __destroy method.

	The purpose is to help GC performance so it does not have to do an attribute
	lookup everytime an object is destroyed.

	This means that a __destroy attribute created at run-time will not be respected
	by the GC, which we think is favorable.
*/
#define ClassHasDestroy(self) FlagTest (self, ClassFlagDESTROY)


/** ClassIsDoc
	Whether or not the Class comes from a documentation compilation.

	This is a documentation hack.

	When compiling documentation, program level classes are flagged ClassFlagEXTERN
	to make things easier and more efficient (no need to do some extra checks)
	but they are also flagged ClassFlagDOC for recognition.
*/
#define ClassIsDoc(self) FlagTest (self, ClassFlagDOC)


/** ClassIsPure
	Whether or not the class defines the __pure decorator which means that it
	must not inherit from the Object class.
*/
#define ClassIsPure(self) FlagTest (self, ClassFlagPURE)


/** ClassId
	Class IDs are currently used simply to speed things up a bit in some places.

	ClassId returns the class ID of the object.
*/
#define CLASS_ID_USER              0
#define CLASS_ID_Object            1
#define CLASS_ID_Class             2
#define CLASS_ID_Module            10
#define CLASS_ID_Var               20
#define CLASS_ID_Function          30
#    define CLASS_ID_Sub           31
#    define CLASS_ID_Method        32
#define CLASS_ID_Exception         40
#define CLASS_ID_Null              41
#define CLASS_ID_Boolean           42
#define CLASS_ID_String            50
#define CLASS_ID_Number            60
#    define CLASS_ID_Real          61
#        define CLASS_ID_Integer   62
#            define CLASS_ID_Int   63
#            define CLASS_ID_Long  64
#        define CLASS_ID_Rational  65
#        define CLASS_ID_Decimal   66
#            define CLASS_ID_Big   67
#            define CLASS_ID_Float 68
#        define CLASS_ID_Inf       69
#    define CLASS_ID_Complex       70
#define CLASS_ID_Container         100
#define CLASS_ID_Iterator          110
#define CLASS_ID_Array             120
#define CLASS_ID_Dict              130
#    define CLASS_ID_Arg           131
#define CLASS_ID_Set               140
#define CLASS_ID_Buffer            150
#define CLASS_ID_Queue             160
#define CLASS_ID_Range             170
#define CLASS_ID_Record            180
#define CLASS_ID_Struct            190
#define CLASS_ID_Stream            200
#    define CLASS_ID_File          201
#    define CLASS_ID_Socket        202
#define CLASS_ID_Thread            210
#define CLASS_ID_Dlib              220
#define CLASS_ID_Opaque            230
#define CLASS_ID_Pointer           240
#define CLASS_ID_Cfunc             250
#define CLASS_ID_Proc              260
#define CLASS_ID_Regex             270
#define CLASS_ID_Task              280
#define CLASS_ID_Tokenizer         500
#define CLASS_ID_Ast               510
#define CLASS_ID_Code              520
#define CLASS_ID_Frame             530
#define CLASS_ID_Weak              540
#define CLASS_ID_Flow              550

/* for a Class just say foo->id */
#define ClassId(o) (IntIs(o) ? CLASS_ID_Int : ((Obj)(o))->klass->id)

/* might be useful when playing with macros */
#define CLASS_ID(kls) CLASS_ID_##kls


/***/
/*----------------------------------------------------------------------------
	methods designed for the compiler/interpreter
----------------------------------------------------------------------------*/

/** ClassDeclare ClassDeclAttr ClassFindAttr
	These functions are specifically designed for the compiler. If you are not
	working with the compiler then you should not call these functions.

	ClassDeclare allocates a new Class object. The compiler must set the
	module attributes itself via ={ModuleSetAttr}.

	ClassDeclAttr is used to declare an attribute (Var or Sub), class or
	instance attribute. It checks for signatures and return 1 on success,
	0 on failure.

	ClassFindAttr looks for an attribute in the super class and mixin classes.
	It looks for class and instance attribute.
*/
extern Class ClassDeclare  (Module mod, String name, long flags);
extern int   ClassDeclAttr (Class self, String name, Symbol value);
extern Any   ClassFindAttr (Class self, String name);


/** ClassCheckPrototype ClassDelAccel
	ClassCheckPrototype check for a correct prototype.
	If listed as a standard name (e.g. "__mul"), returns the static string version
	of the name on success, NULL on failure.
	Otherwise this simply returns v{name}.

	ClassDelAccel deletes the accelerator if it exists.
	This is necessary if the accelerator (which must be a C function) is replaced
	by a Qu function. Only Code.c should be calling thise.
*/
extern String ClassCheckPrototype (Class self, String name, Symbol attr);
extern void   ClassDelAccel       (Class self, String name);



/***/
/*----------------------------------------------------------------------------
	methods designed for C modules
----------------------------------------------------------------------------*/

/** ClassNew
	This function is designed for C modules to create a new Class object.

	ClassNewz is simply a convenience function.

	This function automatically sets the attribute in the associated module
	object, i.e. C{mod->attrs}. No questions asked. No checks are made.
	It assumes that the C programmer is smart enough to know that declaring
	attributes with the same name more than once is an error.

	@warn
	Builtin classes must not call ClassNew from their xxx_setup function.
	Creating classes in initialization phase may only be done from the
	xxx_init function.

	The v{name} field in ClassNewz must be a static C string, e.g. C{"foo"}.
	It may not be a dynamic pointer.
*/
extern Class ClassNew (Module o, String name, long flags, size_t size, free_f f, mark_f m);

#define ClassNewz(mod, name, struc, ffree, fmark) \
	ClassNew (mod, StringStaticz (name), 0, sizeof (struc), \
		(free_f) ffree, (mark_f) fmark)

#define ClassNewFinalz(mod, name, struc, ffree, fmark) \
	ClassNew (mod, StringStaticz (name), ClassFlagFINAL, sizeof (struc), \
		(free_f) ffree, (mark_f) fmark)


/** ClassProp
	Create a C class property.

	@warn
	The v{namez} field in the macros must be a static C string, e.g. C{"foo"}.
	It may not be a dynamic pointer.
*/
extern Var ClassProp (Class self, String name, Any value, long flags);

#define ClassPropClass(self, name, val) \
	ClassProp (self, name, val, VarFlagCLASS)

#define ClassPropInst(self, name, val) \
	ClassProp (self, name, val, VarFlagINST)

#define ClassConst(self, name, val) \
	ClassProp (self, name, val, VarFlagCLASS | VarFlagFROZEN | VarFlagFINAL)


#define ClassPropz(self, namez, val, flags) \
	ClassProp (self, StringStaticz (namez), val, flags)

#define ClassPropClassz(self, namez, val) \
	ClassPropz (self, namez, val, VarFlagCLASS)

#define ClassPropInstz(self, namez, val) \
	ClassPropz (self, namez, val, VarFlagINST)

#define ClassConstz(self, namez, val) \
	ClassPropz (self, namez, val, VarFlagCLASS | VarFlagFROZEN | VarFlagFINAL)



/** ClassMethod
	Create a C class method.

	@warn
	The v{namez} field in the macros must be a static C string, e.g. C{"foo"}.
	It may not be a dynamic pointer.

	These functions will issue a ={FATAL} error if the attribute is a standard
	attribute, e.g. __cmp, and the signature does not match the predefined
	prototype.
*/
extern Sub ClassMethod (Class self, String name, void *cfunc, int argc, long flags);

#define ClassMethodStatic(self, name, faddr, argc) \
	ClassMethod (self, name, (void *) faddr, argc, SubFlagSTATIC)

#define ClassMethodClass(self, name, faddr, argc) \
	ClassMethod (self, name, (void *) faddr, argc, SubFlagCLASS)

#define ClassMethodInst(self, name, faddr, argc) \
	ClassMethod (self, name, (void *) faddr, argc, SubFlagINST)


#define ClassMethodz(self, namez, cfunc, argc, flags) \
	ClassMethod (self, StringStaticz (namez), (void *) cfunc, argc, flags)

#define ClassMethodStaticz(self, namez, faddr, argc) \
	ClassMethodz (self, namez, faddr, argc, SubFlagSTATIC)

#define ClassMethodClassz(self, namez, faddr, argc) \
	ClassMethodz (self, namez, faddr, argc, SubFlagCLASS)

#define ClassMethodInstz(self, namez, faddr, argc) \
	ClassMethodz (self, namez, faddr, argc, SubFlagINST)


#define ClassMethodStaticzF(self, namez, faddr, argc) \
	ClassMethodz (self, namez, faddr, argc, SubFlagSTATIC | SubFlagFINAL)

#define ClassMethodClasszF(self, namez, faddr, argc) \
	ClassMethodz (self, namez, faddr, argc, SubFlagCLASS | SubFlagFINAL)

#define ClassMethodInstzF(self, namez, faddr, argc) \
	ClassMethodz (self, namez, faddr, argc, SubFlagINST | SubFlagFINAL)


/** ClassAlias
	Create an alias for an existing attribute.
*/
extern Any ClassAlias  (Class self, String name, String alias);

#define ClassAliasz(self, namez, aliasz) \
	ClassAlias (self, StringStaticz (namez), StringStaticz (aliasz))


/** ClassSetSuper ClassSetMixin ClassEndDecl
	These functions are designed for C modules to set the super class or
	mixin class. They are also used by the compiler.

	ClassSetSuper sets the super class of a Class.

	ClassSetMixin sets the class as a mixin class.

	ClassEndDecl must be called after ClassSetSuper and ClassSetMixin to
	finalize declaration. It sets up the v{accel} field and mixes the
	attributes for faster lookup.

	All return 1 on success, throw an exception and returns 0 on failure.
	Failure includes, for example, when a method tagged k{__final} is overridden,
	a class tagged __final is subclassed, etc.

	@xmp
		class Foo is Bar (Baz)
		#            |    |
		#            |    ClassSetMixin (Foo, Baz)
		#            |
		#            ClassSetSuper (Foo, Bar)
*/
extern int ClassSetSuper (Class self, Class sup);
extern int ClassSetMixin (Class self, Class klass);
extern int ClassEndDecl  (Class self);


/** ClassFindClass
	Find a superclass or a mixin by name.
*/
extern Class ClassFindClass (Class self, String name);


/** ClassMixes
	Whether the class mixins a specified class.
*/
#define ClassMixes(self, c) (class__mixes (self, c) == True)


/** ClassTree
	Return an ={Array} consiting all the classes in this class's hierachy.
	The first item is this class.
*/
extern Array ClassTree (Class self);


/** ClassInitNew
	ClassInitNew is a convenience functions designed to be called from
	within the f[new} methods of C classes that allow subclassing.

	If calls the __init method if one is defined. It does nothing otherwise.
*/
extern Any ClassInitNew (Class self, Any inst, int argc, Any *argv);


/***/
/*----------------------------------------------------------------------------
	methods for general purposes
----------------------------------------------------------------------------*/

/** ClassModule
	Get the ={Module} object where the class is declared.

	For builtin classes, this always return ={SysModule}.
*/
#define ClassModule(self) (((Class) (self))->parent)


/** ClassOf
	Get the Class of an object.

	@warn
	Watch for the side-effect of this macro.
*/
#define ClassOf(self) (IntIs(self) ? IntClass : ((Obj)(self))->klass)


/** ClassIs ClassIsa ClassISA ClassIsSuperOrMixin
	ClassIs tests whether or not an object is a Class.

	ClassIsa tests whether or not the Class is a super class of another Class.
	It is provided because it is more efficient than the script level method
	Class_isa.

	ClassISA is designed for readability since all of our class names in C code
	must end with I{Class}, e.g. ={Array}, ={Int}, ={Set}, etc.
	so we can say C{ClassISA (x, Array)} instead of C{ClassIsa (x, ArrayClass)}.

	ClassIsSuperOrMixin checks whether v{x} is a mixin or superclass of self.

	@warn
	Watch for the side-effect of the macros.
*/
#define ClassIs(self)       (IntIs(self) ? 0 : (((Obj)(self))->klass == ClassClass))
#define ClassISA(self, kls) ClassIsa ((self), kls##Class)
extern int ClassIsa            (Class self, Class x);
extern int ClassIsSuperOrMixin (Class self, Class x);


/** ClassDepth
	Return the depth of the Class in its class hierarchy. A top-level class
	such as ={Array}, for example, will have a value of 0.
*/
extern int ClassDepth (Class self);


/***/
/*----------------------------------------------------------------------------
	methods specifically designed for builtin classes
----------------------------------------------------------------------------*/

/** Class_setup
	This is called internally by builtin classes to setup the ={TClass} struct.

	This function must be called by all builtin classes from their xxx_setup
	function. Never from xxx_init.
*/
extern void Class_setup (Class self, long flags, size_t size, free_f f, mark_f m);

#define ClassSetup(klass, struc, ffree, fmark) \
	Class_setup (klass, 0, sizeof (struc), (free_f) ffree, (mark_f) fmark)

#define ClassSetupSystem(klass, struc, ffree, fmark) \
	Class_setup (klass, ClassFlagFINAL | ClassFlagABSTRACT, sizeof (struc), (free_f) ffree, (mark_f) fmark)

#define ClassSetupFinal(klass, struc, ffree, fmark) \
	Class_setup (klass, ClassFlagFINAL, sizeof (struc), (free_f) ffree, (mark_f) fmark)

#define ClassSetupAbstract(klass, struc) \
	Class_setup (klass, ClassFlagABSTRACT, sizeof (struc), 0, 0)

#define ClassSetupBase(klass, struc) \
	Class_setup (klass, ClassFlagFINAL | ClassFlagABSTRACT, sizeof (struc), 0, 0)

#define ClassSetupMixin(klass, struc) \
	Class_setup (klass, ClassFlagMIXIN, sizeof (struc), 0, 0)


/***/
/*--------------------------------------------------------------------------*/

/* This is called internally by qu.c to initialize all builtin classes. */
extern void ClassInit ();

/* published for efficiency */
extern Boolean class__mixes (Class self, Class c);

/* for Object.c */
extern Array ClassMixins (Class self);

/* for Reduce.c */
extern int ClassAccessible (Class self, Symbol from);

/***/
#define _ATTRS_HEADER
#include "attrs.h"

#endif /*_H_Class_*/
