/* Code.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Code_
#define _H_Code_ 1


/** Code
	This is the final Code class.

	Only ={Module} and ={Sub} objects have Code objects.
	={Class} objects share it with the module where they are declared.
	This means that the initialization code within the class block is executed
	when the module code is initialized.

	A Code object can not live alone because the objects contained in
	the bytecode array (C{code->bytes}) are stored in C{Module->literals}.

	Sub objects will eventually refer to a Module object so C{Module->literals}
	is always automatically referred by a Code object.

	@xmp
		Sub -> Module
		Sub -> ... -> Class -> Module
*/
extern Class CodeClass;


/** TCode
	This represents an instance of the Code class.
*/
struct TCode
{
	OBJ_HEAD;
	unsigned char *bytes;     /* Bytecode */
	Symbol         parent;    /* The associated Module or Sub object */
	int            len;       /* Bytecode length */
	int            stacksize; /* Stack size required (including for locals) */
};


/** CodeIs
	Whether or not an object is a Code object.
*/
#define CodeIs(o) (ClassOf(o) == CodeClass)


/** CodeFlag
	Flags used by Code objects.

	Do not access these flags directly.
	Use the accessor macros provided instead.
*/
#define CodeFlagJIT Flag0  /* 'bytes' point to JIT code */


/** CodeIsJit
	Whether this is JIT-ed code.
*/
#define CodeIsJit(self) FlagTest (self, CodeFlagJIT)


/** CodeGen
	Generate the Code object for a module.

	This function is called by the compiler.
	You must not call this function directly.
*/
extern int CodeGen (Module mod);


/** CodePcName CodePcType
	Get the name or type of an instruction.

	If it does not exist then ENone is thrown.

	See #{PC} for the list.
*/
extern String CodePcName (int type);
extern Int    CodePcType (int type);


/** CodeModule
	Return the Module that own this Code.

	If this is Module level code then C{self->parent} is returned, otherwise
	returns C{SubModule (self->parent)}.
*/
extern Module CodeModule (Code self);


/** CodePrint
	Print code to v{stdout}.

	CodePrintLiteral prints any literal.
	It is intended for Ast so we get a uniform print out.
*/
extern Null CodePrint        (Code self);
extern int  CodePrintLiteral (Any x);


/***/
/*-----------------------------------------------------------------------

	INSTRUCTION LIST

-----------------------------------------------------------------------*/

/** PC
	List of instructions.
	An instruction may have as much as 3 arguments.

	See ={Bytecode} for details.
*/
#define CODE_IN_HEADER
#define p_(name, arg) PC_##name,
enum
{
#include "Codes.h"
};
#undef p_
#undef CODE_IN_HEADER

/* for optimizers - be careful with the boundaries if used directly */
extern String PcName [PC_LAST_INSTRUCTION + 1];
extern int    PcType [PC_LAST_INSTRUCTION + 1];


/** CODE_BLOCK
	Type of blocks.
*/
#define CODE_BLOCK_NONE     0   /* not a block */
#define CODE_BLOCK_SWITCH   1
#define CODE_BLOCK_DO       2
#define CODE_BLOCK_LOOP     3
#define CODE_BLOCK_WHILE    4
#define CODE_BLOCK_FOR      5
#define CODE_BLOCK_RANGE    6
#define CODE_BLOCK_ATOMIC   7
#define CODE_BLOCK_TRY      8
#define CODE_BLOCK_CATCH    9
#define CODE_BLOCK_ELSETRY  10
#define CODE_BLOCK_FINALLY  11

#define CODE_NUMBLOCKS 12


/***/
/*----------------------------------------------------------------------------
	JIT
----------------------------------------------------------------------------*/

/** CodeJit
	These are the interfaces to JIT code.

	A JIT engine must set all of these variables.

	CodeJitGenerate is called by the compiler to generate JIT code.
	It must return 1 on success, 0 on failure. If it is generating ={SubIsJit}
	then it must free C{Code->bytes} by using ={GcMemFree} and replace it
	with JIT code then call ={SubJitSet} to give the Sub the appropriate flags
	and callers. If it produces ={SubIsExtern}, then it must set ={Sub}->value
	to the JIT code then call ={SubJitSetExtern}.

	CodeJitPrint is called by code__print for JIT code.
	It must return 1 on success, 0 on failure.
*/
typedef int (*codejitgen_f  ) (Sub  u);
typedef int (*codejitprint_f) (Code c);

extern codejitgen_f   CodeJitGenerate;
extern codejitprint_f CodeJitPrint;

/* call it */
#define CodeJitCall(f) (((subc1_f) (f)->code->bytes) (f))


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Code class. */
extern void CodeSetup ();
extern void CodeInit  ();

/* for the interpreter - list of binop functions */
extern binop_f CodeBinopList [PC_LAST_INSTRUCTION + 1];

#endif /*_H_Code_*/
