/* Codes.h - Copyright (c) Marc Krisnanto */

/* --- designed to be included more than once */

/** Bytecode
	List of instructions.
*/
#define PC_0  0x000  /* no argument */
#define PC_L  0x011  /* 1 long argument */
#define PC_A  0x021  /* 1 Any argument */
#define PC_LL 0x112  /* 2 long arguments */
#define PC_LA 0x122  /* 1 long argument, 1 Any argument */
#define PC_AL 0x212  /* 1 Any argument, 1 long argument */
#define PC_AA 0x222  /* 2 Any arguments */

p_(end_marker, PC_0)
	/*
		Instruction 0 is used to mark the end of the bytecode.
		It is never executed.
	*/

p_(ARRAY, PC_L)
	/*
		Create a new Array from items on the stack.
		The argument is the number of items on the stack.
		This instruction never fails (except for out of memory) because the
		compiler has set a limit on the number of items to COMPILE_MAXARGS.
		See Compile.h.

		[x, y, z]

		| z  <-- SP
		| y
		| x                 array <-- SP
		| ...               ...

	*/

p_(ARG, PC_L)
	/*
		Create a new Arg from items on the stack.
		The argument is the number of items on the stack.
		8 means 4 key/value pairs.
		This instruction never fails (except for out of memory) because the
		compiler has set a limit on the number of items to COMPILE_MAXARGS.
		See Compile.h.

		foo (a:b, c:d)

		| d  <-- SP
		| c
		| b
		| a                 arg <-- SP
		| ...               ...
	*/

p_(ARGC, PC_0)
	/*
		Push the number of arguments passed to the function/method on
		the stack. This instruction is generated by the keyword __argc__.

		|                 argc  <-- SP
		| ...  <-- SP     ...
	*/

p_(AS, PC_0)
p_(ASLL, PC_LL)
p_(ASLX, PC_LA)
	/*
		This instruction is generated by the binary "as" operator.
		It translates into Sys_as (x, y).

		x as y

		| y  <-- SP
		| x                 result  <-- SP
		| ...               ...
	*/

p_(AUTOATTR, PC_0)
	/*
		If the value on top of the stack is a function/method then
		call it without any argument. If not, drop the value on top
		of the stack.

		x.y

		| y  <-- SP
		| x                 result  <-- SP
		| ...               ...
	*/

p_(AUTOATTREX, PC_0)
	/*
		This does the same as AUTOATTR except that the object is not dropped.
		It is used for assignable methods.
		It corresponds to GETATTR.

		a.b &= c  where a.b is a method

		| name <-- SP     attr <-- SP    result  <-- SP
		| obj             obj            obj
		| ...             ...            ...
	*/

p_(AUTOATTREX2, PC_0)
	/*
		This does the same as AUTOATTREX except for the stack arrangement.
		It is used for assignable methods.
		It corresponds to GETATTR2.

		a.(b) &= c  where a.b is a method

		|                 attr <-- SP    result <-- SP
		| name <-- SP     name           name
		| obj             obj            obj
		| ...             ...            ...
	*/

p_(BINOP, PC_L)
	/*
		Special binary operators that do not yet have a unique bytecode
		or corresponding method.
		The argument is an index to CodeBinopList.

		x -:- y

		| y  <-- SP
		| x                 result  <-- SP
		| ...               ...
	*/

p_(BREAK, PC_L)
	/*
		Jump after block.
		The argument is the number of blocks to skip.
		This instruction is generated by the 'break' keyword.
		It must reset the stack pointer.

		break x

		| ... <-- SP
		| ...
		| ...
		| ... <-- BP      ... <-- BP = SP
	*/

p_(CALL, PC_L)
	/*
		Call object using the items on the stack.
		The number of items is this instruction's argument.

		u (x, y, z)

		| z  <-- SP
		| y
		| x
		| u               result <-- SP
		| ...             ...
	*/

p_(CALLFLAT, PC_L)
	/*
		Call object using the items on the stack which contains flattened items.
		The number of items is this instruction's argument.

		u (x, *y, z)

		| z  <-- SP
		| y
		| x
		| u               result <-- SP
		| ...             ...

		CALLXFLAT does the same but the second argument is a static Sub to call
		and the stack layout is:

		u (x, y, z)

		| z  <-- SP
		| y
		| x               result <-- SP
		| ...             ...
	*/

p_(CALLATTR, PC_L)
	/*
		Call method using the items on the stack.
		The number of items is this instruction's argument.

		u.v (x, y, z)

		| z  <-- SP
		| y
		| x
		| v
		| u               result <-- SP
		| ...             ...
	*/

p_(CALLATTRFLAT, PC_L)
	/*
		Call method using the items on the stack which contains flattened items.
		The number of items is this instruction's argument.

		u.v (x, *y, z)

		| z  <-- SP
		| y
		| x
		| v
		| u               result <-- SP
		| ...             ...
	*/

p_(CATCH, PC_LA)
	/*
		Catch exceptions.
		The first argument is the jump target to the next exception.
		The second is the exception list. It is one of:
		- nil which means to catch all exceptions
		- an exception class
		- an Array of exception classes
		This instruction does not affect the stack.
		If the exception thrown does not match then jump, otherwise
		goto the next instruction.

		catch x, y
	*/

p_(CATCHSET, PC_A)
	/*
		Set the current exception to local variable.
		This instruction does not affect the stack.

		catch x, y -> v
	*/

p_(CLOSURE, PC_A)
	/*
		Create closure object.
		The argument is the Sub object to use.
		This instruction never fails, except for out of memory.

		lambda (x) return x + y

		|                 object <-- SP
		| ... <-- SP      ...
	*/

p_(CMP, PC_0)
p_(CMPLL, PC_LL)
p_(CMPLX, PC_LA)
	/*
		Compare the two top values on the stack.
		This instruction is generated by the <=> operator.
		It translates to Sys_cmp (x, y).
		It never fails and always return an Int: < 0, 0, or > 0.

		x <-> y

		| y  <-- SP
		| x               -1/0/1 <-- SP
		| ...             ...
	*/

p_(CONCAT, PC_0)
p_(CONCATLL, PC_LL)
p_(CONCATLX, PC_LA)
	/*
		Concatenate the top two values on the stack as strings.
		This instruction is generated by the _ operator.
		It translates into Sys_concat (x.__str, y.__str).
		It never fails except for out of memory.

		x _ y

		| y   <-- SP
		| x               str  <-- SP
		| ...             ...
	*/

p_(CONTINUE, PC_L)
	/*
		Jump to where the block continues.
		This instruction is generated by the 'continue' and 'redo' keywords.
		The argument is the number of blocks to break.
		This instruction must restore the stack before jumping.

		continue x

		| ... <-- SP
		| ...
		| ...
		| ... <-- BP      ... <-- BP = SP
	*/

p_(DECR, PC_0)
p_(DECRLPOST, PC_L)
p_(DECRLPRE, PC_L)
	/*
		Perform an decrement on the two value on the stack.
		This is generated by the -- operator.
		It translates into Sys_decr (x).

		--x

		| x  <-- SP       result  <-- SP
		| ...             ...

		DECRLPOST and DECRLPRE are the same as INCR except that the argument
		is a lexical variable index for post/pre-increment
	*/


p_(DICT, PC_L)
	/*
		Create a new Dict from items on the stack.
		The argument is the number of items on the stack.
		8 means 4 key/value pairs.
		This instruction never fails (except for out of memory) because the
		compiler has set a limit on the number of items to COMPILE_MAXARGS.
		See Compile.h.

		[a:b, c:d]

		| d  <-- SP
		| c
		| b
		| a                 dict <-- SP
		| ...               ...
	*/

p_(DISCARD, PC_0)
	/*
		Discard the value on top of the stack.
		This instruction is generated for each statement.

		| a  <-- SP
		| ...             ... <-- SP
	*/

p_(DIV, PC_0)
p_(DIVLL, PC_LL)
p_(DIVLX, PC_LA)
	/*
		Divide the two top values on the stack.
		This instruction is generated by the / operator.
		It translates to Sys_div (x, y).

		x / y

		| y  <-- SP
		| x               result <-- SP
		| ...             ...
	*/

p_(DROP1, PC_0)
p_(DROP1LL, PC_LL)
p_(DROP1LX, PC_LA)
	/*
		Drop the top value on the stack.

		| a  <-- SP
		| b               a <-- SP
		| ...             ...
	*/

p_(DROP2, PC_0)
	/*
		Drop the top value on the stack.

		| a  <-- SP
		| b
		| c               a <-- SP
		| ...             ...
	*/

p_(DROP3, PC_0)
	/*
		Drop the top value on the stack.

		| a  <-- SP
		| b
		| c
		| d               a <-- SP
		| ...             ...
	*/

p_(DUP1, PC_0)
	/*
		Duplicate the top value on the stack.

		|                 a  <-- SP
		| a   <-- SP      a
		| ...             ...
	*/

p_(DUP2, PC_0)
	/*
		Duplicate the top 2 values on the stack.

		|                 a  <-- SP
		|                 b
		| a   <-- SP      a
		| b               b
		| ...             ...
	*/

p_(DUP3, PC_0)
	/*
		Duplicate the top 3 values on the stack.

		|                 a  <-- SP
		|                 b
		|                 c
		| a   <-- SP      a
		| b               b
		| c               c
		| ...             ...
	*/

p_(EACH, PC_0)
	/*
		Get an iterator for the object on top of the stack.
		This instruction is generated by the 'each' keyword.
		It translates into Sys_each (x).

		each x

		| x  <-- SP       iter  <-- SP
		| ...             ...
	*/

p_(END, PC_0)
	/*
		Return to caller.
		This instruction is generated at the end of all bytecodes.
		It does not change the stack.
	*/

p_(ENTER, PC_LL)
	/*
		Enter a block.
		The first argument is the target to jump to when leaving the block.
		The second argument is the type of the block (one of CODE_BLOCK_XXX).
		This instruction does not affect the stack.
	*/

p_(EQ, PC_0)
p_(EQLL, PC_LL)
p_(EQLX, PC_LA)
	/*
		Compare the two top values on the stack for equality.
		This instruction is generated by the == operator.
		It translates into Sys_eq (x, y).
		It never fails and always return true or false.

		| y  <-- SP
		| x               true/false  <-- SP
		| ...             ...
	*/

p_(FORMAT, PC_L)
	/*
		Create formatted string.
		This instruction is generated by the %% operator.
		It translates into Sys_format (x, y, z ...).
		The argument is the number of items on the stack.

		x %% (y, z)

		| z  <-- SP
		| y
		| x               result <-- SP
		| ...             ...
	*/

p_(FLAT, PC_0)
	/*
		Flatten the value on top of the stack.
		The value on top of the stack must be an Array.
		A VarNewFlattener object is created.

		| a    <-- SP     var   <-- SP
		| ...             ...
	*/

p_(FRAME, PC_0)
	/*
		Push the Frame object on the stack.
		This instruction is generated by the keyword __frame__.

		|                 frame  <-- SP
		| ...  <-- SP     ...
	*/

p_(GE, PC_0)
p_(GELL, PC_LL)
p_(GELX, PC_LA)
	/*
		Compare the two top values on the stack.
		This instruction is generated by the >= operator.
		It translates into Sys_ge (x, y).
		It never fails and always return true or false.

		x >= y

		| y  <-- SP
		| x               true/false  <-- SP
		| ...             ...
	*/

p_(GETATTR, PC_0)
	/*
		Get attribute of the two top values on the stack.
		This instruction is generated by the . operator.
		It translates into iSys_getattr (obj, name).
		The running frame object is used to determine accessibility.
		Note that the value on the stack can be anything. If it is not
		a String then an exception must be thrown.
		This instruction is always followed by AUTOATTR or CALLATTR.

		foo.(bar || baz)

		| name <-- SP     attr <-- SP
		| obj             obj
		| ...             ...
	*/

p_(GETATTR2, PC_0)
	/*
		This is the same as GETATTREX except for the stack arrangement.
		It is generated by some expressions.

		foo.(baz) <-> bar[far]

		|                 attr <-- SP
		| name <-- SP     name
		| obj             obj
		| ...             ...
	*/

p_(GETATTRX, PC_A)
	/*
		This is the same as GETATTR except that the attribute name is
		the argument of this instruction and is always a String.

		foo.bar

		|                 attr <-- SP
		| obj <-- SP      obj
		| ...             ...

		This instruction is always followed by AUTOATTR or CALLATTR.
		That is the reason of the stack arrangement.
	*/

p_(GETATTRSELF, PC_A)
	/*
		This is the same as GETATTRX except that the object is 'self'.

		self.bar

		|                 attr <-- SP
		|                 self
		| ... <-- SP      ...

		This instruction is always followed by AUTOATTR or CALLATTR.
		That is the reason of the stack arrangement.
	*/

p_(GETITEM, PC_0)
	/*
		Get item of the two top values on the stack.
		This instruction is generated by the [] operator.
		It translates into Sys_getitem (x, y).

		x [y]

		| y  <-- SP
		| x               result <-- SP
		| ...             ...
	*/

p_(GETITEM2, PC_0)
	/*
		This is the same as GETITEM except for the stack arrangement.
		It is generated by some expressions.

		a[b] <-> c[d]

		obj[x]

		|                 result <-- SP
		| x   <-- SP      x
		| obj             obj
		| ...             ...
	*/


p_(GETITEMX, PC_A)
	/*
		This is the same as GETITEM except that the index is the argument
		of this instruction.

		obj[x]

		| obj <-- SP      result <-- SP
		| ...             ...
	*/

p_(GETITEML, PC_L)
	/*
		Same as GETITEM except that the argument is a lexical variable index.

		x [y]

		| y               result <-- SP
		| ...             ...
	*/

p_(GETITEMLX, PC_LA)
	/*
		Same as GETITEM except that the argument is a lexical variable index
		and the index is a literal value.

		|                 result <-- SP
		| ... <-- SP      ...
	*/

p_(GETGLOBAL, PC_A)
	/*
		Push module/class level variable value on top of the stack.
		The argument is the Var object.

						value <-- SP
		... <-- SP      ...
	*/

p_(GETLEXICAL, PC_L)
	/*
		Push lexical variable value on top of the stack.
		The argument is the index (location) of the value on the local
		variable stack.

		|                 value <-- SP
		| ... <-- SP      ...
		| ...
		| ... <-- BP
		| vn         --+
		| ...          |
		| ...          +-- local variable stack
		| v1           |
		| v0  <-- SS --+
	*/

p_(GETSLICE, PC_0)
	/*
		Get slice from the top values on the stack.
		This instruction is generated by the [,] operator.
		This translates into Sys.__getslice (x, y, z).

		x [y, z]

		| z  <-- SP
		| y
		| x               result <-- SP
		| ...             ...
	*/

p_(GETSLICE2, PC_0)
	/*
		This is the same as GETSLICE except for the stack arrangement.
		It is generated by some expressions.

		x [y, z] <-> foo

		|                 result <-- SP
		| z  <-- SP       z
		| y               y
		| x               x
		| ...             ...
	*/

p_(GETSWITCH, PC_0)
	/*
		Put the switch block value on top of the stack.
		This instruction is generated by the 'case' keyword.

		case x

		|                 value  <-- SP
		| ...  <-- SP     ...
	*/

p_(GT, PC_0)
p_(GTLL, PC_LL)
p_(GTLX, PC_LA)
	/*
		Compare the two top values on the stack.
		This instruction is generated by the > operator.
		It translates into Sys_gt (x, y).
		It never fails and always return true or false.

		x >= y

		| y  <-- SP
		| x               true/false  <-- SP
		| ...             ...
	*/

p_(IAND, PC_0)
p_(IANDLL, PC_LL)
p_(IANDLX, PC_LA)
	/*
		Perform bitwise/itemwise-and operation on the top two values
		on the stack.
		This instruction is generated by the & operator.
		It translates into Sys_iand (x, y).

		x & y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(IDIV, PC_0)
p_(IDIVLL, PC_LL)
p_(IDIVLX, PC_LA)
	/*
		Perform integer division with the top two values on the stack.
		This instruction is generated by the \ operator.
		It translates into Sys_idiv (x, y).

		x \ y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(IOR, PC_0)
p_(IORLL, PC_LL)
p_(IORLX, PC_LA)
	/*
		Perform bitwise/itemwise-or operation on the top two values
		on the stack.
		This instruction is generated by the | operator.
		It translates into Sys_ior (x, y).

		x | y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(IN, PC_0)
p_(INLL, PC_LL)
p_(INLX, PC_LA)
	/*
		Perform an inclusion test on the top two values on the stack.
		This is generated by the 'in' keyword.
		It translates into Sys__in (x, y).

		y in x

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/


p_(INCR, PC_0)
p_(INCRLPOST, PC_L)
p_(INCRLPRE, PC_L)
	/*
		Perform an increment on the top value on the stack.
		This is generated by the ++ operator.
		It translates into Sys_incr (x).

		++x

		| x  <-- SP       result  <-- SP
		| ...             ...

		INCRLPOST and INCRLPRE are the same as INCR except that the argument
		is a lexical variable index for post/pre-increment
	*/

p_(INOT, PC_0)
	/*
		Perform bitwise/itemwise-not operation on the top two values
		on the stack.
		This instruction is generated by the ~ operator.
		It translates into Sys_inot (x).

		~x

		| x  <-- SP       result  <-- SP
		| ...             ...
	*/

p_(INITMOD, PC_A)
	/*
		Initialize module.
		This instruction does not change the stack but it consumes one
		temporary slot while in progress.
		This is generated by the 'use' keyword for modules that have not
		yet been initialized.
	*/

p_(IS, PC_0)
p_(ISLL, PC_LL)
p_(ISLX, PC_LA)
	/*
		Perform an 'is a' test on the two top values on the stack.
		This is generated by the 'is' keyword.
		It translates into Sys_is (x, y).
		It never fails and always return true or false.

		x is y

		| y  <-- SP
		| x               true/false  <-- SP
		| ...             ...
	*/

p_(ITERNEXT, PC_A)
	/*
		Get the next iterator value.
		The argument is the thing to set. It is one of:
		- Var object = set global variable
		- Int = set a lexical variable
		- Array of Var/Int to set more than one variables
		This instruction does not affect the stack.
	*/

p_(IXOR, PC_0)
p_(IXORLL, PC_LL)
p_(IXORLX, PC_LA)
	/*
		Perform bitwise/itemwise-xor operation on the top two values
		on the stack.
		This instruction is generated by the ^ operator.
		It translates into Sys_ixor (x, y).

		x ^ y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/


p_(JMP, PC_L)
	/*
		Preform an unconditional jump.
		The argument is the address to jump to.
	*/

p_(JMPFALSE, PC_L)
	/*
		Discard value on top of the stack. Jump if it was false.
		The argument is the address to jump to.

		| x   <-- SP
		| ...             ...  <-- SP

	*/

p_(JMPTRUE, PC_L)
	/*
		Discard value on top of the stack. Jump if it was true.
		The argument is the address to jump to.

		| x   <-- SP
		| ...             ...  <-- SP

	*/

p_(LE, PC_0)
p_(LELL, PC_LL)
p_(LELX, PC_LA)
	/*
		Compare the two top values on the stack.
		This instruction is generated by the <= operator.
		It translates into Sys_le (x, y).
		It never fails and always return true or false.

		x <= y

		| y  <-- SP
		| x               true/false  <-- SP
		| ...             ...
	*/

p_(LEAVE, PC_0)
	/*
		Leave the current block and jump to the address set by ENTER.
		This instruction resets the stack pointer.

		| ... <-- SP
		| ...
		| ... <-- BP      ... <-- BP = SP
	*/

p_(LIKE, PC_0)
p_(LIKELL, PC_LL)
p_(LIKELX, PC_LA)
	/*
		Compare the two top values on the stack for likeness.
		This instruction is generated by the === operator.
		It translates into Sys_like (x, y).
		It never fails and always return true or false.

		x === y

		| y  <-- SP
		| x               true/false  <-- SP
		| ...             ...
	*/

p_(LINE, PC_L)
	/*
		Set the current line number.
		The argument is the line number to set.
		This instruction does not affect the stack.
	*/

p_(LITERAL, PC_A)
	/*
		Push object on stack.
		The argument is the object to push.

		|                 value <-- SP
		| ... <-- SP      ...
	*/

p_(LSHIFT, PC_0)
p_(LSHIFTLL, PC_LL)
p_(LSHIFTLX, PC_LA)
	/*
		Perform bitwise/itemwise-left-shift operation on the top two
		values on the stack.
		This instruction is generated by the << operator.
		It translates into Sys_lshift (x, y).

		x << y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(LT, PC_0)
p_(LTLL, PC_LL)
p_(LTLX, PC_LA)
	/*
		Compare the two top values on the stack.
		This instruction is generated by the < operator.
		It translates into Sys_lt (x, y).
		It never fails and always return true or false.

		x < y

		| y  <-- SP
		| x               true/false  <-- SP
		| ...             ...
	*/

p_(MINUS, PC_0)
p_(MINUSLL, PC_LL)
p_(MINUSLX, PC_LA)
	/*
		Perform subtraction on the two top values on the stack.
		This instruction is generated by the - operator.
		It translates into Sys_minus (x, y).

		x - y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(MUL, PC_0)
p_(MULLL, PC_LL)
p_(MULLX, PC_LA)
	/*
		Perform multiplication on the two top values on the stack.
		This instruction is generated by the * operator.
		It translates into Sys_mul (x, y).

		x * y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(NE, PC_0)
p_(NELL, PC_LL)
p_(NELX, PC_LA)
	/*
		Compare the two top values on the stack for inequality.
		This instruction is generated by the != operator.
		It translates into Sys_ne (x, y).
		It never fails and always return true or false.

		x != y

		| y  <-- SP
		| x               true/false  <-- SP
		| ...             ...
	*/

p_(NEG, PC_0)
	/*
		Perform unary negation on the top value on the stack.
		This instruction is generated by the unary - operator.
		It translates into Sys_neg (x).

		-x

		| x  <-- SP       result  <-- SP
		| ...             ...
	*/

p_(NOP, PC_0)
	/*
		Does nothing.
	*/

p_(NOT, PC_0)
	/*
		Perform boolean inversion on the top value on the stack.
		This instruction is generated by the unary ! operation.
		It translates into (Sys_true (x) == True) ? False : True.
		It never fails and always return True or False.

		!x

		| x <-- SP        true/false <-- SP
		| ...             ...
	*/

p_(OF, PC_0)
p_(OFLL, PC_LL)
p_(OFLX, PC_LA)
	/*
		This instruction is generated by the binary "of" operator.
		It translates into Sys_of (x, y).

		x of y

		| y  <-- SP
		| x                 result  <-- SP
		| ...               ...
	*/

p_(PARAM, PC_LL)
	/*
		Set function parameters default value.
		The argument is the index of the lexical variable to set.
		This instruction does not affect the stack.

		foo (a, b = 1)

			if __argc__ < 1
				b = 1
			;;
	*/

p_(PASS, PC_0)
	/*
		Give-up CPU for other threads.
		This instruction is generated by the 'pass' keyword.
		It does not change the stack.
	*/

p_(PLUS, PC_0)
p_(PLUSLL, PC_LL)
p_(PLUSLX, PC_LA)
	/*
		Perform addition on the two top values on the stack.
		This instruction is generated by binary + operator.
		It translates into Sys_plus (x, y).

		x + y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(POP, PC_0)
	/*
		Perform a pop operation on the value on top the stack.
		This instruction is generated by the >>> operator.
		It translates into y = Sys_pop (x).

		x >>> y

		| x  <-- SP       result  <-- SP
		| ...             ...
	*/

p_(POS, PC_0)
	/*
		Perform unary positive on top value on the stack.
		This instruction is generated by the unary + operator.
		It translates into Sys_pos (x).

		+x

		| x   <-- SP      result  <-- SP
		| ...             ...
	*/

p_(POW, PC_0)
p_(POWLL, PC_LL)
p_(POWLX, PC_LA)
	/*
		Perform math power on the two top values on the stack.
		This instruction is generated by the ** operator.
		It translates into Sys_pow (x, y).

		x ** y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(PRINT, PC_A)
	/*
		Print the object.
		This instruction does not change the stack.
	*/

p_(PRINTTOP, PC_0)
	/*
		Print the object on top of the stack then discard it.
		Printing does HTML escaping, '<' -> '&lt;', etc.

		| x  <-- SP
		| ...             <-- SP
	*/

p_(PRINTHTML, PC_0)
	/*
		Same as PRINTTOP but does HTML escaping, '<' -> '&lt;', etc.
	*/

p_(PUSH, PC_0)
p_(PUSHLL, PC_LL)
p_(PUSHLX, PC_LA)
	/*
		Perform a push operation on the two top values on the stack.
		This instruction is generated by the <<< operator.
		It translates into Sys_push (x, y).

		x <<< y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(RANGE, PC_0)
	/*
		Create a Range object from the two top values on the stack.
		This instruction is generated by the .. operator.
		It translates into Range (x, y).

		x .. y

		| y  <-- SP
		| x               range  <-- SP
		| ...             ...
	*/

p_(RATIO, PC_0)
p_(RATIOLL, PC_LL)
p_(RATIOLX, PC_LA)
	/*
		Create a Rational number from the two top values on the stack.
		This instruction is generated by the _/ operator.
		It translates into Sys__ratio (x, y).

		x _/ y

		| x  <-- SP
		| y               result  <-- SP
		| ...             ...
	*/

p_(REM, PC_0)
p_(REMLL, PC_LL)
p_(REMLX, PC_LA)
	/*
		Perform remainder on the two top values on the stack.
		This instruction is generated by the % operator.
		It translates into Sys_rem (x, y).

		x - y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(RET, PC_0)
	/*
		Set the top value on the stack as the function return value.
		This is generated by the 'return' keyword.
		It does not change the stack.

		return foo

		| x   <-- SP      x  <-- SP
		| ...             ...
	*/

p_(RIN, PC_0)
p_(RINLL, PC_LL)
p_(RINLX, PC_LA)
	/*
		This the inverse of IN.
		It translates into Sys__rin (x, y).

		y in x

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/

p_(ROT3, PC_0)
	/*
		Rotate the top 3 values on the stack.

		| c  <-- SP       a  <-- SP
		| b               c
		| a               b
		| ...             ...
	*/

p_(ROT4, PC_0)
	/*
		Rotate the top 4 values on the stack.

		| d  <-- SP       a  <-- SP
		| c               d
		| b               c
		| a               b
		| ...             ...
	*/

p_(ROT4X, PC_0)
	/*
		Rotate the top 4 values on the stack.
		This is generated by a post-increment/decrement operator.

		| d  <-- SP       c  <-- SP
		| c               b
		| b               a
		| a               d
		| ...             ...
	*/

p_(RSHIFT, PC_0)
p_(RSHIFTLL, PC_LL)
p_(RSHIFTLX, PC_LA)
	/*
		Perform bitwise/itemwise-right-shift operation on the top two
		values on the stack.
		This instruction is generated by the >> operator.
		It translates into Sys_rshift (x, y).

		x >> y

		| y  <-- SP
		| x               result  <-- SP
		| ...             ...
	*/


p_(SELF, PC_0)
	/*
		Push the value of self on stack.
		This instruction is generated by the keyword 'self'.

		self

						self <-- SP
		... <-- SP      ...
	*/

p_(SET, PC_L)
	/*
		Create a new Set from items on the stack.
		The argument is the number of items on the stack.
		This instruction never fails (except for out of memory) because the
		compiler has set a limit on the number of items to COMPILE_MAXARGS.
		See Compile.h.

		{x, y, z}

		| z  <-- SP
		| y
		| x                 set  <-- SP
		| ...               ...

	*/

p_(SETATTR, PC_0)
	/*
		Set attribute of the two top values on the stack.
		It translates into iSys_setattr (x, y, z).
		The running frame object is used to determine accessibility.
		Note that the value on the stack can be anything. If it is not
		a String then an exception must be thrown.

		x.(y) = z

		| z  <-- SP
		| y
		| x               z  <-- SP
		| ...             ...
	*/

p_(SETATTRX, PC_A)
	/*
		This is the same as SETATTR except that the attribute name is
		the argument of this instruction and is always a String.

		x.y = z

		| z  <-- SP
		| x               z  <-- SP
		| ...             ...
	*/

p_(SETATTRSELF, PC_A)
	/*
		This is the same as SETATTRX except that the object is 'self'.

		self.x = y

		| y  <-- SP       y  <-- SP
		| ...             ...
	*/

p_(SETGLOBAL , PC_A)
p_(SETGLOBALX, PC_A)
	/*
		Set a module/class level variable value with the value on top of the
		stack. The argument is the Var object.

		This instruction is generated by an assignment.
		Since assignments are expressions then the value must be left
		on the stack.

		x = y

		| y  <-- SP      y  <-- SP
		| ...             ...

		SETGLOBAL  is used for untyped variables.
		SETGLOBALX means the variable is typed.
	*/

p_(SETITEM, PC_0)
	/*
		Set item of the top values on the stack.
		This translates into Sys_setitem (x, y, z).

		x [y] = z

		| z  <-- SP
		| y
		| x               z  <-- SP
		| ...             ...
	*/

p_(SETITEMX, PC_A)
	/*
		This is the same as SETITEM except that the index argument is the
		argument of this instruction.

		x [y] = z

		| z  <-- SP
		| x               z  <-- SP
		| ...             ...
	*/

p_(SETITEML, PC_L)
	/*
		This is the same as SETITEM except that the argument is a lexical
		variable index.

		x [y] = z

		| z  <-- SP
		| y               z  <-- SP
		| ...             ...
	*/

p_(SETITEMLX, PC_LA)
	/*
		This is the same as SETITEML except that the index is a literal.

		x [y] = 1

		| y  <-- SP       1  <-- SP
		| ...             ...
	*/

p_(SETLEXICAL, PC_L)
p_(SETLEXICALX, PC_LA)
p_(SETLEXICALLL, PC_LL)
p_(SETLEXICALLX, PC_LA)
	/*
		Set a lexical variable value with the value on top of the stack.
		The argument is the index (location) of the value on the local
		variable stack.

		This instruction is generated by an assignment.
		Since assignments are expressions then the value must be left
		on the stack.

		x = y

		| y  <-- SP       y  <-- SP
		| ...             ...

		SETLEXICALX does the same but the second argument is a validator
		(type) of the variable.

		SETLEXICALLL does the same but the second argument is the index
		of another lexical.

		SETLEXICALLX does the same but the second argument is a literal.
	*/

p_(SETSLICE, PC_0)
	/*
		Set slice of the top values on the stack.
		This translates into Sys_setslice (x, y, z, v).

		x [y, z] = v

		| v  <-- SP
		| z
		| y
		| x               v  <-- SP
		| ...             ...
	*/

p_(SUPER, PC_A)
	/*
		Get superclass attribute of two top values on the stack.
		The argument is the class object.
		It translates into ObjGetSuperAttr (self, class, name) where
		self is the 'self' value of the Frame.
		Note that the value on the stack can be anything. If it is not
		a String then an exception must be thrown.
		This instruction is always followed by AUTOATTR or CALLATTR.

		super.(x)

		|                 attr <-- SP
		| x    <-- SP     self
		| ...             ...
	*/

p_(SUPERX, PC_AA)
	/*
		This is the same as SUPER except that the name argument is the
		second argument of this instruction and is always a String.

		super.x

		|                 attr <-- SP
		|                 self
		| ...  <-- SP     ...
	*/

p_(SWAP, PC_0)
	/*
		Switch execution to another frame.
		This instruction is generated by the __swap__ keyword.
		It does not affect the stack.

		__swap__ x

		| x  <-- SP       x    <-- SP
		| ...             ...
	*/

p_(THROW, PC_L)
	/*
		Throw exception.
		The argument is the number of parameters on the stack: 0, 1 or 2.
		0 means to rethrow the last exception, otherwise the first argument
		is the exception class (guaranteed by the compiler to be an exception
		class), and the second is the message.
		This instruction resets the stack.

		throw x, y

		| y  <-- SP
		| x
		| ...
		| ...
		| ...  <-- BP     ... <-- BP = SP
		| ...
	*/

p_(TRUE, PC_0)
	/*
		Convert the value on top of the stack into a Boolean.
		This translates into Sys_true (x).
		It never fails and always return true or false.

		| x  <-- SP         true/false  <-- SP
		| ...               ...
	*/
#if 0
p_(UNSAFE, PC_0)
	/*
		If the caller is an unsafe source then throw ESecurity.
		This instruction does not affect the stack.

		UNSAFE is emitted if the Sub has the __unsafe tag.
	*/
#endif
p_(XCAT, PC_L)
	/*
		Perform concatenations of items on the stack.
		The argument is the number of items on the stack.
		This instruction is generated by the multiple _ operator.
		It translates into iSys_xcat (x, y. ...).

		x _ y _ z

		| z  <-- SP
		| y
		| x               result  <-- SP
		| ...             ...
	*/

p_(XCHG, PC_0)
	/*
		Exchange the top 2 values on the stack.

		| a  <-- SP         b  <-- SP
		| b                 a
		| ...               ...
	*/

p_(XCHG3, PC_0)
	/*
		Special rotation for the <-> operator.

		| f <-- SP        c  <-- SP
		| e               e
		| d               d
		| c               f
		| b               b
		| a               a
		| ...             ...
	*/

p_(XCHG4, PC_0)
	/*
		Special rotation for the <-> operator.

		| g <-- SP        c  <-- SP
		| f               f
		| e               e
		| d               d
		| c               g
		| b               b
		| a               a
		| ...             ...
	*/

p_(YIELD, PC_0)
	/*
		Yield the value on top of the stack.
		This instruction is generated by the keyword 'yield'.
		It does not affect the stack.

		yield x

		| x  <-- SP         x  <-- SP
		| ...               ...
	*/

#ifdef CODE_IN_HEADER
/* !!! update this value as necessary */
#define PC_LAST_INSTRUCTION PC_YIELD
#endif

#if PC_LAST_INSTRUCTION > 254
#error somebody messed things up
#endif
