/* Complex.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Complex_
#define _H_Complex_ 1


/** Complex
	This is the final Complex class.
*/
extern Class ComplexClass;


/** TComplex
	This represents an instance of the Complex class.

	Both the real and imaginary parts may be any ={Real} objects including
	={InfPositive} or ={InfNegative}.
*/
struct TComplex
{
	OBJ_HEAD;
	Real real;   /* real part */
	Real imag;   /* imaginary part */
};


/** ComplexPredefined
	Predefined system Complex objects.
*/
/* the real and imaginary parts are Int objects */
extern Complex Complex00;       /*  0 + 0j */
extern Complex Complex01;       /*  0 + 1j */
extern Complex Complex0_1;      /*  0 - 1j */
extern Complex Complex10;       /*  1 + 0j */
extern Complex Complex11;       /*  1 + 1j */
extern Complex Complex1_1;      /*  1 - 1j */
extern Complex Complex_10;      /* -1 + 0j */
extern Complex Complex_11;      /* -1 + 1j */
extern Complex Complex_1_1;     /* -1 - 1j */

/* same as above but the real and imaginary parts are Big objects */
extern Complex ComplexBig00;
extern Complex ComplexBig01;
extern Complex ComplexBig0_1;
extern Complex ComplexBig10;
extern Complex ComplexBig11;
extern Complex ComplexBig1_1;
extern Complex ComplexBig_10;
extern Complex ComplexBig_11;
extern Complex ComplexBig_1_1;

extern Complex ComplexInfp;    /* infp + 0j */
extern Complex ComplexInfn;    /* infn + 0j */


/** ComplexStatic
	Create a new system Complex object, i.e. not in GC.
*/
extern Complex ComplexToStatic (Complex x);


/** __Complex__
	If the ={NumberContext} specifies REALIFY_COMPLEX and the imaginary part is
	zero then this function returns the real part.

	Otherwise if the context does not allow Complex numbers to be returned then
	this function throws EComplex, return a Complex otherwise.

	@warn
	You must always use this to return a Complex result.
*/
extern Number __Complex__ (Real r, Real i);


/** ComplexNew
	Create a new Complex object from various C types and other objects.

	All always succeed except when the argument is an infinity and the
	={NumberContext} does not allow infinities, in which case EInf is thrown.
	If you don't want EInf then use ComplexNew.
*/
extern Complex ComplexNew  (Real real, Real imag);
extern Complex ComplexNewr (Real real, Real imag);
extern Complex ComplexNew1 (Real real);

#define        ComplexNewh(v)   ComplexNew1 ((Real) IntNewh (v))
#define        ComplexNewuh(v)  ComplexNew1 ((Real) IntNewuh (v))
#define        ComplexNewi(v)   ComplexNew1 ((Real) IntegerNewi (v))
#define        ComplexNewui(v)  ComplexNew1 ((Real) IntegerNewui (v))
#define        ComplexNewl(v)   ComplexNew1 ((Real) IntegerNewl (v))
#define        ComplexNewul(v)  ComplexNew1 ((Real) IntegerNewul (v))
extern Complex ComplexNewli     (long v);
#define        ComplexNewd(v)   ComplexNew1 ((Real) FloatNewd (v))
#ifdef HAVE_LONG_LONG
#define        ComplexNewll(v)  ComplexNew1 ((Real) IntegerNewll (v))
#define        ComplexNewull(v) ComplexNew1 ((Real) IntegerNewull (v))
#endif
#ifdef HAVE_LONG_DOUBLE
#define        ComplexNewld(v)  ComplexNew1 ((Real) DecimalNewld (v))
#endif

#define        ComplexNewInt(x)      ComplexNew1 ((Real) x)
#define        ComplexNewLong(x)     ComplexNew1 ((Real) x)
#define        ComplexNewRational(x) ComplexNew1 ((Real) x)
#define        ComplexNewFloat(x)    ComplexNew1 ((Real) x)
#define        ComplexNewBig(x)      ComplexNew1 ((Real) x)
#define        ComplexNewComplex(x)  (x)
#define        ComplexNewInteger(x)  ComplexNew1 ((Real) x)
#define        ComplexNewDecimal(x)  ComplexNew1 ((Real) x)
#define        ComplexNewReal(x)     ComplexNew1 (x)
extern Complex ComplexNewNumber      (Number x);
#define        ComplexNewBoolean(x)  (((x) == True) ? Complex10 : Complex00);
extern Complex ComplexNewString      (String x);


/** ComplexTo
	Convert a Complex to another.

	For C type converions, errno is set on failure.
	For object conversions, NULL is returned on failure.
*/
#define ComplexToh(x)        RealToh   ((x)->real)
#define ComplexTouh(x)       RealTouh  ((x)->real)
#define ComplexToi(x)        RealToi   ((x)->real)
#define ComplexToui(x)       RealToui  ((x)->real)
#define ComplexToli(x)       RealToli  ((x)->real)
#define ComplexTol(x)        RealTol   ((x)->real)
#define ComplexToul(x)       RealToul  ((x)->real)
#define ComplexTod(x)        RealTod   ((x)->real)
#ifdef HAVE_LONG_LONG
#define ComplexToll(x)       RealToll  ((x)->real)
#define ComplexToull(x)      RealToull ((x)->real)
#endif
#ifdef HAVE_LONG_DOUBLE
#define ComplexTold(x)       RealTold  ((x)->real)
#endif

#define       ComplexToInt(x)      RealToInt      ((x)->real)
#define       ComplexToLong(x)     RealToLong     ((x)->real)
#define       ComplexToRational(x) RealToRational ((x)->real)
#define       ComplexToFloat(x)    RealToFloat    ((x)->real)
#define       ComplexToBig(x)      RealToBig      ((x)->real)
#define       ComplexToComplex(x)  (x)
#define       ComplexToInteger(x)  RealToInteger ((x)->real)
#define       ComplexToDecimal(x)  RealToDecimal ((x)->real)
#define       ComplexToReal(x)     ((x)->real)
#define       ComplexToNumber(x)   ((Number) x)
#define       ComplexToBoolean(x)  (ComplexIsZero (x) ? False : True)
extern String ComplexToString      (Complex x);


/** ComplexSign
	ComplexSign return -1, 0, 1 if the real part is negative, 0, positive
	respectively. The imaginary part is ignored.
*/
#define ComplexSign(x)  RealSign ((x)->real)


/** ComplexIs
	Identity functions.

	ComplexIs tests whether or not an object is a Complex.

	ComplexIsNeg return 1 if the real part is negative, and 0 otherwise.
	The imaginary part is ignored.

	ComplexIsZero return 1 if both the real and imaginary parts are 0.
	Return 0 oherwise.

	ComplexIsOne return 1 if the real part is 1 or -1.
	Return 0 otherwise.
	The imaginary part is ignored.

	ComplexIsWhole return 1 if the real part is a whole number.
	Return 0 otherwise.
	The imaginary part is ignored.

	ComplexIsFrac return 1 if if 0 < |x.real| < 1.
	Returns 0 otherwise.
	The imaginary part is ignored.

	ComplexIsOdd return 1 if the real part is a whole and odd number.
	Return 0 otherwise.
	The imaginary part is ignored.

	ComplexIsReal return 1 if the imaginary part is 0.
	Return 0 otherwise.
*/
#define    ComplexIs(x)      (ClassOf (x) == ComplexClass)
extern int ComplexIsZero     (Complex x);
#define    ComplexIsNeg(x)   RealIsNeg   ((x)->real)
#define    ComplexIsOne(x)   RealIsOne   ((x)->real)
#define    ComplexIs1(x)     RealIs1     ((x)->real)
#define    ComplexIsWhole(x) RealIsWhole ((x)->real)
#define    ComplexIsOdd(x)   RealIsOdd   ((x)->real)
#define    ComplexIsFrac(x)  RealIsFrac  ((x)->real)
#define    ComplexIsReal(x)  RealIsZero  ((x)->imag)


/** ComplexUnary
	Unary functions.

	ComplexPos returns the number itself.

	ComplexNeg returns a Complex with both the real and imaginary parts negated.

	ComplexIncr returns a Complex with both the real and imaginary parts incremented by 1.

	ComplexDecr returns a Complex with both the real and imaginary parts decremented by 1.
*/
#define        ComplexPos(x) (x)
extern Complex ComplexNeg    (Complex self);
extern Complex ComplexIncr   (Complex self);
extern Complex ComplexDecr   (Complex self);
extern Real    ComplexAbs    (Complex self);


/** ComplexEq
	Whether the value of a Complex is the same as the value of another Number object.
	All return 0 or 1.

	For Real objects then 1 is returned if the real part is the same as the
	object and the imaginary part is 0.

	For Complex objects, 1 is returned if both real and imaginary parts are the same.
*/
extern int ComplexEqInt      (Complex x, Int y);
extern int ComplexEqLong     (Complex x, Long y);
extern int ComplexEqRational (Complex x, Rational y);
extern int ComplexEqFloat    (Complex x, Float y);
extern int ComplexEqBig      (Complex x, Big y);
extern int ComplexEqComplex  (Complex x, Complex y);
extern int ComplexEqInf      (Complex x, Inf y);
extern int ComplexEqInteger  (Complex x, Integer y);
extern int ComplexEqDecimal  (Complex x, Decimal y);
extern int ComplexEqReal     (Complex x, Real y);
extern int ComplexEqNumber   (Complex x, Number y);


/** ComplexLike
	Whether the absolute value of a Complex is the same as the absolute value of another
	Number object.
	All return 0 or 1.

	For Real objects then 1 is returned if the absolute value of the real part
	is the same as the absolute value of the object and the imaginary part is 0.

	For Complex objects, 1 is returned if both real and imaginary parts have the same
	absolute values.
*/
extern int ComplexLikeInt      (Complex x, Int y);
extern int ComplexLikeLong     (Complex x, Long y);
extern int ComplexLikeRational (Complex x, Rational y);
extern int ComplexLikeFloat    (Complex x, Float y);
extern int ComplexLikeBig      (Complex x, Big y);
extern int ComplexLikeComplex  (Complex x, Complex y);
extern int ComplexLikeInf      (Complex x, Inf y);
extern int ComplexLikeInteger  (Complex x, Integer y);
extern int ComplexLikeDecimal  (Complex x, Decimal y);
extern int ComplexLikeReal     (Complex x, Real y);
extern int ComplexLikeNumber   (Complex x, Number y);


/** ComplexCmp
	Compare an Integer with Complex Decimal object.
	All return -1, 0 or 1 in the usual manner.

	For Real objects then the value is compared to the real part. If it is the same
	then the sign of the imaginary part is returned.

	For Complex objects, both real and imaginary parts are compared.
*/
extern int ComplexCmpInt      (Complex x, Int y);
extern int ComplexCmpLong     (Complex x, Long y);
extern int ComplexCmpRational (Complex x, Rational y);
extern int ComplexCmpFloat    (Complex x, Float y);
extern int ComplexCmpBig      (Complex x, Big y);
extern int ComplexCmpComplex  (Complex x, Complex y);
extern int ComplexCmpInf      (Complex x, Inf y);
extern int ComplexCmpInteger  (Complex x, Integer y);
extern int ComplexCmpDecimal  (Complex x, Decimal y);
extern int ComplexCmpReal     (Complex x, Real y);
extern int ComplexCmpNumber   (Complex x, Number y);


/** ComplexCmpAbs
	Compare the absolute value of a Complex with the absolute value of another
	Number object.
	All return -1, 0, or -1 in the usual manner.

	The absolute value of the Complex number is used to do the comparisson, which
	formula is C{sqrt (z.real * z.real + z.imag * z.imag)}.
*/
extern int ComplexCmpAbsInt      (Complex x, Int y);
extern int ComplexCmpAbsLong     (Complex x, Long y);
extern int ComplexCmpAbsRational (Complex x, Rational y);
extern int ComplexCmpAbsFloat    (Complex x, Float y);
extern int ComplexCmpAbsBig      (Complex x, Big y);
extern int ComplexCmpAbsComplex  (Complex x, Complex y);
extern int ComplexCmpAbsInf      (Complex x, Inf y);
extern int ComplexCmpAbsInteger  (Complex x, Integer y);
extern int ComplexCmpAbsDecimal  (Complex x, Decimal y);
extern int ComplexCmpAbsReal     (Complex x, Real y);
extern int ComplexCmpAbsNumber   (Complex x, Number y);


/** ComplexPlus
	Add a Complex with another.
*/
extern Number ComplexPlusInt      (Complex x, Int y);
extern Number ComplexPlusLong     (Complex x, Long y);
extern Number ComplexPlusRational (Complex x, Rational y);
extern Number ComplexPlusFloat    (Complex x, Float y);
extern Number ComplexPlusBig      (Complex x, Big y);
extern Number ComplexPlusComplex  (Complex x, Complex y);
extern Number ComplexPlusInf      (Complex x, Inf y);
extern Number ComplexPlusInteger  (Complex x, Integer y);
extern Number ComplexPlusDecimal  (Complex x, Decimal y);
extern Number ComplexPlusReal     (Complex x, Real y);
extern Number ComplexPlusNumber   (Complex x, Number y);


/** ComplexMinus
	Subtract a Complex with another.
*/
extern Number ComplexMinusInt      (Complex x, Int y);
extern Number ComplexMinusLong     (Complex x, Long y);
extern Number ComplexMinusRational (Complex x, Rational y);
extern Number ComplexMinusFloat    (Complex x, Float y);
extern Number ComplexMinusBig      (Complex x, Big y);
extern Number ComplexMinusComplex  (Complex x, Complex y);
extern Number ComplexMinusInf      (Complex x, Inf y);
extern Number ComplexMinusInteger  (Complex x, Integer y);
extern Number ComplexMinusDecimal  (Complex x, Decimal y);
extern Number ComplexMinusReal     (Complex x, Real y);
extern Number ComplexMinusNumber   (Complex x, Number y);


/** ComplexRminus
	Subtract another with a Complex.
	This is only for efficiency.
*/
extern Number ComplexRminusInt      (Complex x, Int y);
extern Number ComplexRminusLong     (Complex x, Long y);
extern Number ComplexRminusRational (Complex x, Rational y);
extern Number ComplexRminusFloat    (Complex x, Float y);
extern Number ComplexRminusBig      (Complex x, Big y);


/** ComplexMul
	Multiply a Complex with another.
*/
extern Number ComplexMulInt      (Complex x, Int y);
extern Number ComplexMulLong     (Complex x, Long y);
extern Number ComplexMulRational (Complex x, Rational y);
extern Number ComplexMulFloat    (Complex x, Float y);
extern Number ComplexMulBig      (Complex x, Big y);
extern Number ComplexMulComplex  (Complex x, Complex y);
extern Number ComplexMulInf      (Complex x, Inf y);
extern Number ComplexMulInteger  (Complex x, Integer y);
extern Number ComplexMulDecimal  (Complex x, Decimal y);
extern Number ComplexMulReal     (Complex x, Real y);
extern Number ComplexMulNumber   (Complex x, Number y);


/** ComplexDiv
	Divide a Complex with another.
*/
extern Number ComplexDivInt      (Complex x, Int y);
extern Number ComplexDivLong     (Complex x, Long y);
extern Number ComplexDivRational (Complex x, Rational y);
extern Number ComplexDivFloat    (Complex x, Float y);
extern Number ComplexDivBig      (Complex x, Big y);
extern Number ComplexDivComplex  (Complex x, Complex y);
extern Number ComplexDivInf      (Complex x, Inf y);
extern Number ComplexDivInteger  (Complex x, Integer y);
extern Number ComplexDivDecimal  (Complex x, Decimal y);
extern Number ComplexDivReal     (Complex x, Real y);
extern Number ComplexDivNumber   (Complex x, Number y);


/** ComplexRdiv
	Divide another with a Complex.
	This is only for efficiency.
*/
extern Number ComplexRdivInt      (Complex x, Int y);
extern Number ComplexRdivLong     (Complex x, Long y);
extern Number ComplexRdivRational (Complex x, Rational y);
extern Number ComplexRdivFloat    (Complex x, Float y);
extern Number ComplexRdivBig      (Complex x, Big y);
extern Number ComplexRdivComplex  (Complex x, Complex y);


/** ComplexIdiv
	Integer division of a Complex with another.
*/
extern Integer ComplexIdivInt      (Complex x, Int y);
extern Integer ComplexIdivLong     (Complex x, Long y);
extern Integer ComplexIdivRational (Complex x, Rational y);
extern Integer ComplexIdivFloat    (Complex x, Float y);
extern Integer ComplexIdivBig      (Complex x, Big y);
extern Integer ComplexIdivComplex  (Complex x, Complex y);
extern Integer ComplexIdivInf      (Complex x, Inf y);
extern Integer ComplexIdivInteger  (Complex x, Integer y);
extern Integer ComplexIdivDecimal  (Complex x, Decimal y);
extern Integer ComplexIdivReal     (Complex x, Real y);
extern Integer ComplexIdivNumber   (Complex x, Number y);


/** ComplexRidiv
	Integer division of another with a Complex.
	This is only for efficiency.
*/
extern Integer ComplexRidivInt      (Complex x, Int y);
extern Integer ComplexRidivLong     (Complex x, Long y);
extern Integer ComplexRidivRational (Complex x, Rational y);
extern Integer ComplexRidivFloat    (Complex x, Float y);
extern Integer ComplexRidivBig      (Complex x, Big y);


/** ComplexRem
	Remainder of dividing a Complex with another.
*/
extern Number ComplexRemInt      (Complex x, Int y);
extern Number ComplexRemLong     (Complex x, Long y);
extern Number ComplexRemRational (Complex x, Rational y);
extern Number ComplexRemFloat    (Complex x, Float y);
extern Number ComplexRemBig      (Complex x, Big y);
extern Number ComplexRemComplex  (Complex x, Complex y);
extern Number ComplexRemInf      (Complex x, Inf y);
extern Number ComplexRemInteger  (Complex x, Integer y);
extern Number ComplexRemDecimal  (Complex x, Decimal y);
extern Number ComplexRemReal     (Complex x, Real y);
extern Number ComplexRemNumber   (Complex x, Number y);


/** ComplexRrem
	Remainder of dividing another with a Complex.
	This is only for efficiency.
*/
extern Number ComplexRremInt      (Complex x, Int y);
extern Number ComplexRremLong     (Complex x, Long y);
extern Number ComplexRremRational (Complex x, Rational y);
extern Number ComplexRremFloat    (Complex x, Float y);
extern Number ComplexRremBig      (Complex x, Big y);


/** ComplexPow
	Raise a Complex to the power of another.
*/
extern Number ComplexPowInt      (Complex x, Int y);
extern Number ComplexPowLong     (Complex x, Long y);
extern Number ComplexPowRational (Complex x, Rational y);
extern Number ComplexPowFloat    (Complex x, Float y);
extern Number ComplexPowBig      (Complex x, Big y);
extern Number ComplexPowComplex  (Complex x, Complex y);
extern Number ComplexPowInf      (Complex x, Inf y);
extern Number ComplexPowInteger  (Complex x, Integer y);
extern Number ComplexPowDecimal  (Complex x, Decimal y);
extern Number ComplexPowReal     (Complex x, Real y);
extern Number ComplexPowNumber   (Complex x, Number y);


/** ComplexRpow
	Raise another to the power of a Complex.
	This is only for efficiency.
*/
extern Number ComplexRpowInt      (Complex x, Int y);
extern Number ComplexRpowLong     (Complex x, Long y);
extern Number ComplexRpowRational (Complex x, Rational y);
extern Number ComplexRpowFloat    (Complex x, Float y);
extern Number ComplexRpowBig      (Complex x, Big y);


/** ComplexRound
	Rounding functions.

	ComplexCeil returns the result of rounding the real and imaginary parts to the
	nearest integer towards positive infinity.

	ComplexFloor returns the result of rounding the real and imaginary parts to the
	nearest integer towards negative infinity.

	ComplexTrunc returns the result of rounding the real and imaginary parts to the
	nearest integer towards zero, that is, it removes the fraction part.
*/
extern Number ComplexFloor (Complex x);
extern Number ComplexCeil  (Complex x);
extern Number ComplexTrunc (Complex x);
extern Number ComplexFrac  (Complex x);
extern Number ComplexRound (Complex x, int prec);


/** ComplexPrint
	Print a Complex to v{stdout}.
*/
extern Null ComplexPrint (Complex self);


/** ComplexFormat
	Format a Complex.

	See ={Format} for details;
*/
extern String ComplexFormat (Complex x, format_s *f);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Complex class */
extern void ComplexSetup ();
extern void ComplexInit  ();

/* this needs to be filled in by the Math module */
typedef Number (*complexpowbig_f) (Real ar, Real ai, Real br, Real bi);
extern complexpowbig_f ComplexPowComplex_math;

typedef Real (*complexabs_f) (Complex x);
extern complexabs_f ComplexAbs_math;

typedef Real (*complexarg_f) (Complex x);
extern complexarg_f ComplexArg_math;

/* for Sys.c */
extern Number ComplexReturn (Any r, Any i);


#endif /*_H_Complex_*/
