/* Container.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Container_
#define _H_Container_ 1


/** Container
	This is the parent class of ={Array}, ={Set}, ={Dict}, ={Queue} and ={Record}.
*/
extern Class ContainerClass;


/** TContainer CONTAINER_HEAD
	Container instances must use CONTAINER_HEAD instead of ={OBJ_HEAD}.

	See also ={InterpScope}.

	If defined, CONTAINER_WITH_PROTECT allows for program level protection.
	It takes an extra pointer overhead but it is a nice feature.
*/
#define CONTAINER_WITH_PROTECT

#ifdef CONTAINER_WITH_PROTECT

#define CONTAINER_HEAD \
	OBJ_HEAD; \
	Any   scope;  /* protector scope - see InterpScope */ \
	long  len     /* number of items <= GC_MAX_ITEMS */

#else

#define CONTAINER_HEAD \
	OBJ_HEAD; \
	long  len     /* number of items <= GC_MAX_ITEMS */

#endif

struct TContainer
{
	CONTAINER_HEAD;
};


/** ContainerIs
	Whether or not an object is a Container.
*/
#define ContainerIs(o) (ObjInstOf(o) == ContainerClass)


/** ContainerFlag
	The following flags are used by Container objects.

	Do not access these flags directly.
	Use the accessor macros provided instead.
	If none is defined then it means you must not touch the flag.
*/
#define ContainerFlagPROTECTED Flag0   /* protected by program level */
#define ContainerFlagFROZEN    Flag1   /* immutable? */
#define ContainerFlagFIXED     Flag2   /* resizable? */


/** ContainerIsFrozen ContainerFreeze ContainerUnfreeze ContainerIsFixed
	ContainerIsFrozen checks whether or not the Container is immutable.

	ContainerFreeze must only be used after you have created a new Container and
	don't want the Container to be modified by others.

	ContainerUnfreeze must only be used on an Container you have created.
	Normally, to modify the Container then freeze it again.

	ContainerIsFixed checks whether or not the Container is resizable.
*/
#define ContainerIsFrozen(self)  FlagTest  (self, ContainerFlagFROZEN | ContainerFlagPROTECTED)
#define ContainerFreeze(self)    FlagSet   (self, ContainerFlagFROZEN)
#define ContainerUnfreeze(self)  FlagUnset (self, ContainerFlagFROZEN)
#define ContainerIsFixed(self)   FlagTest  (self, ContainerFlagFIXED)
#define ContainerFix(self)       FlagSet   (self, ContainerFlagFIXED)
#define ContainerUnFix(self)     FlagUnset (self, ContainerFlagFIXED)


/** __ContainerProtect
	If you are not the creator of the Container then use these macros to
	temporarily make the Container immutable.

	The correct idiom is:
	@code
		...
		...
		__ContainerProtectBegin (self)
			...
			result = ... do not THROW here -----------+
			...                                       |
		__ContainerProtectEnd;                        |
		...                     THROW here instead <--+
		...
*/
#ifdef CONTAINER_WITH_PROTECT

#define __ContainerProtectBegin(self) \
BEGIN \
	Container tHiS = (Container) self; \
	Any       pRoT = tHiS->scope; \
	tHiS->scope = SysModule; \
	FlagSet (tHiS, ContainerFlagPROTECTED)

#define __ContainerProtectEnd \
	tHiS->scope = pRoT; \
	if (!tHiS->scope) \
		FlagUnset (tHiS, ContainerFlagPROTECTED); \
END

#else /*CONTAINER_WITH_PROTECT*/

#define __ContainerProtectBegin(self) \
BEGIN \
	Container tHiS = (Container) self; \
	assert (!ContainerIsFrozen (tHiS)); \
	ContainerFreeze (tHiS)

#define __ContainerProtectEnd \
	ContainerUnfreeze (tHiS); \
END

#endif /*CONTAINER_WITH_PROTECT*/


/** CONTAINER_INIT CONTAINER_MARK
	Container classes must call CONTAINER_INIT from their allocation routines
	and CONTAINER_MARK this from within their f{mark} method.
*/
#ifdef CONTAINER_WITH_PROTECT
#define CONTAINER_INIT(self)  (self)->scope = 0
#define CONTAINER_MARK(self)  GcXmark ((self)->scope)
#else
#define CONTAINER_INIT(self)  BEGIN END
#define CONTAINER_MARK(self)  BEGIN END
#endif


/** CONTAINER_ENSURE
	Throw EModify if the Container is frozen, fixed or both.
*/
#define CONTAINER_ENSURE_NOT_FROZEN(self) \
	REQUIRE (EModify, !ContainerIsFrozen (self))

#define CONTAINER_ENSURE_NOT_FIXED(self) \
	REQUIRE (EModify, !ContainerIsFixed (self))

#define CONTAINER_ENSURE_NOT_FROZEN_FIXED(self) \
	REQUIRE (EModify, !ContainerIsFrozen (self) && !ContainerIsFixed (self))


/** CONTAINER_ENSURE_INDEX
	Translate negative index and ensure it is within a specified length.
*/
#define CONTAINER_ENSURE_INDEX(idx, len) \
BEGIN \
	if ((idx) < 0) \
		idx += len; \
	REQUIRE (EIndex, 0 <= (idx) && (idx) < (len)); \
END


/** CONTAINER_QUOTE
	If it is a String then quote it.
	Otherwise make it a String.
*/
#define CONTAINER_QUOTE(x) (StringIs (x) ? StringQuote ((String) x) : Sys_string (x))


/** CONTAINER_PRINT_QUOTED
	If it is a String then print it quoted.
	Otherwise print as is.
*/
#define CONTAINER_PRINT_QUOTED(x) \
	(StringIs (x) ? StringPrint (x, '\'') : SysPrint (x))


/** CONTAINER_TRUE
	The default truth value of a container.
*/
#define CONTAINER_TRUE(self) ((self)->len != 0)


/** ContainerSizer
	Get a nice size.
*/
extern long ContainerSizer (long cap);

/* not smaller than this */
#define CONTAINER_SIZE_MIN 8


/** ContainerMask
	Various functions for Dict/Set like objects.

	ContainerMaskSizer get a nice mask value.

	ContainerMaskGrow returns 0 if growing is not required, the new mask value
	otherwise.

	ContainerMaskShrink returns 0 if shrinking is not required, the new mask value
	otherwise.
*/
extern long ContainerMaskSizer (long cap);

/* small mask must be n^2 - 1 */
#define CONTAINER_MASK_MIN 15

/* limit mask must be n^2 - 1 */
#define CONTAINER_MASK_MAX 0X3FFFFFFF

inline static long ContainerMaskGrow (long mask, long len)
{
	if (mask < CONTAINER_MASK_MAX && len > (mask << 1) / 3)
		return ((mask + 1) << 1) - 1;
	return 0;
}

inline static long ContainerMaskShrink (long mask, long len)
{
	if (mask > CONTAINER_MASK_MIN && len < (mask / 3))
		return ((mask + 1) >> 1) - 1;
	return 0;
}


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Container class. */
extern void ContainerSetup ();
extern void ContainerInit  ();


#endif /*_H_Container_*/
