/* Decimal.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Decimal_
#define _H_Decimal_ 1


/** Decimal
	This is the abstract parent class of ={Float} and ={Big}.
*/
extern Class DecimalClass;


/** DecimalPredefined
	Predefined system objects.
*/
#define Decimal0  ((Decimal) Float0)   /*  0.0 */
#define Decimal1  ((Decimal) Float1)   /*  1.0 */
#define Decimal2  ((Decimal) Float2)   /*  2.0 */
#define Decimal_1 ((Decimal) Float_1)  /* -1.0 */
#define Decimal_2 ((Decimal) Float_2)  /* -2.0 */


/** DecimalNew
	Create a new Decimal from other objects.

	For C type conversions, all functions always succeed and the returned
	value is either a ={Float}. Only DecimalNewld returns a ={Big}.

	For object conversions, an exception is thrown and NULL is returned on failure.
	On success either a ={Float} or ={Big} is returned.
*/
#define DecimalNewh(v)   DecimalNewd (v)
#define DecimalNewuh(v)  DecimalNewd (v)
#define DecimalNewi(v)   DecimalNewd (v)
#define DecimalNewui(v)  DecimalNewd (v)
#define DecimalNewli(v)  DecimalNewd (v)
#define DecimalNewl(v)   DecimalNewd (v)
#define DecimalNewul(v)  DecimalNewd (v)
#define DecimalNewd(v)   ((Decimal) (FloatNewd (v)))
#if HAVE_LONG_LONG
#define DecimalNewll(v)  DecimalNewd (v)
#define DecimalNewull(v) DecimalNewd (v)
#endif
#ifdef HAVE_LONG_DOUBLE
extern Decimal DecimalNewld  (long double v);
#endif

#define DecimalNewInt(x)     IntToDecimal (x)
#define DecimalNewLong       LongToDecimal (x)
#define DecimalNewRational   RationalToDecimal (x)
#define DecimalNewFloat(x)   FloatToDecimal (x)
#define DecimalNewBig(x)     BigToDecimal (x)
#define DecimalNewComplex(x) RealNewComplex ((x)->real)
#define DecimalNewInteger(x) IntegerToDecimal (x)
#define DecimalNewDecimal(x) ((Decimal) x)
#define DecimalNewReal(x)    RealToDecimal (x)
#define DecimalNewNumber(x)  NumberToDecimal (x)


/** DecimalNewdx
	Create a new Decimal/Real object from a C double by checking whether or not the
	value is NaN or infinity.

	If the value is NaN then EValue is thrown and NULL is returned.

	If the value is infinity then EInf is thrown and NULL is returned unless the
	={NumberContext} allows infinities to be returned, in which case one of
	={InfPositive} or ={InfNegative} is returned based on the sign.
*/
extern Real DecimalNewdx (double v);
#ifdef HAVE_LONG_DOUBLE
extern Real DecimalNewldx (long double v);
#endif


/** DecimalTo
	Convert a Decimal into a C type or another object.

	All of C conversions accept any object. If the object is not a Decimal then
	errno is set to EDOM and 0 or NULL is returned. Otherwise errno is set to
	ERANGE if the value does not fit into the specified type or object. Otherwise
	errno is set to zero indicating success.

	For object conversions, NULL is returned on failure.

	These are just convenient wrappers around ={FloatTo} and ={BigTo}.
	If you already know what the object is, then it is more efficient to call them
	instead of these functions.
*/
#define                   DecimalToh(x)   Number_ltoh   (DecimalTol (x))
#define                   DecimalTouh(x)  Number_ultouh (DecimalToul (x))
#define                   DecimalToi(x)   Number_ltoi   (DecimalTol (x))
#define                   DecimalToui(x)  Number_ultoui (DecimalToul (x))
#define                   DecimalToli(x)  Number_ltoli  (DecimalTol (x))
extern long               DecimalTol   (Any x);
extern unsigned long      DecimalToul  (Any x);
extern float              DecimalTof   (Any x);
extern double             DecimalTod   (Any x);
#ifdef HAVE_LONG_LONG
extern long long          DecimalToll  (Any x);
extern unsigned long long DecimalToull (Any x);
#endif
#ifdef HAVE_LONG_DOUBLE
extern long double        DecimalTold  (Any x);
#endif

extern Int      DecimalToInt        (Decimal x);
extern Long     DecimalToLong       (Decimal x);
extern Rational DecimalToRational   (Decimal x);
extern Float    DecimalToFloat      (Decimal x);
extern Big      DecimalToBig        (Decimal x);
#define         DecimalToComplex(x) ComplexNew1 ((Real) (x))
extern Integer  DecimalToInteger    (Decimal x);
#define         DecimalToDecimal(x) (x)
#define         DecimalToReal(x)    ((Real) (x))
#define         DecimalToNumber(x)  ((Number) (x))


/** DecimalSign
	DecimalSign returns -1, 0, 1 if the value is negative, 0, positive respectively.
*/
extern int DecimalSign (Decimal x);


/** DecimalIs
	Identity functions.

	DecimalIs tests whether an object is a Decimal.

	DecimalIsNeg returns 1 if the value is negative.
	Returns 0 otherwise.

	DecimalIsZero returns 1 if the value is 0.
	Returns 0 otherwise.

	DecimalIsOne returns 1 if the value is 1 or -1.
	Returns 0 otherwise.

	DecimalIs1 returns 1 if the value is 1.
	Returns 0 otherwise.

	DecimalIsWhole returns 1 if the value does not have a fraction.
	Returns 0 otherwise.

	DecimalIsFrac returns 1 if 0 < |x| < 1.
	Returns 0 otherwise.

	DecimalIsOdd returns 1 if the value is a whole and odd number.
	Returns 0 otherwise.
*/
#define DecimalIs(x) (FloatIs (x) || BigIs (x))
extern int DecimalIsZero  (Decimal x);
extern int DecimalIsNeg   (Decimal x);
extern int DecimalIsOne   (Decimal x);
extern int DecimalIs1     (Decimal x);
extern int DecimalIsWhole (Decimal x);
extern int DecimalIsFrac  (Decimal x);
extern int DecimalIsOdd   (Decimal x);


/** DecimalUnary
	Unary functions.

	DecimalPos returns the number itself, obviously.

	DecimalNeg returns a negated value.

	DecimalIncr returns the number incremented by 1.

	DecimalDecr returns the number decremented by 1.

	These are just convenient wrappers around
	={FloatPos} ={FloatNeg} ={FloatIncr} ={FloatDecr}
	={BigPos} ={BigNeg} ={BigIncr} ={BigDecr}.
*/
extern Decimal DecimalPos  (Decimal x);
extern Decimal DecimalNeg  (Decimal x);
extern Real    DecimalIncr (Decimal x);
extern Real    DecimalDecr (Decimal x);


/** DecimalEq
	Whether the value of a Decimal is the same as the value of another Number object.
	All return 0 or 1.

	These are just convenient wrappers around ={FloatEq} and ={BigEq}.
*/
extern int DecimalEqInt      (Decimal x, Int y);
extern int DecimalEqLong     (Decimal x, Long y);
extern int DecimalEqRational (Decimal x, Rational y);
extern int DecimalEqFloat    (Decimal x, Float y);
extern int DecimalEqBig      (Decimal x, Big y);
extern int DecimalEqComplex  (Decimal x, Complex y);
extern int DecimalEqInf      (Decimal x, Inf y);
extern int DecimalEqInteger  (Decimal x, Integer y);
extern int DecimalEqDecimal  (Decimal x, Decimal y);
extern int DecimalEqReal     (Decimal x, Real y);
extern int DecimalEqNumber   (Decimal x, Number y);


/** DecimalLike
	Whether the absolute value of a Decimal is the same as the absolute value of another
	Number object.
	All return 0 or 1.

	These are just convenient wrappers around ={FloatLike} and ={BigLike}.
*/
extern int DecimalLikeInt      (Decimal x, Int y);
extern int DecimalLikeLong     (Decimal x, Long y);
extern int DecimalLikeRational (Decimal x, Rational y);
extern int DecimalLikeFloat    (Decimal x, Float y);
extern int DecimalLikeBig      (Decimal x, Big y);
extern int DecimalLikeComplex  (Decimal x, Complex y);
extern int DecimalLikeInf      (Decimal x, Inf y);
extern int DecimalLikeInteger  (Decimal x, Integer y);
extern int DecimalLikeDecimal  (Decimal x, Decimal y);
extern int DecimalLikeReal     (Decimal x, Real y);
extern int DecimalLikeNumber   (Decimal x, Number y);


/** DecimalCmp
	Compare a Number with another Decimal object.
	All return -1, 0, or -1 in the usual manner.

	These are just convenient wrappers around ={FloatCmp} and ={BigCmp}.
*/
extern int DecimalCmpInt      (Decimal x, Int y);
extern int DecimalCmpLong     (Decimal x, Long y);
extern int DecimalCmpRational (Decimal x, Rational y);
extern int DecimalCmpFloat    (Decimal x, Float y);
extern int DecimalCmpBig      (Decimal x, Big y);
extern int DecimalCmpComplex  (Decimal x, Complex y);
extern int DecimalCmpInf      (Decimal x, Inf y);
extern int DecimalCmpInteger  (Decimal x, Integer y);
extern int DecimalCmpDecimal  (Decimal x, Decimal y);
extern int DecimalCmpReal     (Decimal x, Real y);
extern int DecimalCmpNumber   (Decimal x, Number y);


/** DecimalCmpAbs
	Compare the absolute value of a Decimal with the absolute value of another
	Number object.
	All return -1, 0, or -1 in the usual manner.

	These are just convenient wrappers around ={FloatCmpAbs} and ={BigCmpAbs}.
*/
extern int DecimalCmpAbsInt      (Decimal x, Int y);
extern int DecimalCmpAbsLong     (Decimal x, Long y);
extern int DecimalCmpAbsRational (Decimal x, Rational y);
extern int DecimalCmpAbsFloat    (Decimal x, Float y);
extern int DecimalCmpAbsBig      (Decimal x, Big y);
extern int DecimalCmpAbsComplex  (Decimal x, Complex y);
extern int DecimalCmpAbsInf      (Decimal x, Inf y);
extern int DecimalCmpAbsInteger  (Decimal x, Integer y);
extern int DecimalCmpAbsDecimal  (Decimal x, Decimal y);
extern int DecimalCmpAbsReal     (Decimal x, Real y);
extern int DecimalCmpAbsNumber   (Decimal x, Number y);


/** DecimalPlus
	Add a Decimal with another Number object.

	These are just convenient wrappers around ={FloatPlus} and ={BigPlus}.
*/
extern Real    DecimalPlusInt      (Decimal x, Int y);
extern Real    DecimalPlusLong     (Decimal x, Long y);
extern Real    DecimalPlusRational (Decimal x, Rational y);
extern Real    DecimalPlusFloat    (Decimal x, Float y);
extern Real    DecimalPlusBig      (Decimal x, Big y);
extern Complex DecimalPlusComplex  (Decimal x, Complex y);
extern Real    DecimalPlusInf      (Decimal x, Inf y);
extern Real    DecimalPlusInteger  (Decimal x, Integer y);
extern Real    DecimalPlusDecimal  (Decimal x, Decimal y);
extern Real    DecimalPlusReal     (Decimal x, Real y);
extern Number  DecimalPlusNumber   (Decimal x, Number y);


/** DecimalMinus
	Subtract a Decimal with another Number object.

	These are just convenient wrappers around ={FloatMinus} and ={BigMinus}.
*/
extern Real    DecimalMinusInt      (Decimal x, Int y);
extern Real    DecimalMinusLong     (Decimal x, Long y);
extern Real    DecimalMinusRational (Decimal x, Rational y);
extern Real    DecimalMinusFloat    (Decimal x, Float y);
extern Real    DecimalMinusBig      (Decimal x, Big y);
extern Complex DecimalMinusComplex  (Decimal x, Complex y);
extern Real    DecimalMinusInf      (Decimal x, Inf y);
extern Real    DecimalMinusInteger  (Decimal x, Integer y);
extern Real    DecimalMinusDecimal  (Decimal x, Decimal y);
extern Real    DecimalMinusReal     (Decimal x, Real y);
extern Number  DecimalMinusNumber   (Decimal x, Number y);


/** DecimalMul
	Multiply a Decimal with another Number object.

	These are just convenient wrappers around ={FloatMinus} and ={BigMinus}.
*/
extern Real    DecimalMulInt      (Decimal x, Int y);
extern Real    DecimalMulLong     (Decimal x, Long y);
extern Real    DecimalMulRational (Decimal x, Rational y);
extern Real    DecimalMulFloat    (Decimal x, Float y);
extern Real    DecimalMulBig      (Decimal x, Big y);
extern Complex DecimalMulComplex  (Decimal x, Complex y);
extern Real    DecimalMulInf      (Decimal x, Inf y);
extern Real    DecimalMulInteger  (Decimal x, Integer y);
extern Real    DecimalMulDecimal  (Decimal x, Decimal y);
extern Real    DecimalMulReal     (Decimal x, Real y);
extern Number  DecimalMulNumber   (Decimal x, Number y);


/** DecimalDiv
	Divide a Decimal with another Number object.

	These are just convenient wrappers around ={FloatDiv} and ={BigDiv}.
*/
extern Real    DecimalDivInt      (Decimal x, Int y);
extern Real    DecimalDivLong     (Decimal x, Long y);
extern Real    DecimalDivRational (Decimal x, Rational y);
extern Real    DecimalDivFloat    (Decimal x, Float y);
extern Real    DecimalDivBig      (Decimal x, Big y);
extern Complex DecimalDivComplex  (Decimal x, Complex y);
extern Real    DecimalDivInf      (Decimal x, Inf y);
extern Real    DecimalDivInteger  (Decimal x, Integer y);
extern Real    DecimalDivDecimal  (Decimal x, Decimal y);
extern Real    DecimalDivReal     (Decimal x, Real y);
extern Number  DecimalDivNumber   (Decimal x, Number y);


/** DecimalIdiv
	Decimal division of a Decimal with another Number object.

	These are just convenient wrappers around ={FloatIdiv} and ={BigIdiv}.
*/
extern Integer DecimalIdivInt      (Decimal x, Int y);
extern Integer DecimalIdivLong     (Decimal x, Long y);
extern Integer DecimalIdivRational (Decimal x, Rational y);
extern Integer DecimalIdivFloat    (Decimal x, Float y);
extern Integer DecimalIdivBig      (Decimal x, Big y);
extern Integer DecimalIdivComplex  (Decimal x, Complex y);
extern Integer DecimalIdivInf      (Decimal x, Inf y);
extern Integer DecimalIdivInteger  (Decimal x, Integer y);
extern Integer DecimalIdivDecimal  (Decimal x, Decimal y);
extern Integer DecimalIdivReal     (Decimal x, Real y);
extern Integer DecimalIdivNumber   (Decimal x, Number y);


/** DecimalRem
	Remainder of dividing a Decimal with another Number object.

	These are just convenient wrappers around ={FloatRem} and ={BigRem}.
*/
extern Real    DecimalRemInt      (Decimal x, Int y);
extern Real    DecimalRemLong     (Decimal x, Long y);
extern Real    DecimalRemRational (Decimal x, Rational y);
extern Real    DecimalRemFloat    (Decimal x, Float y);
extern Real    DecimalRemBig      (Decimal x, Big y);
extern Complex DecimalRemComplex  (Decimal x, Complex y);
extern Decimal DecimalRemInf      (Decimal x, Inf y);
extern Real    DecimalRemInteger  (Decimal x, Integer y);
extern Real    DecimalRemDecimal  (Decimal x, Decimal y);
extern Real    DecimalRemReal     (Decimal x, Real y);
extern Number  DecimalRemNumber   (Decimal x, Number y);


/** DecimalPow
	Raise a Decimal to the power of another Number object.

	These are just convenient wrappers around ={FloatPow} and ={BigPow}.
*/
extern Real    DecimalPowInt      (Decimal x, Int y);
extern Real    DecimalPowLong     (Decimal x, Long y);
extern Number  DecimalPowRational (Decimal x, Rational y);
extern Number  DecimalPowFloat    (Decimal x, Float y);
extern Number  DecimalPowBig      (Decimal x, Big y);
extern Complex DecimalPowComplex  (Decimal x, Complex y);
extern Real    DecimalPowInf      (Decimal x, Inf y);
extern Number  DecimalPowInteger  (Decimal x, Integer y);
extern Number  DecimalPowDecimal  (Decimal x, Decimal y);
extern Number  DecimalPowReal     (Decimal x, Real y);
extern Number  DecimalPowNumber   (Decimal x, Number y);


/** DecimalPrint
	Print a Decimal to v{stdout}.

	Note that this might throw an exception because writing to v{stdout} may
	fail for various reasons.

	This is just a convenient wrapper around ={FloatPrint} and ={BigPrint}.
*/
extern Null DecimalPrint (Decimal x);


/** DecimalFormat
	Format a Decimal.

	See ={Format} for details.

	This is just a convenient wrapper around ={FloatFormat} and ={BigFormat}.
*/
extern String DecimalFormat (Decimal x, format_s *f);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Float class */
extern void DecimalSetup ();
extern void DecimalInit  ();

#endif /*_H_Decimal_*/
