/* Dict.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Dict_
#define _H_Dict_ 1

#include "Container.h"


/** Dict
	This is the final Dict class.

	@warn
	Arg is a final subclass of Dict.

	Do not use it.
	It is an experimental special class used for passing named arguments to
	functions/methods.

	Currently program level functions/methods do not support named arguments
	for efficiency (otherwise every single Sub will need one more Arg field).
	Thus C functions/method should not support it as well to avoid confusion.

	In the future, if we ever decide to support named arguments then this is
	the class to use.

	Note that the compiler/interpreter already supports this feature (the
	PC_ARG instruction). At the moment it is completely useless.
*/
extern Class DictClass;
extern Class ArgClass;


/** DictIs
	Whether or not an object is a Dict.

	Always use this macro to test whether or not an object is a Dict.
	In the future, if we ever decide to let the Dict be suclassed then the
	macro will change to C{ObjInstOf (o, DictClass)}.

	@warn
	Arg is experimental. Do not use it.
*/
#define DictIs(o) (ClassOf(o) == DictClass)
#define ArgIs(o)  (ClassOf(o) == ArgClass)


/** TDict
	This represents an instance of the Dict class.

	See also ={CONTAINER_HEAD}.

	@warn 
	Never access anything directly except v{len}.
*/
typedef struct dictentry_s dictentry_s;
struct dictentry_s
{
	dictentry_s  *prev;    /* previous item */
	dictentry_s  *next;    /* next item */
	dictentry_s  *right;   /* next item in bucket */
	Any           key;     /* the key */
	Any           value;   /* the value */
	long          hash;    /* key hash value */
};

struct TDict
{
	CONTAINER_HEAD;
	long          mask;     /* current mask */
	Any           ktype;    /* key type/validator, maybe 0 */
	Any           vtype;    /* value type/validator, maybe 0 */
	dictentry_s **buckets;  /* buckets */
	dictentry_s   sentinel; /* sentinel */
};


/** DictFlag
	Flags used by Dict objects.

	Never set/unset these flags directly.
	Use the available flag accessors instead.
*/
#define DictFlagPROTECTED ContainerFlagPROTECTED /* program level protected? */
#define DictFlagFROZEN    ContainerFlagFROZEN    /* mutable? implies DictFlagFIXED */
#define DictFlagFIXED     ContainerFlagFIXED     /* resizable? */
#define DictFlagENUM      Flag10                 /* built from enum statement? */


/** DictIsFrozen
	Whether or not the Dict is mutable.

	DictFreeze and DictUnfreeze sets and unsets the flag.
	They must be used with care:
	@ul
	.To freeze a Dict you must own the global lock.

	.If you are not the creator of the Dict, you must call DictFreeze/DictUnfreeze
	 by pairs. You must only do this in one function.

	.As a rule of thumb, only call DictFreeze after creating a static Dict object.
*/
#define DictIsFrozen(self)  ContainerIsFrozen (self)
#define DictFreeze(self)    ContainerFreeze   (self)
#define DictUnfreeze(self)  ContainerUnfreeze (self)


/** DictIsFixed
	Whether or not the Dict is resizable.

	Only Dict.c may set/unset this flag.
	Others may only query this flag.
*/
#define DictIsFixed(self)  ContainerIsFixed (self)


/** DictIsPrinting
	Whether or not we are currently printing the Dict.

	Only Dict.c may set/unset the DictFlagPRINT flag.
	Others may only query this flag.
*/
#define DictIsPrinting(self)  FlagTest ((self), DictFlagPRINT)


/** DictIsEnum
	Whether the Dict came from an C{enum} statement.

	Only the compiler may set this flag when building a Dict from
	an enum statement. Others may only query this flag.
*/
#define DictIsEnum(self)  FlagTest ((self), DictFlagENUM)
#define DictSetEnum(self) FlagSet  ((self), DictFlagENUM)


/** DictEMPTY
	This is a predefined system frozen empty Dict object.

	C{const [:]} points to this object.

	@warn
	Never ever modify this Dict.
*/
extern Dict DictEMPTY;


/** DictNew DictRoot DictStatic
	Create a new Dict with a specified initial capacity.

	DictRoot creates a root Dict object.
	It is always marked by the GC and may contain anything.

	DictStatic creates a system static Dict object.
	It is never marked by the GC and may only contain other static objects
	such as Int, ={FloatStatic}, ={StringStatic}, etc.

	Make sure v{len} is smaller than ={GC_MAX_ITEMS}.

	DictNewArgv is called by the interpreter.
	You should have no reason to call this function directly.
*/
extern Dict DictNew     (long len);
extern Dict DictRoot    (long len);
extern Dict DictStatic  (long len);
extern Dict DictNewArgv (int argc, Any *argv);


/** DictCopy
	Get a copy of the Dict.
*/
extern Dict DictCopy (Dict self);


/** DictClear
	Empty the Dict.

	@warn
	An exception is thrown if the Dict is frozen.
*/
extern Dict DictClear (Dict self);


/** DictGetItem
	DictGetItem find the associated value of a key without throwing an exception.
	Return NULL if the key is not found.

	DictSetItem set the value of an existing item or create a new entry.
	Return 0 and throw EModify, EArgValue, or EMemory on failure.
	Return 1 on success.

	DictDelete remove a key.
	Return 0 and throw EModify on failure.
	Return 1 on success.

	DictDeleteAny remove any arbitrary key without resizing the Dict.
	If the Dict is empty the nothing happens.
*/
extern Any  DictGetItem   (Dict self, Any key);
extern Any  DictGetItemz  (Dict self, const char *key, long len);
extern int  DictSetItem   (Dict self, Any key, Any val);
extern int  DictDelete    (Dict self, Any key);
extern void DictDeleteAny (Dict self);


/** DictLike
	Return 1 if all keys in self are found in v{other}, 0 otherwise.
*/
extern int DictLike (Dict self, Dict other);


/*) DictOf
	Set key/value type. To set key then type may be C{[type, nil] or C{type}.
	To set the value, it must be C{[nil, type]}. To set both it must be
	C{[keytype, valuetype]}.
*/
extern Dict DictOf (Dict self, Any type);


/** DictWalk
	Walk through the items.

	Generally you should not call this function unless you are the creator
	of the Dict.

	Note that the Dict is not frozen while walking thus may be modified at any
	time. The algorithm used, however, makes it safe from crashing.

	v{b} and v{i} are index used to keep a safe iteration.
	You don't need to know what they are except that they must be set
	to zero on the first call.

	@xmp
		long b = 0, i = 0;
		Any  key, val;

		while ((val = DictWalk (MyFooDict, &key, &b, &i)) != 0)
		{
			... do something with key/val ...
		}
*/
extern Any DictWalk (Dict self, Any *key, long *b, long *i);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Dict class */
extern void DictSetup ();
extern void DictInit  ();


/*
	DictArgNew is experimental.
	It is used when creating a Dict when calling a function/method by argument
	name. Do not use it.
*/
extern Dict DictArgNew  (int argc, Any *argv);


#endif /*_H_Dict_*/
