/* Exception.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Exception_
#define _H_Exception_ 1


/** Exception
	This is the Exception class.

	It is the base class of all exception classes.
*/
extern Class ExceptionClass;


/** TException
	This represents an instance of an Exception class.

	Note that the message is an instance property named v{msg}.
*/
struct TException
{
	OBJ_HEAD;
};


/** ExceptionNewClass
	Create a new exception class.

	Note that you can also use ={ClassNew} and ={SetSuper} and
	={AttrSet} but this is much more convenient.
*/
extern Class ExceptionNewClass (Module m, Class sup, const char *name, const char *doc);


/** ExceptionErrno
	Find the exception class associated with v{errno}.
*/
extern Class ExceptionErrno (int eno);


/** ExceptionClassIs
	Whether or not the object is an exception class.
*/
#define ExceptionClassIs(o) (ClassIs(o) && ClassIsa ((Class)(o), ExceptionClass))


/** ExceptionIs
	Whether or not the object is an instance of an exception class.
*/
#define ExceptionIs(o) ObjectInstOf (o, ExceptionClass)


/** EXCEPTION_LAST_IS
	Check whether the last exception thrown by this thread is a specific class.
*/
#define EXCEPTION_LAST_IS(kls) \
	(ThreadCurrent->eclass == kls##Class)


/** ExceptionExitCode
	The exit code set by the builtin -{__exit__} function is the error code
	number passed to the system f{exit} function.
*/
extern int ExceptionExitCode;


/** THROW
	Use these macros to throw exceptions from C code.

	For the method or function name to be appropriately included in the
	message, you must name it using this convention:

	@ul
	. C{Xxx__xxx} : for standard methods.
	. C{Xxx_xxx}  : for non-standard methods or functions.

	@xmp
		static Int Foo__cmp (Foo self, Int a)
		{
			...
		}

		static Int Foo_bar (Int a, String b)
		{
			...
		}

	@warn
	v{strz} must be a static string, e.g. C{"foo bar"}. It may not be a
	dynamic pointer.

	The arguments to THROWF must be C values, i.e. ala f{fprint}.

	@xmp
		void foo_bar (String s)
		{
			...
			THROWF (EValue, "%s is fooed", s);        # wrong!
			THROWF (EValue, "%s is fooed", s->str);   # correct
			...
		}
*/
#define RETHROW \
BEGIN \
	ThreadRethrow (); \
	return 0; \
END

#define THROW(kls) \
BEGIN \
	ThreadThrow (kls##Class, 0); \
	return 0; \
END

#define THROWV(kls, v) \
BEGIN \
	ThreadThrow (kls##Class, 0); \
	return v; \
END

#define THROWS(kls, s) \
BEGIN \
	ThreadThrow (kls##Class, s); \
	return 0; \
END

#define THROWZ(kls, strz) \
BEGIN \
	ThreadThrow (kls##Class, StringNewz (strz)); \
	return 0; \
END

#define THROWF(kls, ...) \
BEGIN \
	ThreadThrow (kls##Class, StringNewf (__VA_ARGS__)); \
	return 0; \
END

#define THROWERRNO \
BEGIN \
	ThreadThrow (ExceptionErrno (errno), 0); \
	return 0; \
END

#define THROWERRNOF(fmt, ...) \
BEGIN \
	ThreadThrow (ExceptionErrno (errno), StringNewf (fmt, __VA_ARGS__)); \
	return 0; \
END

#define THROWERRNOV(val) \
BEGIN \
	ThreadThrow (ExceptionErrno (errno), 0); \
	return (val); \
END


/** ENSURE REQUIRE
	Use these macros to throw an exception by testing an expression.

	@block
	Use ENSURE to override ={Thread}->ewhere set by a previous function
	call.

	@xmp
		static Int _foo_do_something (Int x)
		{
			assert (IntIs (x));
			...
			THROW (EFoo);
				--> sets ThreadCurrent->ewhere to "_foo_do_something"
			...
			return Int0;
		}

		static sub Foo_bar (Int x)
		{
			long n;

			ArgTOLI (x, n);
			REQUIRE (EBar, n > 0);
			ENSURE (_do_something (x));
				--> if _foo_do_something throws the this will override
				    ThreadCurrent->ewhere to "Foo_bar" then return 0
			...
		}
*/
#define ENSURE(expr) \
BEGIN \
	if ((expr) == 0) \
		RETHROW; \
END


#define ENSUREZ(expr, msgz) \
BEGIN \
	if ((expr) == 0) \
	{ \
		ThreadSetEmsg (StringNewz (msgz)); \
		RETHROW; \
	} \
END


#define ENSUREV(expr, ret) \
BEGIN \
	if ((expr) == 0) \
	{ \
		ThreadRethrow (); \
		return ret; \
	} \
END


#define REQUIREZ(kls, expr, strz) \
BEGIN \
	if (!(expr)) \
	{ \
		ThreadThrow (kls##Class, StringNewz (strz)); \
		return 0; \
	} \
END

#define REQUIRE(kls, expr) \
BEGIN \
	if (!(expr)) \
	{ \
		ThreadThrow (kls##Class, 0); \
		return 0; \
	} \
END


/** ExceptionDefMsg
	Get the default message of an exception class.
*/
extern String ExceptionDefMsg (Class self);


/** ExceptionGetMsg
	Get the message associated with an Exception object.
*/
extern Any ExceptionGetMsg (Any x);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize exception classes */
extern void ExceptionSetup ();
extern void ExceptionInit  ();

#define _EXCEPTION_HEADER
#include "Exceptions.h"


/* for Sys.c */
extern Exception Exception_new    (Class self, int argc, Any *argv);


#endif /*_H_Exception_*/
