/* File.h - Copyright (c) Marc Krisnanto */

#ifndef _H_File_
#define _H_File_ 1


/** File
	This is the final File class.

	Most methods will throw an ESystem exception class according to v{errno}.
	See ={THROW} and ={ExceptionErrno}.
*/
extern Class FileClass;


/** TFile
	This represents an instance of the File class.

	The struct is intentionally hidden.
*/
/* TFile is hidden */


/** FilePathSepStr
	The file path separator as a ={String} object, e.g. C{"/"}.
*/
extern String FilePathSepStr;


/** FileIs
	Whether or not the object is a File object.
*/
#define FileIs(o) (ClassOf(o) == FileClass)


/** FileDebug
	If 1, then some File operations such as chmod, copy, rename, dump, etc.
	are not actually performed.
*/
extern int FileDebug;


/** FileFunction
	Convenience functions needed by some other modules.

	I'll document these later :-)
*/
extern String FileDir       (String path);
extern String FileExt       (String path);
extern int    FileExist     (String path, String type);
extern int    FileIsabs     (String path);
extern int    FileIsdir     (String path);
extern int    FileIsroot    (String path);
extern int    FileIstype    (String path, String filtertype);
extern Set    FileList      (String path, String filtertype, String filtername, int strip);
extern String FileLoadEx    (String path, long timeout, long max);
extern String FileLoad      (String path);
extern String FileMake      (String path, String defdir, String defext);
extern String FileName      (String path);
extern String FileNameExt   (String path);
extern String FileRepext    (String path, String ext);
extern String FileStrip     (String path);
extern String FileToabs     (String path, String defdir);
extern int    FileValid     (Any path, int allowspace);


/** FileMatch
	Match name with pattern.

	Wildcards recognized are: C{"*"} C{"?"} and C{"."}

	@xmp
		"f?o.*"
*/
extern int FileMatch (const char *pat, long lp, const char *name, long ln);


/** FileGetPermission
	Get permision mode from a string specification.

	On success 1 is returned and v{p} will contain the flags (a combination
	of S_IRUSR, S_IWUSR, S_IXUSR, etc.).

	On failure, 0 is returned.

	The string must be a combination of:
	@args
	.C{u}  user
	.C{g}  group
	.C{o}  others

	@block
	each followed by a combination of:
	@args
	.C{r}  has read permission
	.C{w}  has write permission
	.C{x}  has execute permission

	@xmp
		"urwx grw orw"
*/
extern int FileGetPermission (const char *s, int *p);


/** FileGetOpen
	Get file open mode from a string specification.

	On success 1 is returned and v{mode} and v{perm} will contain the open
	and permission flags.

	On failure, 0 is returned.

	The mode is a combination of:
	@args
		.C{r}  for read           (O_RDONLY)
		.C{w}  for write          (O_WRONLY)
		.C{rw} for read and write (O_RDWR)
		.C{a}  for append only    (O_APPEND)
	@block
	followed by a combination of:
	@args
		.C{t}  truncate after opened    (O_TRUNC)
		.C{c}  create if does not exist (O_CREAT)
		.C{l}  dont follow link         (O_NOFOLLOW)
		.C{n}  non-blocking             (O_NONBLOCK)
		.C{x}  exclusive                (O_EXCL)
		.C{s}  auto synchronize         (O_SYNC)

	@block
	and possibly followed by C{":"} then a permission (see ={FileGetPermission}).

	@xmp
		"rwtc:urwxgrworw"
*/
extern int FileGetOpen (const char *s, int *mode, int *perm);


/** FileGetFilter
	Get filter mode from a string specification.

	On success 1 is returned and v{filter} will contain the flags.

	On failure, 0 is returned.

	@args
	.C{s}   socket           (S_ISSOCK)
	.C{l}   symbolic link    (S_IFLNK)
	.C{f}   regular file     (S_IFREG)
	.C{b}   block device     (S_IFBLK)
	.C{d}   directory        (S_IFDIR)
	.C{c}   character device (S_IFCHR)
	.C{i}   fifo             (S_ISFIFO)
	.C{h}   hidden file      (S_IFHIDDEN)
	.C{v}   sticky bit       (S_ISVTX)
	.C{u}   set UID bit      (S_ISUID) or user  permission if followed by C{rwx} combination
	.C{g}   set GID bit      (S_ISGID) or group permission if followed by C{rwx} combination
	.C{o}                    others permission followed by C{rwx} combination
*/
extern int FileGetFilter (const char *s, int *filter);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the File class. */
extern void FileSetup ();
extern void FileInit  ();

/* for Sub.c __interface */
extern int FileModified (String path, double lasttime);

#endif /*_H_File_*/
