/* Float.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Float_
#define _H_Float_ 1


/** Float
	This is the final Float class.
*/
extern Class FloatClass;


/** TFloat
	This represents an instance of the Float class.
*/
struct TFloat
{
	OBJ_HEAD;
	double value; /* may not be NaN or infinities */
};


/** FloatFlags
	Flags used by Float objects.
*/
/* all reserved */
#define FloatFOO  Flag0  /* just to annoy you */


/** FloatPredefined
	Predefined system Float objects.

	See ={FloatStatic} to create your own.

	@warn
	FloatMIN is a negative value thus contrary to DBL_MIN which is a positive
	value.
*/
extern Float FloatMIN; /* -DBL_MAX */
extern Float FloatMAX; /* +DBL_MAX */
extern Float Float0;   /*  0.0 */
extern Float Float1;   /*  1.0 */
extern Float Float2;   /*  2.0 */
extern Float Float_1;  /* -1.0 */
extern Float Float_2;  /* -2.0 */


/** FloatNew
	Create a new Float object from various C types or other objects.

	@warn
	You may not pass a NaN or infinity to FloatNew.
*/
#define      FloatNew       FloatNewd
#define      FloatNewh(v)   FloatNewd (v)
#define      FloatNewuh(v)  FloatNewd (v)
#define      FloatNewi(v)   FloatNewd (v)
#define      FloatNewui(v)  FloatNewd (v)
#define      FloatNewl(v)   FloatNewd (v)
#define      FloatNewul(v)  FloatNewd (v)
#define      FloatNewli(v)  FloatNewd (v)
extern Float FloatNewd      (double v);
#ifdef HAVE_LONG_LONG
#define      FloatNewll(v)  FloatNewd (v)
#define      FloatNewull(v) FloatNewd (v)
#endif
#ifdef HAVE_LONG_DOUBLE
extern Float FloatNewld     (long double v);
#endif

#define      FloatNewInt(x)      FloatNewd (IntTol (x))
extern Float FloatNewLong        (Long x);
extern Float FloatNewRational    (Rational x);
#define      FloatNewFloat(x)    (x)
#define      FloatNewBig(x)      BigToFloat (x)
#define      FloatNewComplex(x)  ComplexToFloat (x)
extern Float FloatNewInteger     (Integer x);
extern Float FloatNewDecimal     (Decimal x);
extern Float FloatNewReal        (Real x);
extern Float FloatNewNumber      (Number x);
extern Float FloatNewString      (String x);
#define      FloatNewBoolean(x)  (((x) == True) ? Float1 : Float0)


/** FloatNewdx
	Create a new Float/Real object from a C double by checking whether or not the
	value is NaN or infinity.

	If the value is NaN then EValue is thrown and NULL is returned.

	If the value is infinity then EInf is thrown and NULL is returned unless the
	={NumberContext} allows infinities to be returned, in which case one of
	={InfPositive} or ={InfNegative} is returned based on the sign.
*/
extern Real FloatNewdx (double v);
#ifdef HAVE_LONG_DOUBLE
extern Real FloatNewldx (long double v);
#endif


/** FloatStatic
	Create a new system Float object, i.e. not in GC.
*/
extern Float FloatStatic   (double v);
extern Float FloatToStatic (Float x);


/** FloatTo
	Convert a Float to a C type or another object.

	For C type conversions, the functions set v{errno} to zero on success,
	to EDOM if conversion is not possible or ERANGE if the value falls out of the
	Float range. The I{unsigned} versions fail with ERANGE if the value is negative.

	For object conversions, only FloatToInt can fail (in which case NULL is returned
	and an exception set) because it is always possible to convert a Float into a
	={Long}, ={Big}, ={Rational}, and ={Complex}.
*/
#define                   FloatToh(x)   Number_ltoh   (FloatTol  (x))
#define                   FloatTouh(x)  Number_ultouh (FloatToul (x))
#define                   FloatToi(x)   Number_ltoi   (FloatTol  (x))
#define                   FloatToui(x)  Number_ultoui (FloatToul (x))
#define                   FloatToli(x)  Number_ltoli  (FloatTol  (x))
extern long               FloatTol      (Float x);
extern unsigned long      FloatToul     (Float x);
extern float              FloatTof      (Float x);
extern double             FloatTod      (Float x);
#ifdef HAVE_LONG_LONG
extern long long          FloatToll     (Float x);
extern unsigned long long FloatToull    (Float x);
#endif
#ifdef HAVE_LONG_DOUBLE
extern long double        FloatTold     (Float x);
#endif

extern Int     FloatToInt         (Float x);
#define        FloatToFloat(x)    (x)
#define        FloatToLong(x)     LongNewd ((x)->value)
#define        FloatToRational(x) RationalNewd ((x)->value)
#define        FloatToBig(x)      BigNewd ((x)->value)
#define        FloatToComplex(x)  ComplexNew1 ((Real) (x))
extern Integer FloatToInteger     (Float x);
#define        FloatToDecimal(x)  ((Decimal) (x))
#define        FloatToReal(x)     ((Real) (x))
#define        FloatToNumber(x)   ((Number) x)
#define        FloatToBoolean(x)  ((x)->value ? True : False)
extern String  FloatToString      (Float x);


/** FloatSign
	FloatSign returns -1, 0, or 1 if the number is negative, 0, or positive respectively.
*/
extern int FloatSign (Float x);


/** FloatIs
	Identity functions.

	FloatIs tests whether or not an object is a Float.

	FloatIsNeg returns 1 if the number is negative.
	Returns 0 otherwise.

	FloatIsZero returns 1 if the number is zero.
	Returns 0 otherwise.

	FloatIsOne returns 1 if the number is -1 or 1.
	Returns 0 otherwise.

	FloatIs1 return 1 if the number is 1.
	Returns 0 otherwise.

	FloatIsWhole return 1 if the number is whole.
	Returns 0 otherwise.
	The system function f{modf} is used to test.

	FloatIsFrac return 1 if 0 < x < 1.
	Returns 0 otherwise.

	FloatIsOdd return 1 if the number is odd.
	Returns 0 otherwise.
	The system function f{frexp} and f{fmod} is used to test.
*/
#define    FloatIs(x)      (ClassOf (x) == FloatClass)
#define    FloatIsZero(x)  ((x)->value == 0)
#define    FloatIsNeg(x)   ((x)->value < 0)
#define    FloatIs1(x)     ((x)->value == 1)
extern int FloatIsFrac     (Float x);
#define    FloatIsWhole(x) (modf ((x)->value, &FloatIsWholeHelper) == 0)
extern int FloatIsOne      (Float x);
extern int FloatIsOdd      (Float x);
/*helper*/
extern double FloatIsWholeHelper; /* do not use! */


/** FloatUnary
	Unary functions.

	FloatPos returns the number itself, obviously.

	FloatNeg returns a negated value.
	The result is a Float.

	FloatIncr returns the Float incremented by 1.
	The result is either a Float or a Big.

	FloatDecr returns the Float decremented by 1.
	The result is either an Float or a Big.

	Note that all always succeed.
*/
#define FloatPos(x)   (x)
extern Float   FloatNeg  (Float self);
extern Decimal FloatIncr (Float self);
extern Decimal FloatDecr (Float self);


/** FloatEq
	Whether the value of a Float is the same as the value of another Number object.
	All return 0 or 1.
*/
#define    FloatEqInt(x, y)     ((x)->value == IntTod (y))
extern int FloatEqLong          (Float x, Long y);
extern int FloatEqRational      (Float x, Rational y);
#define    FloatEqFloat(x, y)   ((x)->value == (y)->value)
#define    FloatEqBig(x, y)     BigEqFloat (y, x)
#define    FloatEqComplex(x, y) ComplexEqFloat (y, x)
#define    FloatEqInf(x, y)     (0)
extern int FloatEqInteger       (Float x, Integer y);
extern int FloatEqDecimal       (Float x, Decimal y);
extern int FloatEqReal          (Float x, Real y);
extern int FloatEqNumber        (Float x, Number y);


/** FloatLike
	Whether the absolute value of a Float is the same as the absolute value of another
	Number object.
	All return 0 or 1.
*/
#define    FloatLikeInt(x, y)     (FloatCmpAbsInt (x, y) == 0)
#define    FloatLikeLong(x, y)    (FloatCmpAbsLong (x, y) == 0)
extern int FloatLikeRational      (Float x, Rational y);
extern int FloatLikeFloat         (Float x, Float y);
#define    FloatLikeBig(x, y)     BigLikeFloat (y, x)
#define    FloatLikeComplex(x, y) ComplexLikeFloat (y, x)
#define    FloatLikeInf(x, y)     (0)
extern int FloatLikeInteger       (Float x, Integer y);
extern int FloatLikeDecimal       (Float x, Decimal y);
extern int FloatLikeReal          (Float x, Real y);
extern int FloatLikeNumber        (Float x, Number y);


/** FloatCmp
	Compare a Float with another Number object.
	All return -1, 0 or 1 in the usual manner.
*/
extern int FloatCmpInt           (Float x, Int y);
extern int FloatCmpLong          (Float x, Long y);
extern int FloatCmpRational      (Float x, Rational y);
extern int FloatCmpFloat         (Float x, Float y);
#define    FloatCmpBig(x, y)     (-BigCmpFloat (y, x))
#define    FloatCmpComplex(x, y) (-ComplexCmpFloat (y, x))
#define    FloatCmpInf(x, y)     (-InfCmpFloat (y, x))
extern int FloatCmpInteger       (Float x, Integer y);
extern int FloatCmpDecimal       (Float x, Decimal y);
extern int FloatCmpReal          (Float x, Real y);
extern int FloatCmpNumber        (Float x, Number y);


/** FloatCmpAbs
	Compare the absolute value of a Float with the absolute value of another
	Number object.
	All return -1, 0 or 1 in the usual manner.
*/
extern int FloatCmpAbsInt           (Float x, Int y);
extern int FloatCmpAbsLong          (Float x, Long y);
extern int FloatCmpAbsRational      (Float x, Rational y);
extern int FloatCmpAbsFloat         (Float x, Float y);
#define    FloatCmpAbsBig(x, y)     (-BigCmpAbsFloat (y, x))
#define    FloatCmpAbsComplex(x, y) (-ComplexCmpAbsFloat (y, x))
#define    FloatCmpAbsInf(x, y)     (-1)
extern int FloatCmpAbsInteger       (Float x, Integer y);
extern int FloatCmpAbsDecimal       (Float x, Decimal y);
extern int FloatCmpAbsReal          (Float x, Real y);
extern int FloatCmpAbsNumber        (Float x, Number y);


/** FloatPlus
	Add a Float with another Number object.
*/
extern Decimal FloatPlusInt           (Float x, Int y);
extern Real    FloatPlusLong          (Float x, Long y);
extern Real    FloatPlusRational      (Float x, Rational y);
extern Decimal FloatPlusFloat         (Float x, Float y);
#define        FloatPlusBig(x, y)     BigPlusFloat (y, x)
#define        FloatPlusComplex(x, y) ComplexPlusFloat (y, x)
#define        FloatPlusInf(x, y)     InfPlusFloat (y, x)
extern Real    FloatPlusInteger       (Float x, Integer y);
extern Real    FloatPlusDecimal       (Float x, Decimal y);
extern Real    FloatPlusReal          (Float x, Real y);
extern Number  FloatPlusNumber        (Float x, Number y);


/** FloatMinus
	Subtract a Float with another Number object.
*/
extern Decimal FloatMinusInt           (Float x, Int y);
extern Real    FloatMinusLong          (Float x, Long y);
extern Real    FloatMinusRational      (Float x, Rational y);
extern Decimal FloatMinusFloat         (Float x, Float y);
#define        FloatMinusBig(x, y)     BigRminusFloat (y, x)
#define        FloatMinusComplex(x, y) ComplexRminusFloat (y, x)
#define        FloatMinusInf(x, y)     InfPlusFloat (InfNeg (y), x)
extern Real    FloatMinusInteger       (Float x, Integer y);
extern Real    FloatMinusDecimal       (Float x, Decimal y);
extern Real    FloatMinusReal          (Float x, Real y);
extern Number  FloatMinusNumber        (Float x, Number y);


/** FloatRminus
	Subtract another Number object with a Float.
	This is only for efficiency.
*/
extern Decimal FloatRminusInt      (Float x, Int y);
extern Real    FloatRminusLong     (Float x, Long y);
extern Real    FloatRminusRational (Float x, Rational y);


/** FloatMul
	Multiply a Float with another Number object.
*/
extern Decimal FloatMulInt           (Float x, Int y);
extern Real    FloatMulLong          (Float x, Long y);
extern Real    FloatMulRational      (Float x, Rational y);
extern Decimal FloatMulFloat         (Float x, Float y);
#define        FloatMulBig(x, y)     BigMulFloat (y, x)
#define        FloatMulComplex(x, y) ComplexMulFloat (y, x)
#define        FloatMulInf(x, y)     InfMulFloat (y, x)
extern Real    FloatMulInteger       (Float x, Integer y);
extern Real    FloatMulDecimal       (Float x, Decimal y);
extern Real    FloatMulReal          (Float x, Real y);
extern Number  FloatMulNumber        (Float x, Number y);


/** FloatDiv
	Divide a Float with another Number object.
*/
extern Real   FloatDivInt           (Float x, Int y);
extern Real   FloatDivLong          (Float x, Long y);
extern Real   FloatDivRational      (Float x, Rational y);
extern Real   FloatDivFloat         (Float x, Float y);
#define       FloatDivBig(x, y)     BigRdivFloat (y, x)
#define       FloatDivComplex(x, y) ComplexRdivFloat (y, x)
extern Float  FloatDivInf           (Float x, Inf y);
extern Real   FloatDivInteger       (Float x, Integer y);
extern Real   FloatDivDecimal       (Float x, Decimal y);
extern Real   FloatDivReal          (Float x, Real y);
extern Number FloatDivNumber        (Float x, Number y);


/** FloatRdiv
	Divide another Number object with a Float.
	This is only for efficiency.
*/
extern Real FloatRdivInt      (Float x, Int y);
extern Real FloatRdivLong     (Float x, Long y);
extern Real FloatRdivRational (Float x, Rational y);


/** FloatIdiv
	Integer division of a Float with another Number object.
*/
extern Integer FloatIdivInt           (Float x, Int y);
extern Integer FloatIdivLong          (Float x, Long y);
extern Integer FloatIdivRational      (Float x, Rational y);
extern Integer FloatIdivFloat         (Float x, Float y);
#define        FloatIdivBig(x, y)     BigRidivFloat (y, x)
#define        FloatIdivComplex(x, y) ComplexRidivFloat (y, x)
extern Integer FloatIdivInf           (Float x, Inf y);
extern Integer FloatIdivInteger       (Float x, Integer y);
extern Integer FloatIdivDecimal       (Float x, Decimal y);
extern Integer FloatIdivReal          (Float x, Real y);
extern Integer FloatIdivNumber        (Float x, Number y);


/** FloatRidiv
	Integer division of another Number object with a Float.
	This is only for efficiency.
*/
extern Integer FloatRidivInt      (Float x, Int y);
extern Integer FloatRidivLong     (Float x, Long y);
extern Integer FloatRidivRational (Float x, Rational y);


/** FloatRem
	Remainder of dividing a Float with another Number object.
*/
extern Real    FloatRemInt           (Float x, Int y);
extern Real    FloatRemLong          (Float x, Long y);
extern Real    FloatRemRational      (Float x, Rational y);
extern Real    FloatRemFloat         (Float x, Float y);
#define        FloatRemBig(x, y)     BigRremFloat (y, x)
#define        FloatRemComplex(x, y) ComplexRremFloat (y, x)
extern Float   FloatRemInf           (Float x, Inf y);
extern Real    FloatRemInteger       (Float x, Integer y);
extern Real    FloatRemDecimal       (Float x, Decimal y);
extern Real    FloatRemReal          (Float x, Real y);
extern Number  FloatRemNumber        (Float x, Number y);


/** FloatRrem
	Remainder of dividing another Number object with a Float.
	This is only for efficiency.
*/
extern Real FloatRremInt      (Float x, Int y);
extern Real FloatRremLong     (Float x, Long y);
extern Real FloatRremRational (Float x, Rational y);


/** FloatPow
	Raise a Float to the power of another Number object.
*/
extern Number FloatPowInt           (Float x, Int y);
extern Number FloatPowLong          (Float x, Long y);
extern Number FloatPowRational      (Float x, Rational y);
extern Number FloatPowFloat         (Float x, Float y);
#define       FloatPowBig(x, y)     BigRpowFloat (y, x)
#define       FloatPowComplex(x, y) ComplexRpowFloat (y, x)
extern Real   FloatPowInf           (Float x, Inf y);
extern Number FloatPowInteger       (Float x, Integer y);
extern Number FloatPowDecimal       (Float x, Decimal y);
extern Number FloatPowReal          (Float x, Real y);
extern Number FloatPowNumber        (Float x, Number y);


/** FloatRpow
	Raise another Number object to the power of a Float.
	This is only for efficiency.
*/
extern Number FloatRpowInt      (Float x, Int y);
extern Number FloatRpowLong     (Float x, Long y);
extern Number FloatRpowRational (Float x, Rational y);


/** FloatRound
	Rounding functions.

	FloatCeil rounds the number to the nearest integer towards positive infinity.

	FloatFloor rounds the number to the nearest integer towards negative infinity.

	FloatTrunc rounds the number towards zero, that is, it removes the fraction part.

	FloatRound rounds a Float to the specified number of fraction digits.

	Note that all of these functions always succeed.
*/
extern Decimal FloatCeil  (Float x);
extern Decimal FloatFloor (Float x);
extern Float   FloatTrunc (Float x);
extern Float   FloatRound (Float x, int prec);


/** FloatFrac
	Return the fraction part.
*/
extern Float FloatFrac (Float x);


/** FloatScan
	Scan a Float number.

	v{s} is the start of the string.
	If v{neg} is true then the scanner will not skip leading blanks and
	will not read the sign, that is, the string may not contain the sign
	nor leading spaces.

	On success FloatScand returns a C double value while FloatScan returns a
	Float object. Both update v{s} to the next cursor position after the number.

	On failure FloatScand sets v{errno} and returns NULL while FloatScan throws
	ESyntax and return NULL.
*/
extern double FloatScand (char **s, int neg);
extern Float  FloatScan  (char **s, int neg);


/** FloatPrint
	Print a Float to v{stdout}.

	Note that this might throw an exception because writing to v{stdout} may
	fail for various reasons.
*/
extern Null FloatPrint (Float self);


/** FloatFormat
	Format a Float.

	See ={Format} for details.
*/
extern String FloatFormat (Float x, format_s *f);


/** FloatHash
	Return the hash value of a C double.

	This is the function used to calculate the hash of a Float object.
	The value should match the value of other ={Number} objects which have
	the same value (the equality test returns 1).
*/
#define FloatHashd(v) ((long) (v))


/** FloatIntegerFactor
	Return the Integer which must be multiplied to the float to get an Integer.

	@xmp
		FloatIntegerFactor (1.2) = 10
		FloatIntegerFactor (1.23) = 100
*/
extern Integer FloatIntegerFactor (Float x);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Float class */
extern void FloatSetup ();
extern void FloatInit ();

#endif /*_H_Float_*/
