/* Frame.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Frame_
#define _H_Frame_ 1

#include "Compile.h"


/** Frame
	The final Frame class.

	Frames are created to execute program level code, i.e. initializing modules
	and calling functions/methods.
*/
extern Class FrameClass;


/** TFrame
	This represents an instance of the Frame class.
*/
typedef struct frblk_s frblk_s;
struct frblk_s
{
	Any    obj;     /* switch value or for-each iterator object or 0 */
	long   ex1;     /* \__ for builtin class iterators */
	long   ex2;     /* /   used internally by interp.c */
	long   cont;    /* where to 'continue' */
	long   nextip;  /* where to jump when leaving the block */

	/*
		Note that obj is not necessarily an Iterator. It may be an Array, Dict,
		Range, or Set. The interpreter does not create an Iterator object for
		those. This is much more efficient and faster.

		For bytecode, using pointers for the ip and jump targets is not a win.
		I have tried it and lost about 10% speed. And that includes a number
		of delay jump strategies. If you can do better, let me know. -marc-

		JIT uses 'nextip' as the address of block info embedded in the JIT
		code and 'cont' as *jit_insn. That is why we use longs.
	*/
};

/* be careful not to exceed GC_SIZE_LIMIT */
struct TFrame
{
	OBJ_HEAD;
	Code    code;    /* the associated Code object */
	Any     self;    /* the 'self' object */
	Any     result;  /* result */
	Class   eclass;  /* last exception thrown */
	long    ip;      /* current instruction pointer */
	Any    *ss;      /* stack base */
	Any    *bp;      /* start of stack after locals */
	Any    *sp;      /* current stack pointer */
	int     argc;    /* number of arguments passed */
	long    line;    /* current line */
	int     atomic;  /* depth of atomic block */
	int     level;   /* block level */

	frblk_s blocks [CompileMAXLEVEL]; /* for/switch/loop/etc. blocks */
};


/** FrameIs
	Whether or not an object is a Frame.
*/
#define FrameIs(o) (ClassOf(o) == FrameClass)


/** FRAME_CSTACK_MAX
	Non-generator frames generally use the C stack provided that they don't
	require a stack bigger than this.

	Setting this to a big value is not something you want to do.
*/
#define FRAME_CSTACK_MAX 32


/** FrameFlag
	Flags used by Frame objects.

	Do not access these flags directly.
	Use the accessor macros instead.
*/
#define FrameFlagSUB       Flag0  /* Sub or Module code? */
#define FrameFlagGENERATOR Flag1  /* is a generator */
#define FrameFlagRUNNING   Flag2  /* currently running? */
#define FrameFlagDONE      Flag3  /* completed? */
#define FrameFlagTHROWN    Flag4  /* exception thrown? */
#define FrameFlagRETURNING Flag5  /* return issued? */
#define FrameFlagJIT       Flag6  /* a JIT frame */
#define FrameFlagSWAPPING  Flag7  /* yielded from __swap__ */
#define FrameFlagCFRAME    Flag8  /* frame is on the C stack (not allocated) */


/** FrameIsSub
	Whether this is a Sub or Module code frame.
*/
#define FrameIsSub(self) FlagTest (self, FrameFlagSUB)


/** FrameIsGenerator
	Whether this comes from a generator Sub.
*/
#define FrameIsGenerator(self) FlagTest (self, FrameFlagGENERATOR)


/** FrameIsRunning
	Whether the frame is currently running.

	FrameFlagRUNNING may only be set/unset by the interpreter.
*/
#define FrameIsRunning(self)     FlagTest  (self, FrameFlagRUNNING)
#define FrameSetRunning(self)    FlagSet   (self, FrameFlagRUNNING)
#define FrameUnsetRunning(self)  FlagUnset (self, FrameFlagRUNNING)


/** FrameIsDone
	Whether the frame has completed running.

	FrameSetDone should only be set by the interpreter.
	Once set, it must never be unset.
*/
#define FrameIsDone(self)  FlagTest (self, FrameFlagDONE)
#define FrameSetDone(self) FlagSet  (self, FrameFlagDONE)


/** FrameIsThrown
	Whether an exception is thrown.

	The FrameFlagTHROWN should only be set by the interpreter.
*/
#define FrameIsThrown(self)    FlagTest  (self, FrameFlagTHROWN)
#define FrameSetThrown(self)   FlagSet   (self, FrameFlagTHROWN)
#define FrameUnsetThrown(self) FlagUnset (self, FrameFlagTHROWN)


/** FrameIsReturning
	Whether k{return} has been issued.

	FrameSetReturning should only be used by the interpreter.
	Once set, it must never be unset.
*/
#define FrameIsReturning(self)  FlagTest (self, FrameFlagRETURNING)
#define FrameSetReturning(self) FlagSet  (self, FrameFlagRETURNING)


/** FrameIsJit
	Whether this is a JIT frame.

	FrameFlagJIT may only be set by the interpreter.
	Once set, it must never be unset.
*/
#define FrameIsJit(self)  FlagTest (self, FrameFlagJIT)
#define FrameSetJit(self) FlagSet (self, FrameFlagJIT)


/** FrameIsSwapping
	Whether __swap__ has been issued.

	FrameFlagSWAPPING should only be set/unset by the interpreter.
*/
#define FrameIsSwapping(self)    FlagTest  (self, FrameFlagSWAPPING)
#define FrameSetSwapping(self)   FlagSet   (self, FrameFlagSWAPPING)
#define FrameUnsetSwapping(self) FlagUnset (self, FrameFlagSWAPPING)


/** FrameIsCframe
	Whether the frame is on the C stack (not GcMemMalloc'ed).
*/
#define FrameIsCframe(self) FlagTest (self, FrameFlagCFRAME)


/** FrameModule
	Get the Module that owns the Frame.
*/
extern Module FrameModule (Frame self);


/** FrameNext
	Get the next yielded value.
*/
extern Any FrameNext (Frame self);


/** FrameWhere
	Get a string that tells you the whereabouts of this Frame.
*/
extern String FrameWhere (Frame self);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Frame class. */
extern void FrameSetup ();
extern void FrameInit  ();

/* create frame to run Module code */
#define FrameNewModule(mod) Frame_New (((Module) mod)->code, mod, 0)
extern Frame Frame_New (Code c, Any theself, int nlocal);

/* for Interp.c */
extern Frame _frame_new_c (Frame f, Any *stack, Sub u, Any theself, int argc, Any *argv);
extern Frame _frame_new_a (Sub u, Any theself, int argc, Any *argv);
extern Frame _frame_new_g (Sub u, Any theself, int argc, Any *argv);

/* throw/rethrow */
extern void FrameRethrow (Frame f);
extern void FrameThrow   (Frame f, Class c);

/* dont touch yet */
extern Null FrameStop (Frame self, int argc, Any *argv);
extern Any  FrameCall (Frame self, int argc, Any *argv);

#endif /*_H_Frame_*/
