/* Inf.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Inf_
#define _H_Inf_ 1


/** Inf
	This is the final Inf class.
*/
extern Class InfClass;


/** InfPositive InfNegative
	These are system objects representing positive and negative infinity.

	They are the keywords infp and infn in program level.
*/
extern Inf InfPositive;
extern Inf InfNegative;


/** InfTo
	These are required by other macros.

	All C type conversions fail with errno set to ERANGE.

	All object conversions may fail and throw an exception.
	Those calling Inf_toc always throw EValue.
	Those calling __Inf__ throw EInf if the ={NumberContext} does not allow
	infinities.
*/
extern short              InfToh   (Inf x);
extern unsigned short     InfTouh  (Inf x);
extern int                InfToi   (Inf x);
extern unsigned int       InfToui  (Inf x);
extern long               InfTol   (Inf x);
extern unsigned long      InfToul  (Inf x);
extern long               InfToli  (Inf x);
extern float              InfTof   (Inf x);
extern double             InfTod   (Inf x);
#ifdef HAVE_LONG_LONG
extern long long          InfToll  (Inf x);
extern unsigned long long InfToull (Inf x);
#endif
#ifdef HAVE_LONG_DOUBLE
extern long double        InfTold  (Inf x);
#endif

#define        InfToInt(x)      Inf_toc ()
#define        InfToLong(x)     Inf_toc ()
#define        InfToRational(x) Inf_toc ()
#define        InfToFloat(x)    Inf_toc ()
#define        InfToBig(x)      Inf_toc ()
extern Complex InfToComplex     (Inf x);
#define        IntToInf(x)      __Inf__ (InfSign (x))
#define        InfToInteger(x)  Inf_toc ()
#define        InfToDecimal(x)  Inf_toc ()
#define        InfToReal(x)     ((Real) __Inf__ (InfSign (x)))
#define        InfToNumber(x)   ((Number) InfToReal(x))
extern String  InfToString      (Inf x);
/*helper*/
extern Any Inf_toc ();


/** InfSign
	Infinity sign.
	This returns 1 if v{x} is positive infinity, -1 otherwise.
*/
#define InfSign(x) (((x) == InfPositive) ? 1 : -1)



/** InfIs
	Infinity identity functions.

	InfIs returns 1 if v{x} is an infinity.
	Returns 0 otherwise.

	InfIsNeg returns 1 if v{x} is negative infinity.
	Returns 0 otherwise.

	All the others return 0, obviously.
*/
#define InfIs(x)      (ClassOf (x) == InfClass)
#define InfIsNeg(x)   (((x) == InfPositive) ? 0 : 1)
#define InfIsZero(x)  (0)
#define InfIsOne(x)   (0)
#define InfIs1(x)     (0)
#define InfIsWhole(x) (1)
#define InfIsFrac(x)  (0)
#define InfIsOdd(x)   (0)


/** InfUnary
	Infinity unary functions.
*/
#define InfPos(x)   (x)
#define InfNeg(x)   ((x) == InfPositive ? InfNegative : InfPositive)
#define InfIncr(x)  __Inf__ (InfSign (x))
#define InfDecr(x)  __Inf__ (InfSign (x))


/** InfEq
	Infinity equality tests.
	All return 0 or 1.
*/
#define    InfEqInt(x, y)      (0)
#define    InfEqFloat(x, y)    (0)
#define    InfEqLong(x, y)     (0)
#define    InfEqRational(x, y) (0)
#define    InfEqBig(x, y)      (0)
extern int InfEqComplex        (Inf x, Complex y);
#define    InfEqInf(x, y)      ((x) == (y))
extern int InfEqInteger        (Inf x, Integer y);
extern int InfEqDecimal        (Inf x, Decimal y);
extern int InfEqReal           (Inf x, Real y);
extern int InfEqNumber         (Inf x, Number y);


/** InfLike
	Infinity absolute value equality tests.
	All return 0 or 1.
*/
#define    InfLikeInt(x, y)      (0)
#define    InfLikeLong(x, y)     (0)
#define    InfLikeRational(x, y) (0)
#define    InfLikeFloat(x, y)    (0)
#define    InfLikeBig(x, y)      (0)
extern int InfLikeComplex        (Inf x, Complex y);
#define    InfLikeInf(x, y)      (1)
extern int InfLikeInteger        (Inf x, Integer y);
extern int InfLikeDecimal        (Inf x, Decimal y);
extern int InfLikeReal           (Inf x, Real y);
extern int InfLikeNumber         (Inf x, Number y);


/** InfCmp
	Infinity comparisson functions.
	All return -1, 0 or 1 in the usual manner.
*/
#define    InfCmpInt(x, y)      (((x) == InfPositive) ? 1 : -1)
#define    InfCmpLong(x, y)     (((x) == InfPositive) ? 1 : -1)
#define    InfCmpRational(x, y) (((x) == InfPositive) ? 1 : -1)
#define    InfCmpFloat(x, y)    (((x) == InfPositive) ? 1 : -1)
#define    InfCmpBig(x, y)      (((x) == InfPositive) ? 1 : -1)
extern int InfCmpComplex        (Inf x, Complex y);
extern int InfCmpInf            (Inf x, Inf y);
extern int InfCmpInteger        (Inf x, Integer y);
extern int InfCmpDecimal        (Inf x, Decimal y);
extern int InfCmpReal           (Inf x, Real y);
extern int InfCmpNumber         (Inf x, Number y);


/** InfCmpAbs
	Infinity absolute value comparisson functions.
	All return -1, 0 or 1 in the usual manner.
*/
#define    InfCmpAbsInt(x, y)      (1)
#define    InfCmpAbsLong(x, y)     (1)
#define    InfCmpAbsRational(x, y) (1)
#define    InfCmpAbsFloat(x, y)    (1)
#define    InfCmpAbsBig(x, y)      (1)
extern int InfCmpAbsComplex        (Inf x, Complex y);
#define    InfCmpAbsInf(x, y)      (0)
extern int InfCmpAbsInteger        (Inf x, Integer y);
extern int InfCmpAbsDecimal        (Inf x, Decimal y);
extern int InfCmpAbsReal           (Inf x, Real y);
extern int InfCmpAbsNumber         (Inf x, Number y);


/** InfDiv
	Divide an infinity with another Number object.
*/
#define        InfDivInt(x, y)      Inf_divr (x, IntSign (y))
#define        InfDivFloat(x, y)    Inf_divr (x, FloatSign (y))
#define        InfDivLong(x, y)     Inf_divr (x, LongSign (y))
#define        InfDivRational(x, y) Inf_divr (x, RationalSign (y))
#define        InfDivBig(x, y)      Inf_divr (x, BigSign (y))
extern Number  InfDivComplex        (Inf x, Complex y);
extern Any     InfDivInf            (Inf x, Inf y);
extern Inf     InfDivInteger        (Inf x, Integer y);
extern Inf     InfDivDecimal        (Inf x, Decimal y);
extern Inf     InfDivReal           (Inf x, Real y);
extern Number  InfDivNumber         (Inf x, Number y);
/*helper*/
extern Inf Inf_divr (Inf x, int sign);


/** InfIdiv
	Integer division of an infinity with another Number object.

	All throw EIdiv.
*/
#define InfIdivInt(x, y)      Inf_idivr ()
#define InfIdivFloat(x, y)    Inf_idivr ()
#define InfIdivLong(x, y)     Inf_idivr ()
#define InfIdivRational(x, y) Inf_idivr ()
#define InfIdivBig(x, y)      Inf_idivr ()
#define InfIdivComplex(x, y)  Inf_idivr ()
#define InfIdivInf(x, y)      Inf_idivr ()
#define InfIdivInteger(x, y)  Inf_idivr ()
#define InfIdivDecimal(x, y)  Inf_idivr ()
#define InfIdivReal(x, y)     Inf_idivr ()
#define InfIdivNumber(x, y)   Inf_idivr ()
/*helper*/
extern Integer Inf_idivr ();


/** InfMinus
	Subtract an infinity with another Number object.
*/
#define        InfMinusInt(x, y)      Inf_minusr(x)
#define        InfMinusFloat(x, y)    Inf_minusr(x)
#define        InfMinusLong(x, y)     Inf_minusr(x)
#define        InfMinusRational(x, y) Inf_minusr(x)
#define        InfMinusBig(x, y)      Inf_minusr(x)
extern Number  InfMinusComplex        (Inf x, Complex y);
extern Inf      InfMinusInf            (Inf x, Inf y);
extern Inf     InfMinusInteger        (Inf x, Integer y);
extern Inf     InfMinusDecimal        (Inf x, Decimal y);
extern Inf     InfMinusReal           (Inf x, Real y);
extern Number  InfMinusNumber         (Inf x, Number y);
/*helper*/
extern Inf Inf_minusr (Inf x);


/** InfMul
	Multiply an infinity with another Number object.
*/
#define       InfMulInt(x, y)      Inf_mulr (x, IntSign (y))
#define       InfMulFloat(x, y)    Inf_mulr (x, FloatSign (y))
#define       InfMulLong(x, y)     Inf_mulr (x, LongSign (y))
#define       InfMulRational(x, y) Inf_mulr (x, RationalSign (y))
#define       InfMulBig(x, y)      Inf_mulr (x, BigSign (y))
#define       InfMulComplex(x, y)  ComplexMulInf (y, x)
extern Inf    InfMulInf            (Inf x, Inf y);
extern Inf    InfMulInteger        (Inf x, Integer y);
extern Inf    InfMulDecimal        (Inf x, Decimal y);
extern Inf    InfMulReal           (Inf x, Real y);
extern Number InfMulNumber         (Inf x, Number y);
/*helper*/
extern Number Inf_mulr (Inf x, int sign);


/** InfPlus
	Add an infinity with another Number object.
*/
#define        InfPlusInt(x, y)      Inf_plusr (x)
#define        InfPlusLong(x, y)     Inf_plusr (x)
#define        InfPlusRational(x, y) Inf_plusr (x)
#define        InfPlusFloat(x, y)    Inf_plusr (x)
#define        InfPlusBig(x, y)      Inf_plusr (x)
extern Number  InfPlusComplex        (Inf x, Complex y);
extern Inf     InfPlusInf            (Inf x, Inf y);
extern Inf     InfPlusInteger        (Inf x, Integer y);
extern Inf     InfPlusDecimal        (Inf x, Decimal y);
extern Inf     InfPlusReal           (Inf x, Real y);
extern Number  InfPlusNumber         (Inf x, Number y);
/*helper*/
extern Inf Inf_plusr (Inf x);


/** InfPow
	Raise an infinity to the power of another Number object.
*/
extern Real   InfPowInt      (Inf x, Int y);
extern Real   InfPowLong     (Inf x, Long y);
extern Number InfPowRational (Inf x, Rational y);
extern Number InfPowFloat    (Inf x, Float y);
extern Number InfPowBig      (Inf x, Big y);
extern Number InfPowComplex  (Inf x, Complex y);
extern Real   InfPowInf      (Inf x, Inf y);
extern Real   InfPowInteger  (Inf x, Integer y);
extern Number InfPowDecimal  (Inf x, Decimal y);
extern Number InfPowReal     (Inf x, Real y);
extern Number InfPowNumber   (Inf x, Number y);


/** InfRem
	Remainder of dividing an infinity with another Number object.

	All throw ERem.
*/
#define InfRemInt(x,y)      Inf_remr ()
#define InfRemFloat(x,y)    Inf_remr ()
#define InfRemLong(x,y)     Inf_remr ()
#define InfRemRational(x,y) Inf_remr ()
#define InfRemBig(x,y)      Inf_remr ()
#define InfRemComplex(x, y) Inf_remr ()
#define InfRemInf(x, y)     Inf_remr ()
#define InfRemInteger(x, y) Inf_remr ()
#define InfRemDecimal(x, y) Inf_remr ()
#define InfRemReal(x, y)    Inf_remr ()
#define InfRemNumber(x, y)  Inf_remr ()
/*helper*/
extern Any Inf_remr ();


/** InfRound
	Rounding functions.

	All the functions throw EMath because it is not possible to round an infinity
	to an Integer.
*/
#define InfFloor(x)       ObjectEMath (x)
#define InfCeil(x)        ObjectEMath (x)
#define InfTrunc(x)       ObjectEMath (x)
#define InfFrac(x)        ObjectEMath (x)
#define InfRound(x, prec) ObjectEMath (x)


/** InfPrint
	Print an infinity to v{stdout}.

	Note that this might throw an exception because writing to v{stdout} may
	fail for various reasons.
*/
extern Null InfPrint (Inf self);


/** InfFormat
	Format an infinity.

	See ={Format} for details.
*/
extern String InfFormat (Inf x, format_s *f);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Float class */
extern void InfSetup ();
extern void InfInit  ();

/* for Sys.c */
extern Inf InfReturn (Any x);

#endif /*_H_Inf_*/
