/* Int.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Int_
#define _H_Int_ 1


/** Int
	This is the final Int class.

	An Int object is a C long minus the first bit disguised as a pointer.
	The range is #{IntMIN} .. #{IntMAX}, where IntMIN is intentionally -IntMAX.

	We dont allocate memory for Int objects. We rely on word aligned memory.
	If an object address has the first bit set then it is an Int, otherwise it
	is another object.
*/
extern Class IntClass;


/** IntMIN IntMAX
	The maximum C long value an Int object can hold.

	Note that the minimum value is intentionally -IntMAX.
	Note also that the result of this macro is a long.

	@warn
	The __len method must never return an integer bigger than IntMAX.
	Meaning, In Qu there must be no object which length is bigger then IntMAX.
*/
#define IntMAX (LONG_MAX >> 1)
#define IntMIN (-IntMAX)


/** IntPredefined
	Some predefined Int objects.

	See ={IntNew} to create your own.
*/
#define Int16 IntNew (16)
#define Int10 IntNew (10)
#define Int9  IntNew (9)
#define Int8  IntNew (8)
#define Int7  IntNew (7)
#define Int6  IntNew (6)
#define Int5  IntNew (5)
#define Int4  IntNew (4)
#define Int3  IntNew (3)
#define Int2  IntNew (2)
#define Int1  IntNew (1)
#define Int0  IntNew (0)
#define Int_1 IntNew (-1)
#define Int_2 IntNew (-2)
#define Int_3 IntNew (-3)
#define Int_4 IntNew (-4)
#define Int_5 IntNew (-5)
#define Int_6 IntNew (-6)
#define Int_7 IntNew (-7)
#define Int_8 IntNew (-8)
#define Int_9 IntNew (-9)
#define Int_10 IntNew (-10)
#define Int_16 IntNew (-16)


/** IntNew
	Create a new Int object from various C types or other objects.

	When using IntNew, make sure the C long value is in the range #{IntMIN} .. #{IntMAX}
	otherwise you loose the value. No checks are made!

	IntNewli does the exact same thing as IntNew. It is defined for the sake of consistent
	naming convention.

	IntNewh and IntNewuh never fail because the size of a short is always smaller
	than the size of a I{long minus one bit}.

	The others return NULL and throw EValue if the value fall out of the Int range.

	@warn
	IntNew is a macro with no side effect by itself.
	You must never change this behavior.
*/
#define    IntNew(l)   ((Int) (((long) (l) << 1) | 1))

#define    IntNewh(v)  IntNewl  (v)
#define    IntNewuh(v) IntNewul (v)
#define    IntNewi(v)  IntNewl  (v)
#define    IntNewui(v) IntNewul (v)
#define    IntNewl(v)  IntNew   (v)
extern Int IntNewul    (unsigned long v);
#define    IntNewli(v) IntNew (v)
extern Int IntNewd     (double v);
#ifdef HAVE_LONG_LONG
extern Int IntNewll    (long long v);
extern Int IntNewull   (unsigned long long v);
#endif
#ifdef HAVE_LONG_DOUBLE
extern Int IntNewld    (long double v);
#endif

#define    IntNewInt(x)      (x)
#define    IntNewFloat(x)    FloatToInt (x)
#define    IntNewLong(x)     LongToInt (x)
#define    IntNewRational(x) RationalToInt (x)
#define    IntNewBig(x)      BigToInt (x)
#define    IntNewComplex(x)  ComplexToInt (x)
#define    IntNewInteger(x)  IntegerToInt (x)
#define    IntNewDecimal(x)  DecimalToInt (x)
#define    IntNewReal(x)     RealToInt (x)
#define    IntNewNumber(x)   NumberToInt (x)
#define    IntNewBoolean(x)  (((x) == True) ? Int1 : Int0)
extern Int IntNewString      (String x);


/** IntTo
	Conversion to C types and other objects.

	For C type conversions the functions set v{errno} to zero on success, to EDOM
	if conversion is not possible, to ERANGE if the value does not fit in the type
	range. The I{unsigned} versions fail with ERANGE if the value is negative.

	Like C, if ERANGE is set then a minimum or maximum possible value for the
	corresponding type is returned, e.g. USHORT_MAX, INT_MIN, etc.

	For object conversions, all functions never fail because it is always possible
	to convert an Int into other Number objects.

	@warn
	IntTol is a function or a macro with no side effect by itself.
	You must never change this behavior.
*/
#ifdef RSHIFT_PRESERVES_SIGNBIT
#  define IntTol(x) (((long) (x)) >> 1)
#else
   extern long Int_Tol (long x);
#  define IntTol(x) Int_Tol ((long) (x))
#endif

#define              IntToh(x)   Number_ltoh   (IntTol  (x))
#define              IntTouh(x)  Number_ultouh (IntToul (x))
#define              IntToi(x)   Number_ltoi   (IntTol  (x))
#define              IntToui(x)  Number_ultouh (IntToul (x))
#define              IntToli(x)  (IntTol (x))
extern unsigned long IntToul     (Int x);
#define              IntTof(x)   ((float) IntTol (x))
#define              IntTod(x)   ((double) IntTol (x))
#ifdef HAVE_LONG_LONG
#define              IntToll(x)  ((long long) IntTol (x))
#define              IntToull(x) ((unsigned long long) IntTol (x))
#endif
#ifdef HAVE_LONG_DOUBLE
#define              IntTold(x)  ((long double) IntTol (x))
#endif

#define       IntToInt(x)      (x)
#define       IntToLong(x)     LongNewInt (x)
#define       IntToRational(x) RationalNewInt (x)
#define       IntToFloat(x)    FloatNewInt (x)
#define       IntToBig(x)      BigNewInt (x)
#define       IntToComplex(x)  ComplexNewInt (x)
#define       IntToInteger(x)  ((Integer) (x))
#define       IntToReal(x)     ((Real) (x))
#define       IntToDecimal(x)  ((Decimal) IntToFloat (x))
#define       IntToNumber(x)   ((Number) (x))
extern String IntToString      (Int x);
#define       IntToBoolean(x)  (IntTol (x) ? True : False)
extern Range  IntToRange       (Int x);


/** IntSign
	IntSign returns -1, 0, 1 if the value is negative, 0, positive respectively.
*/
extern int IntSign (Int x);



/** IntIs
	Identity functions.

	IntIs tests whether or not an object is an Int.

	IntIsNeg returns 1 if the value is negative.
	Returns 0 otherwise.

	IntIsZero returns 1 if the value is 0.
	Returns 0 otherwise.

	IntIsOne returns 1 if the value is 1 or -1.
	Returns 0 otherwise.

	IntIs1 returns 1 if the value is 1.
	Returns 0 otherwise.

	IntIsWhole returns 1, obviously.

	InfIsFrac returns 0, obviously.

	IntIsOdd returns 1 if the number is odd.
	Returns 0 otherwise.
*/
#define    IntIs(x)      ((unsigned long) (x) & 1)
#define    IntIsZero(x)  ((x) == Int0)
extern int IntIsOne      (Int x);
#define    IntIs1(x)     ((x) == Int1)
#define    IntIsNeg(x)   (IntTol (x) < 0)
#define    IntIsWhole(x) (1)
#define    IntIsFrac(x)  (0)
#define    IntIsOdd(x)   (IntTol (x) & 1)


/** IntUnary
	Unary Int functions.

	IntPos returns the number itself, obviously.

	IntNeg returns a negated value.
	The result is an Int.

	IntIncr returns the Int incremented by 1.
	The result is either an Int or a Long.

	IntDecr returns the Int decremented by 1.
	The result is either an Int or a Long.

	IntInot returns a bit-not (inverted) value.
	The result is an Int.
*/
#define        IntPos(x) (x)
extern Int     IntNeg    (Int self);
extern Integer IntIncr   (Int self);
extern Integer IntDecr   (Int self);
extern Int     IntInot   (Int self);


/** IntEq
	Whether the value of an Int is the same as the value of another Number object.
	All return 0 or 1.
*/
#define    IntEqInt(x, y)      ((x) == (y))
#define    IntEqLong(x, y)     LongEqInt (y, x)
#define    IntEqRational(x, y) RationalEqInt (y, x)
#define    IntEqFloat(x, y)    FloatEqInt (y, x)
#define    IntEqBig(x, y)      BigEqInt (y, x)
#define    IntEqComplex(x, y)  ComplexEqInt (y, x)
#define    IntEqInf(x, y)      (0)
extern int IntEqInteger        (Int x, Integer y);
extern int IntEqDecimal        (Int x, Decimal y);
extern int IntEqReal           (Int x, Real y);
extern int IntEqNumber         (Int x, Number y);


/** IntLike
	Whether the absolute value of an Int is the same as the absolute value of another.
	Number object.
	All return 0 or 1.
*/
#define    IntLikeInt(x, y)      (IntCmpAbsInt (x, y) == 0)
#define    IntLikeLong(x, y)     LongLikeInt (y, x)
#define    IntLikeRational(x, y) RationalLikeInt (y, x)
#define    IntLikeFloat(x, y)    FloatLikeInt (y, x)
#define    IntLikeBig(x, y)      BigLikeInt (y, x)
#define    IntLikeComplex(x, y)  ComplexLikeInt (y, x)
#define    IntLikeInf(x, y)      (0)
extern int IntLikeInteger        (Int x, Integer y);
extern int IntLikeDecimal        (Int x, Decimal y);
extern int IntLikeReal           (Int x, Real y);
extern int IntLikeNumber         (Int x, Number y);


/** IntCmp
	Compare an Int with another Number object.
	All return -1, 0, or 1 in the usual manner.
*/
extern int IntCmpInt            (Int x, Int y);
#define    IntCmpLong(x, y)     (-LongCmpInt (y, x))
#define    IntCmpRational(x, y) (-RationalCmpInt (y, x))
#define    IntCmpFloat(x, y)    (-FloatCmpInt (y, x))
#define    IntCmpBig(x, y)      (-BigCmpInt (y, x))
#define    IntCmpComplex(x, y)  (-ComplexCmpInt (y, x))
#define    IntCmpInf(x, y)      (-InfCmpInt (y, x))
extern int IntCmpInteger        (Int x, Integer y);
extern int IntCmpDecimal        (Int x, Decimal y);
extern int IntCmpReal           (Int x, Real y);
extern int IntCmpNumber         (Int x, Number y);


/** IntCmpAbs
	Compare the absolute value of an Int with the absolute value of another.
	Number object.
	All return -1, 0, or 1 in the usual manner.
*/
extern int IntCmpAbsInt            (Int x, Int y);
#define    IntCmpAbsLong(x, y)     (-LongCmpAbsInt (y, x))
#define    IntCmpAbsRational(x, y) (-RationalCmpAbsInt (y, x))
#define    IntCmpAbsFloat(x, y)    (-FloatCmpAbsInt (y, x))
#define    IntCmpAbsBig(x, y)      (-BigCmpAbsInt (y, x))
#define    IntCmpAbsComplex(x, y)  (-ComplexCmpAbsInt (y, x))
#define    IntCmpAbsInf(x, y)      (-1)
extern int IntCmpAbsInteger        (Int x, Integer y);
extern int IntCmpAbsDecimal        (Int x, Decimal y);
extern int IntCmpAbsReal           (Int x, Real y);
extern int IntCmpAbsNumber         (Int x, Number y);


/** IntPlus
	Add an Int with another Number object.

	Note that IntPlusInt and IntPlusFloat always succeed.
	IntPlusInt returns an Int or a ={Long}.
	IntPlusFloat returns a ={Float} or a ={Big}.
*/
extern Integer IntPlusInt            (Int x, Int y);
#define        IntPlusLong(x, y)     LongPlusInt (y, x)
#define        IntPlusRational(x, y) RationalPlusInt (y, x)
#define        IntPlusFloat(x, y)    FloatPlusInt (y, x)
#define        IntPlusBig(x, y)      BigPlusInt (y, x)
#define        IntPlusComplex(x, y)  ComplexPlusInt (y, x)
#define        IntPlusInf(x, y)      InfPlusInt (y, x)
extern Real    IntPlusInteger        (Int x, Integer y);
extern Real    IntPlusDecimal        (Int x, Decimal y);
extern Real    IntPlusReal           (Int x, Real y);
extern Number  IntPlusNumber         (Int x, Number y);


/** IntMinus
	Subtract an Int with another.

	Note that IntMinusInt and IntMinusFloat always succeed.
	IntMinusInt returns an Int or a ={Long}.
	IntMinusFloat returns a ={Float} or a ={Big}.
*/
extern Integer IntMinusInt            (Int x, Int y);
#define        IntMinusLong(x, y)     LongRminusInt (y, x)
#define        IntMinusRational(x, y) RationalRminusInt (y, x)
#define        IntMinusFloat(x, y)    FloatRminusInt (y, x)
#define        IntMinusBig(x, y)      BigRminusInt (y, x)
#define        IntMinusComplex(x, y)  ComplexRminusInt (y, x)
#define        IntMinusInf(x, y)      InfPlusInt (InfNeg (y), x)
extern Real    IntMinusInteger        (Int x, Integer y);
extern Real    IntMinusDecimal        (Int x, Decimal y);
extern Real    IntMinusReal           (Int x, Real y);
extern Number  IntMinusNumber         (Int x, Number y);


/** IntMul
	Multiply an Int with another Number object.

	Note that IntMulInt and IntMulFloat always succeed.
	IntMulInt returns an Int or ={Long}.
	IntMulFloat returns a ={Float} or a ={Big}.
*/
extern Integer IntMulInt            (Int x, Int y);
#define        IntMulLong(x, y)     LongMulInt (y, x)
#define        IntMulRational(x, y) RationalMulInt (y, x)
#define        IntMulFloat(x, y)    FloatMulInt (y, x)
#define        IntMulBig(x, y)      BigMulInt (y, x)
#define        IntMulComplex(x, y)  ComplexMulInt (y, x)
#define        IntMulInf(x, y)      InfMulInt (y, x)
extern Real    IntMulInteger        (Int x, Integer y);
extern Real    IntMulDecimal        (Int x, Decimal y);
extern Real    IntMulReal           (Int x, Real y);
extern Number  IntMulNumber         (Int x, Number y);


/** IntDiv
	Divide an Int with another Number object.
*/
extern Real    IntDivInt            (Int x, Int y);
#define        IntDivLong(x, y)     LongRdivInt (y, x)
#define        IntDivRational(x, y) RationalRdivInt (y, x)
#define        IntDivFloat(x, y)    FloatRdivInt (y, x)
#define        IntDivBig(x, y)      BigRdivInt (y, x)
#define        IntDivComplex(x, y)  ComplexRdivInt (y, x)
extern Int     IntDivInf            (Int x, Inf y);
extern Real    IntDivInteger        (Int x, Integer y);
extern Real    IntDivDecimal        (Int x, Decimal y);
extern Real    IntDivReal           (Int x, Real y);
extern Number  IntDivNumber         (Int x, Number y);


/** IntIdiv
	Integer division of an Int with another Number object.
*/
extern Int     IntIdivInt            (Int x, Int y);
#define        IntIdivLong(x, y)     LongRidivInt (y, x)
#define        IntIdivRational(x, y) RationalRidivInt (y, x)
#define        IntIdivFloat(x, y)    FloatRidivInt (y, x)
#define        IntIdivBig(x, y)      BigRidivInt (y, x)
#define        IntIdivComplex(x, y)  ComplexRidivInt (y, x)
#define        IntIdivInteger(x, y)  IntegerRidivInt (y, x)
extern Int     IntIdivInf            (Int x, Inf y);
extern Integer IntIdivInteger        (Int x, Integer y);
extern Integer IntIdivDecimal        (Int x, Decimal y);
extern Integer IntIdivReal           (Int x, Real y);
extern Integer IntIdivNumber         (Int x, Number y);


/** IntRem
	Remainder of dividing an Int with another Number object.
*/
extern Int     IntRemInt            (Int x, Int y);
#define        IntRemLong(x, y)     LongRremInt (y, x)
#define        IntRemRational(x, y) RationalRremInt (y, x)
#define        IntRemFloat(x, y)    FloatRremInt (y, x)
#define        IntRemBig(x, y)      BigRremInt (y, x)
#define        IntRemComplex(x, y)  ComplexRremInt (y, x)
extern Int     IntRemInf            (Int x, Inf y);
extern Integer IntRemInteger        (Int x, Integer y);
extern Real    IntRemDecimal        (Int x, Decimal y);
extern Real    IntRemReal           (Int x, Real y);
extern Number  IntRemNumber         (Int x, Number y);


/** IntPow
	Raise an Int to the power of another Number object.
*/
extern Real   IntPowInt            (Int x, Int y);
#define       IntPowLong(x, y)     LongRpowInt (y, x)
#define       IntPowRational(x, y) RationalRpowInt (y, x)
#define       IntPowFloat(x, y)    FloatRpowInt (y, x)
#define       IntPowBig(x, y)      BigRpowInt (y, x)
#define       IntPowComplex(x, y)  ComplexRpowInt (y, x)
extern Real   IntPowInf            (Int x, Inf y);
extern Number IntPowInteger        (Int x, Integer y);
extern Number IntPowDecimal        (Int x, Decimal y);
extern Number IntPowReal           (Int x, Real y);
extern Number IntPowNumber         (Int x, Number y);


/** IntIand
	Bit-and an Int with another ={Integer} number.
*/
extern Int IntIandInt        (Int x, Int y);
#define    IntIandLong(x, y) LongIandInt (y, x)


/** IntIor
	Bit-or an Int with another ={Integer} number.
*/
extern Int IntIorInt        (Int x, Int y);
#define    IntIorLong(x, y) LongIorInt (y, x)


/** IntIxor
	Bit-xor an Int with another ={Integer} number.
*/
extern Int IntIxorInt        (Int x, Int y);
#define    IntIxorLong(x, y) LongIxorInt (y, x)


/** IntLshift
	Left-shift an Int by the amount of another ={Integer} number.
*/
extern Real IntLshiftInt        (Int x, Int y);
#define     IntLshiftLong(x, y) LongRlshiftInt (y, x)


/** IntRshift
	Right-shift an Int by the amount of another ={Integer} number.
*/
extern Int IntRshiftInt  (Int x, Int y);
extern Int IntRshiftLong (Int x, Long y);


/** IntPrint
	Print an Int to v{stdout}.

	Note that this might throw an exception because writing to v{stdout} may
	fail for various reasons.
*/
extern Null IntPrint (Int self);


/** IntFormat
	Format an Int.

	See ={Format} for details.
*/
extern String IntFormat (Int x, format_s *f);


/** IntScan
	Scan an Int number.

	v{s} is the start of a nil terminated string.

	On success an Int object is returned and v{s} is updated to the next cursor
	position after the number.

	On failure NULL is returned.
	v{errno} is set to ERANGE if the number is an integer but does not fit in the Int
	range, to EDOM if the number is not an integer number.
*/
extern Int IntScan       (char **s, int neg);
extern Int IntScanHex    (char **s, int neg);
extern Int IntScanOct    (char **s, int neg);
extern Int IntScanBinary (char **s, int neg);


/** IntRound
	Rounding functions.

	These are only required by some macros. You should have no reason to call
	them since the result is always the same as the argument.
*/
#define IntCeil(x)        (x)
#define IntFloor(x)       (x)
#define IntTrunc(x)       (x)
#define IntRound(x, prec) (x)


/** IntFrac
	Return the fraction part, which is always Int (0).
*/
#define IntFrac(x) Int0


/** IntBin IntOct IntHex
	Return binary, octal and hexidecimal representation of the Int.
*/
extern String IntBin (Int x);
extern String IntOct (Int x);
extern String IntHex (Int x);


/** IntHash
	Return the hash value of an Int.
*/
#define IntHash(x) (x)


/** IntDigitSum
	Return the sum of the digits.
*/
extern Int IntDigitSum (Int x);


/** IntCMP
	IntCMP compare two C values, return Int(1), Int(0) or Int(-1).
	This is useful for __cmp methods. Notice that the macro has side effect.

	IntCMPVAL does the same for an int value.
*/
#define IntCMP(u, v) (((u) > (v)) ? Int1  : (((u) < (v)) ? Int_1 : Int0))

inline static Int IntCMPVAL (int x)
{
	return (x > 0) ? Int1 : ((x < 0) ? Int_1 : Int0);
}


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Int class */
extern void IntSetup ();
extern void IntInit  ();

#endif /*_H_Int_ */
