/* Integer.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Integer_
#define _H_Integer_ 1


/** Integer
	This is the abstract parent class of ={Int} and ={Long}.
*/
extern Class IntegerClass;


/** IntegerPredefined
	Predefined system objects.
*/
#define Integer0  ((Integer) Int0)   /*  0 */
#define Integer1  ((Integer) Int1)   /*  1 */
#define Integer2  ((Integer) Int2)   /*  2 */
#define Integer_1 ((Integer) Int_1)  /* -1 */
#define Integer_2 ((Integer) Int_2)  /* -2 */


/** IntegerNew
	Functions to convert C types or othe objects into an ={Integer} object.

	For C type conversions, all functions always succeed and the returned
	value is either an ={Int} or ={Long}.

	For object conversions, an exception is thrown and NULL is returned on failure.
*/
#define        IntegerNewh(v)  IntegerNewl  (v)
#define        IntegerNewuh(v) IntegerNewul (v)
#define        IntegerNewi(v)  IntegerNewl  (v)
#define        IntegerNewui(v) IntegerNewul (v)
#define        IntegerNewli(v) IntegerNewl  (v)
extern Integer IntegerNewl     (long v);
extern Integer IntegerNewul    (unsigned long v);
extern Integer IntegerNewd     (double v);
#if HAVE_LONG_LONG
extern Integer IntegerNewll    (long long v);
extern Integer IntegerNewull   (unsigned long long v);
#endif
#ifdef HAVE_LONG_DOUBLE
extern Integer IntegerNewld    (long double v);
#endif

#define IntegerNewInt(x)      IntToInteger (x)
#define IntegerNewLong(x)     FloatToInteger (x)
#define IntegerNewRational(x) RationalToInteger (x)
#define IntegerNewFloat(x)    FloatToInteger (x)
#define IntegerNewBig(x)      BigToInteger (x)
#define IntegerNewComplex(x)  ComplexToInteger (x)
#define IntegerNewInteger(x)  (x)
#define IntegerNewDecimal(x)  DecimalToInteger (x)
#define IntegerNewReal(x)     RealToInteger (x)
#define IntegerNewNumber(x)   NumberToInteger (x)


/** IntegerTo
	Functions to convert any object into C types or other Integer objects.

	For C type conversions, the functions set v{errno} to zero on success.
	On failure, v{errno} is set to EDOM if v{x} is conversion is not possible,
	to ERANGE if the value falls out of the C type range.
	The I{unsigned} versions fail with ERANGE if the value is negative.

	For object conversions, an exception is thrown and NULL is returned on failure.
*/
#define                   IntegerTob(x)    Number_ltob     (IntegerTol  (x))
#define                   IntegerToub(x)   Number_ultoub   (IntegerToul (x))
#define                   IntegerToh(x)    Number_ltoh     (IntegerTol  (x))
#define                   IntegerTouh(x)   Number_ultouh   (IntegerToul (x))
#define                   IntegerToi(x)    Number_ltoi     (IntegerTol  (x))
#define                   IntegerToui(x)   Number_ultoui   (IntegerToul (x))
#define                   IntegerToi16(x)  Number_ltoi16   (IntegerTol  (x))
#define                   IntegerToui16(x) Number_ultoui16 (IntegerToul (x))
#define                   IntegerToi32(x)  Number_ltoi32   (IntegerTol  (x))
#define                   IntegerToui32(x) Number_ultoui32 (IntegerToul (x))
#define                   IntegerToli(x)   Number_ltoli    (IntegerTol  (x))
#define                   IntegerTouli(x)  Number_ltouli   (IntegerTol  (x))
extern long               IntegerTol      (Any x);
extern unsigned long      IntegerToul     (Any x);
extern float              IntegerTof      (Any x);
extern double             IntegerTod      (Any x);
#ifdef HAVE_LONG_LONG
extern long long          IntegerToll     (Any x);
extern unsigned long long IntegerToull    (Any x);
#endif
#ifdef HAVE_LONG_DOUBLE
extern long double        IntegerTold     (Any x);
#endif

extern Int      IntegerToInt         (Integer x);
extern Long     IntegerToLong        (Integer x);
#define         IntegerToRational(x) RationalNew1 (x)
extern Float    IntegerToFloat       (Integer x);
extern Big      IntegerToBig         (Integer x);
#define         IntegerToComplex(x)  ComplexNew1 ((Real) (x))
#define         IntegerToInteger(x)  (x)
#define         IntegerToReal(x)     ((Real) (x))
extern Decimal  IntegerToDecimal     (Integer x);
#define         IntegerToNumber(x)   ((Number) (x))
extern String   IntegerToString      (Integer x);
extern Boolean  IntegerToBoolean     (Integer x);


/** IntegerSign
	IntegerSign returns -1, 0, 1 if the value is negative, 0, positive respectively.
*/
extern int IntegerSign (Integer x);


/** IntegerIs
	Identity functions.

	IntegerIs checks whether an object is an Integer.

	IntegerIsNeg returns 1 if the value is negative.
	Returns 0 otherwise.

	IntegerIsZero returns 1 if the value is 0.
	Returns 0 otherwise.

	IntegerIsOne returns 1 if the value is 1 or -1.
	Returns 0 oherwise.

	IntegerIsWhole always return 1, obviosuly.

	IntegerIsFrac always return 0, obviosuly.

	IntegerIsOdd return 1 if the value is an odd number.
	Returns 0 otherwise.
*/
#define IntegerIs(x) (IntIs (x) || LongIs (x))
extern int IntegerIsZero  (Integer x);
extern int IntegerIsNeg   (Integer x);
extern int IntegerIsOne   (Integer x);
extern int IntegerIs1     (Integer x);
extern int IntegerIsWhole (Integer x);
extern int IntegerIsFrac  (Integer x);
extern int IntegerIsOdd   (Integer x);


/** IntegerUnary
	Unary functions.

	IntegerPos returns the number itself, obviously.

	IntegerNeg returns a negated value.
	The result is an Int.

	IntegerIncr returns the Integer incremented by 1.
	On success, the result is an Int, a Long or ={InfPositive}.
	EBig is thrown if the ={NumberContext} does not allow infinities.

	IntegerDecr returns the Integer decremented by 1.
	On success, the result is an Int, a Long or ={InfNegative}.
	EBig is thrown if the ={NumberContext} does not allow infinities.

	IntegerInot returns a bit-not (inverted) value.
	The result is either an Int or a Long and always succeed.

	These are just convenient wrappers around
	={IntPos} ={IntNeg} ={IntIncr} ={IntDecr} ={IntInot}
	={LongPos} ={LongNeg} ={LongIncr} ={LongDecr} ={LongInot}.
*/
extern Integer IntegerPos  (Integer x);
extern Integer IntegerNeg  (Integer x);
extern Real    IntegerIncr (Integer x);
extern Real    IntegerDecr (Integer x);
extern Integer IntegerInot (Integer x);


/** IntegerEq
	Whether the value of an Integer is the same as the value of another Number object.
	All return 0 or 1.

	These are just convenient wrappers around ={InttEq} and ={LongEq}.
*/
extern int IntegerEqInt      (Integer x, Int y);
extern int IntegerEqLong     (Integer x, Long y);
extern int IntegerEqRational (Integer x, Rational y);
extern int IntegerEqFloat    (Integer x, Float y);
extern int IntegerEqBig      (Integer x, Big y);
extern int IntegerEqComplex  (Integer x, Complex y);
extern int IntegerEqInf      (Integer x, Inf y);
extern int IntegerEqInteger  (Integer x, Integer y);
extern int IntegerEqDecimal  (Integer x, Decimal y);
extern int IntegerEqReal     (Integer x, Real y);
extern int IntegerEqNumber   (Integer x, Number y);


/** IntegerLike
	Whether the absolute value of an Integer is the same as the absolute value of another
	Number object.
	All return 0 or 1.

	These are just convenient wrappers around ={IntLike} and ={LongLike}.
*/
extern int IntegerLikeInt      (Integer x, Int y);
extern int IntegerLikeLong     (Integer x, Long y);
extern int IntegerLikeRational (Integer x, Rational y);
extern int IntegerLikeFloat    (Integer x, Float y);
extern int IntegerLikeBig      (Integer x, Big y);
extern int IntegerLikeComplex  (Integer x, Complex y);
extern int IntegerLikeInf      (Integer x, Inf y);
extern int IntegerLikeInteger  (Integer x, Integer y);
extern int IntegerLikeDecimal  (Integer x, Decimal y);
extern int IntegerLikeReal     (Integer x, Real y);
extern int IntegerLikeNumber   (Integer x, Number y);


/** IntegerCmp
	Compare an Integer with another Decimal object.
	All return -1, 0, or -1 in the usual manner.

	These are just convenient wrappers around ={IntCmp} and ={LongCmp}.
*/
extern int IntegerCmpInt      (Integer x, Int y);
extern int IntegerCmpLong     (Integer x, Long y);
extern int IntegerCmpRational (Integer x, Rational y);
extern int IntegerCmpFloat    (Integer x, Float y);
extern int IntegerCmpBig      (Integer x, Big y);
extern int IntegerCmpComplex  (Integer x, Complex y);
extern int IntegerCmpInf      (Integer x, Inf y);
extern int IntegerCmpInteger  (Integer x, Integer y);
extern int IntegerCmpDecimal  (Integer x, Decimal y);
extern int IntegerCmpReal     (Integer x, Real y);
extern int IntegerCmpNumber   (Integer x, Number y);


/** IntegerCmpAbs
	Compare the absolute value of an Integer with the absolute value of another
	Number object.
	All return -1, 0, or -1 in the usual manner.

	These are just convenient wrappers around ={IntCmpAbs} and ={LongCmpAbs}.
*/
extern int IntegerCmpAbsInt      (Integer x, Int y);
extern int IntegerCmpAbsLong     (Integer x, Long y);
extern int IntegerCmpAbsRational (Integer x, Rational y);
extern int IntegerCmpAbsFloat    (Integer x, Float y);
extern int IntegerCmpAbsBig      (Integer x, Big y);
extern int IntegerCmpAbsComplex  (Integer x, Complex y);
extern int IntegerCmpAbsInf      (Integer x, Inf y);
extern int IntegerCmpAbsInteger  (Integer x, Integer y);
extern int IntegerCmpAbsDecimal  (Integer x, Decimal y);
extern int IntegerCmpAbsReal     (Integer x, Real y);
extern int IntegerCmpAbsNumber   (Integer x, Number y);


/** IntegerPlus
	Add an Integer with another Number object.

	These are just convenient wrappers around ={IntPlus} and ={LongPlus}.
*/
extern Real    IntegerPlusInt      (Integer x, Int y);
extern Real    IntegerPlusLong     (Integer x, Long y);
extern Real    IntegerPlusRational (Integer x, Rational y);
extern Real    IntegerPlusFloat    (Integer x, Float y);
extern Real    IntegerPlusBig      (Integer x, Big y);
extern Complex IntegerPlusComplex  (Integer x, Complex y);
extern Real    IntegerPlusInf      (Integer x, Inf y);
extern Real    IntegerPlusInteger  (Integer x, Integer y);
extern Real    IntegerPlusDecimal  (Integer x, Decimal y);
extern Real    IntegerPlusReal     (Integer x, Real y);
extern Number  IntegerPlusNumber   (Integer x, Number y);


/** IntegerMinus
	Subtract an Integer with another Number object.

	These are just convenient wrappers around ={IntMinus} and ={LongMinus}.
*/
extern Real    IntegerMinusInt      (Integer x, Int y);
extern Real    IntegerMinusLong     (Integer x, Long y);
extern Real    IntegerMinusRational (Integer x, Rational y);
extern Real    IntegerMinusFloat    (Integer x, Float y);
extern Real    IntegerMinusBig      (Integer x, Big y);
extern Complex IntegerMinusComplex  (Integer x, Complex y);
extern Real    IntegerMinusInf      (Integer x, Inf y);
extern Real    IntegerMinusInteger  (Integer x, Integer y);
extern Real    IntegerMinusDecimal  (Integer x, Decimal y);
extern Real    IntegerMinusReal     (Integer x, Real y);
extern Number  IntegerMinusNumber   (Integer x, Number y);


/** IntegerMul
	Multiply an Integer with anothe Number objectr.

	These are just convenient wrappers around ={IntMul} and ={LongMul}.
*/
extern Real    IntegerMulInt      (Integer x, Int y);
extern Real    IntegerMulLong     (Integer x, Long y);
extern Real    IntegerMulRational (Integer x, Rational y);
extern Real    IntegerMulFloat    (Integer x, Float y);
extern Real    IntegerMulBig      (Integer x, Big y);
extern Complex IntegerMulComplex  (Integer x, Complex y);
extern Real    IntegerMulInf      (Integer x, Inf y);
extern Real    IntegerMulInteger  (Integer x, Integer y);
extern Real    IntegerMulDecimal  (Integer x, Decimal y);
extern Real    IntegerMulReal     (Integer x, Real y);
extern Number  IntegerMulNumber   (Integer x, Number y);


/** IntegerDiv
	Divide an Integer with another Number object.

	These are just convenient wrappers around ={IntDiv} and ={LongDiv}.
*/
extern Real    IntegerDivInt      (Integer x, Int y);
extern Real    IntegerDivLong     (Integer x, Long y);
extern Real    IntegerDivRational (Integer x, Rational y);
extern Real    IntegerDivFloat    (Integer x, Float y);
extern Real    IntegerDivBig      (Integer x, Big y);
extern Complex IntegerDivComplex  (Integer x, Complex y);
extern Real    IntegerDivInf      (Integer x, Inf y);
extern Real    IntegerDivInteger  (Integer x, Integer y);
extern Real    IntegerDivDecimal  (Integer x, Decimal y);
extern Real    IntegerDivReal     (Integer x, Real y);
extern Number  IntegerDivNumber   (Integer x, Number y);


/** IntegerIdiv
	Integer division of an Integer with another Number object.

	These are just convenient wrappers around ={IntIdiv} and ={LongIdiv}.
*/
extern Integer IntegerIdivInt      (Integer x, Int y);
extern Integer IntegerIdivLong     (Integer x, Long y);
extern Integer IntegerIdivRational (Integer x, Rational y);
extern Integer IntegerIdivFloat    (Integer x, Float y);
extern Integer IntegerIdivBig      (Integer x, Big y);
extern Integer IntegerIdivComplex  (Integer x, Complex y);
extern Integer IntegerIdivInf      (Integer x, Inf y);
extern Integer IntegerIdivInteger  (Integer x, Integer y);
extern Integer IntegerIdivDecimal  (Integer x, Decimal y);
extern Integer IntegerIdivReal     (Integer x, Real y);
extern Integer IntegerIdivNumber   (Integer x, Number y);


/** IntegerRem
	Remainder of dividing an Integer with another Number object.

	These are just convenient wrappers around ={IntRem} and ={LongRem}.
*/
extern Real    IntegerRemInt      (Integer x, Int y);
extern Real    IntegerRemLong     (Integer x, Long y);
extern Real    IntegerRemRational (Integer x, Rational y);
extern Real    IntegerRemFloat    (Integer x, Float y);
extern Real    IntegerRemBig      (Integer x, Big y);
extern Complex IntegerRemComplex  (Integer x, Complex y);
extern Integer IntegerRemInf      (Integer x, Inf y);
extern Integer IntegerRemInteger  (Integer x, Integer y);
extern Real    IntegerRemDecimal  (Integer x, Decimal y);
extern Real    IntegerRemReal     (Integer x, Real y);
extern Number  IntegerRemNumber   (Integer x, Number y);


/** IntegerPow
	Raise an Integer to the power of another Number object.

	These are just convenient wrappers around ={IntPow} and ={LongPow}.
*/
extern Real    IntegerPowInt      (Integer x, Int y);
extern Real    IntegerPowLong     (Integer x, Long y);
extern Number  IntegerPowRational (Integer x, Rational y);
extern Number  IntegerPowFloat    (Integer x, Float y);
extern Number  IntegerPowBig      (Integer x, Big y);
extern Complex IntegerPowComplex  (Integer x, Complex y);
extern Real    IntegerPowInf      (Integer x, Inf y);
extern Number  IntegerPowInteger  (Integer x, Integer y);
extern Number  IntegerPowDecimal  (Integer x, Decimal y);
extern Number  IntegerPowReal     (Integer x, Real y);
extern Number  IntegerPowNumber   (Integer x, Number y);


/** IntegerIand
	Bit-and an Integer with another Integer object.

	These are just convenient wrappers around ={IntIand} and ={LongIand}.
*/
extern Integer IntegerIandInt     (Integer x, Int y);
extern Integer IntegerIandLong    (Integer x, Long y);
extern Integer IntegerIandInteger (Integer x, Integer y);


/** IntegerIor
	Bit-or an Integer with another Integer object.

	These are just convenient wrappers around ={IntIor} and ={LongIor}.
*/
extern Integer IntegerIorInt     (Integer x, Int y);
extern Integer IntegerIorLong    (Integer x, Long y);
extern Integer IntegerIorInteger (Integer x, Integer y);


/** IntegerIxor
	Bit-xor an Integer with another Integer object.

	These are just convenient wrappers around ={IntIxor} and ={LongIxor}.
*/
extern Integer IntegerIxorInt     (Integer x, Int y);
extern Integer IntegerIxorLong    (Integer x, Long y);
extern Integer IntegerIxorInteger (Integer x, Integer y);


/** IntegerLshift
	Left-shift an Integer by the amount of another Integer object.

	These are just convenient wrappers around ={IntLshift} and ={LongLshift}.
*/
extern Integer IntegerLshiftInt     (Integer x, Int y);
extern Integer IntegerLshiftLong    (Integer x, Long y);
extern Integer IntegerLshiftInteger (Integer x, Integer y);


/** IntegerRshift
	Right-shift an Integer by the amount of another Integer object.

	These are just convenient wrappers around ={IntRshift} and ={LongRshift}.
*/
extern Integer IntegerRshiftInt     (Integer x, Int y);
extern Integer IntegerRshiftLong    (Integer x, Long y);
extern Integer IntegerRshiftInteger (Integer x, Integer y);


/** IntegerPrint
	Print an Integer to v{stdout}.
	Print a Decimal to v{stdout}.

	Note that this might throw an exception because writing to v{stdout} may
	fail for various reasons.

	This is just a convenient wrapper around ={IntPrint} and ={LongPrint}.
*/
extern Null IntegerPrint (Integer x);


/** IntegerFormat
	Format an Integer.

	See ={Format} for details.

	This is just a convenient wrapper around ={IntFormat} and ={LongFormat}.
*/
extern String IntegerFormat (Integer x, format_s *f);


/** IntegerGcd
	Calculate the GCD (Greatest Common Divisor).

	Note that the signs are ignored and the result is a positive Integer.

	@warn
	Calling IntegerGcd with y == 0 throws EDiv.

	You may not call IntegerGcdl with y == 0. No checks are made.
*/
extern long    IntegerGcdl (long x, long y);
extern Integer IntegerGcd  (Integer x, Integer y);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Integer class */
extern void IntegerSetup ();
extern void IntegerInit  ();

#endif /*_H_Integer_*/
