/* Iterator.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Iterator_
#define _H_Iterator_ 1


/** Iterator
	This is the final Iterator class.
*/
extern Class IteratorClass;


/** TIterator
	This represents an instance of the Iterator class.

	v{ex1} and v{ex2} may be used by C code. They are not internally used
	by the Iterator class.

	@warn
	v{pos} may not be NULL because it is used by scripts to create custom
	iterators. If not used then leave it to ={Int0}.
*/
struct TIterator
{
	OBJ_HEAD;
	Any    obj;  /* the object to iterate through */
	void  *met;  /* Sub/Method or C method/function to call for the next value */
	Any    pos;  /* current position, initially Int0 and may not be 0 */
	long   ex1;  /* optional value, initially 0 - may be used as pointer */
	long   ex2;  /* -ditto- */

	/*
		'met' is a iterator_f C method/function address for ITERATOR_EXTERN
		a Sub/Method object otherwise (marked by Iterator)
	*/
};


/** iterator_f
	iterator_f is the C function prototype to be called when the flag
	v{IteratorFlagEXTERN} is set (thus the field v{met} is a pointer to a C function
	instead of a Sub).
*/
typedef Any (*iterator_f) (Any self, Iterator i);


/** IteratorIs
	Whether or not an object is a Iterator.
*/
#define IteratorIs(o) (ClassOf(o) == IteratorClass)


/** IteratorFlag
	Flags used by Iterator objects.

	Do not access these flags directly.
	Use the accessor macros provided instead.
*/
#define IteratorFlagEXTERN    Flag0  /* field "met" is a pointer to a C function? */
#define IteratorFlagKEYS      Flag1  /* iterating through keys? */
#define IteratorFlagVALUES    Flag2  /* iterating through values? */
#define IteratorFlagMUTABLES  Flag3  /* yield frozen or mutable object? */


/** IteratorIsExtern
	Whether the field 'met' is a pointer to a C function.

	See ={ITERATOR_NEW} to setup the IteratorFlagEXTERN flag.

	The IteratorFlagEXTERN flag must only be set by the creator of the Iterator object.
	Once set, it must never be unset.
*/
#define IteratorIsExtern(self)  FlagTest (self, IteratorFlagEXTERN)
#define IteratorSetExtern(self) FlagSet  (self, IteratorFlagEXTERN)


/** IteratorYieldMutables
	Whether the iterator requires to return mutable objects.

	The IteratorFlagMUTABLES flag must only be set by the creator of the Iterator object.
	Once set, it must never be unset.
*/
#define IteratorYieldMutables(self)    FlagTest (self, IteratorFlagMUTABLES)
#define IteratorSetYieldMutables(self) FlagSet  (self, IteratorFlagMUTABLES)


/** IteratorYieldKeys
	Whether the Iterator object iterates through keys.

	The IteratorFlagKEYS flag must only be set by the creator of the Iterator object.
	Once set, it must never be unset.
*/
#define IteratorYieldKeys(self)    FlagTest (self, IteratorFlagKEYS)
#define IteratorSetYieldKeys(self) FlagSet  (self, IteratorFlagKEYS)


/** IteratorYieldValues
	Whether the Iterator object iterates through keys.

	The IteratorFlagVALUES flag must only be set by the creator of the Iterator object.
	Once set, it must never be unset.
*/
#define IteratorYieldValues(self)    FlagTest (self, IteratorFlagVALUES)
#define IteratorSetYieldValues(self) FlagSet  (self, IteratorFlagVALUES)


/** ITERATOR_SETUP ITERATOR_NEW
	Initialize or create a new Iterator object.

	v{i} is a Iterator type variable, e.g. C{Iterator i;}

	v{o} is the object to iterate through.

	v{u} is the Sub to call for ITERATOR_NEW and a #{iterator_f} C function for
	ITERATOR_NEW_EXTERN.

	See Array.c for an example.

	@warn
	i->pos may not be NULL. Set to &{Int#Int0} if not used.
*/
#define ITERATOR_SETUP(i, o, u) \
BEGIN \
	i->obj = (Obj) o; \
	i->pos = Int0; \
	i->met = u; \
	i->ex1 = i->ex2 = 0; \
END

#define ITERATOR_SETUP_EXTERN(i, o, u) \
BEGIN \
	ITERATOR_SETUP (i, o, u); \
	IteratorSetExtern (i); \
END

#define ITERATOR_NEW(i, o, u) \
BEGIN \
	i = GcNew (Iterator); \
	ITERATOR_SETUP (i, o, u); \
END

#define ITERATOR_NEW_EXTERN(i, o, u) \
BEGIN \
	i = GcNew (Iterator); \
	ITERATOR_SETUP_EXTERN (i, o, u); \
END


/** ITERATOR_STOP
	Throw EStop if v{expr} is false.
*/
#define ITERATOR_STOP(expr) REQUIRE (EStop, (expr))


/** IteratorDo
	Iteratorate through an object at C level.

	IteratorDo is the iterator. It takes an object v{x} and a callback function.
	Optionally pass v{data} to the callback.

	IteratorDo knows how to iterate through some builtin classes thus is more
	efficient than creating an Iterator object or performed at program level.

	itercallback_f is a callback function that will be called for each item.
	It must return one of:
	@args
	.ITERATOR_CALLBACK_ERROR     an error occured
	.ITERATOR_CALLBACK_END       no more items or EStop is thrown
	.ITERATOR_CALLBACK_CONTINUE  continue with the next item
	.ITERATOR_CALLBACK_RESULT    stop with a result

	@block
	v{k} and v{v} are set as follow:
	@rcc
	.| B{object} | B{item}       | B{idx}
	.| String    | character     | index
	.| Array     | item          | index
	.| Dict      | key           | value
	.| Set       | item          | index
	.| Frame     | yielded value | index
	.| Range     | item          | index
	.| Iterator  | next item     | index
	.| Record    | field value   | index
	.| Struct    | field name    | field type
	.| Buffer    | character     | index
	.| Flow      | item          | index

	@block
	IteratorDo returns one of:
	@args
	.ITERATOR_CALLBACK_ERROR   an error occured
	.ITERATOR_CALLBACK_END     no more items or EStop is thrown
	.ITERATOR_CALLBACK_RESULT  stop with a result
*/
typedef int (*itercallback_f) (Any item, Any idx, void *data);
extern int IteratorDo (Any x, itercallback_f f, void *data);

#define ITERATOR_CALLBACK_ERROR    0
#define ITERATOR_CALLBACK_END      1
#define ITERATOR_CALLBACK_CONTINUE 2
#define ITERATOR_CALLBACK_RESULT   3


/** IteratorFrom
	Make sure we ge an Iterator, not an Array, Dict, etc.

	The interpreter knows how to iterate through some builtin classes
	and the program level __each method is allowed to return those classes
	(see Sys_each).

	If v{x} does not support __each then EImplement is thrown.
*/
extern Iterator IteratorFrom (Any x);


/** IteratorNext
	Get the next value.

	Throws EStop if there is no more value.
*/
extern Any IteratorNext (Iterator self);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Iterator class. */
extern void IteratorSetup ();
extern void IteratorInit  ();

#endif /*_H_Iterator_*/
