/* Long.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Long_
#define _H_Long_ 1


/** Long
	This is the final Long class.
*/
extern Class LongClass;


/** LongDIGIT LongDIGIT2 LongSDIGIT2 LongSIZE LongFORMAT
	Constants used by the Long class.

	LongDIGIT is the C type used for the digits.

	LongDIGIT2 is twice LongDIGIT unsigned while LongSDIGIT2 is signed.

	LongSIZE is the size of the digit.

	LongFORMAT is the format of a digit. This must match LongSIZE.

	LongMAXLEN is the practical maximum allocatable number of digits.
	It must be such that it is always possible to convert a Long to a Big
	and vice versa. It must also fit in an Int.
	It should be be BigMAXLEN / 4.
*/
#ifdef HAVE_LONG_LONG
#  if SIZEOF_LONG_LONG == (SIZEOF_LONG + SIZEOF_LONG)
#    define LongFORMAT     2
#    define LongSIZE       SIZEOF_LONG
#    define LongSDIGIT     long
#    define LongSDIGIT2    long long
#    define Long2ISLL      1
#  elif SIZEOF_LONG_LONG == (SIZEOF_INT + SIZEOF_INT)
#    define LongFORMAT     1
#    define LongSIZE       SIZEOF_INT
#    define LongSDIGIT     int
#    define LongSDIGIT2    long long
#    define Long2ISLL      1
#  else
#    define LongFORMAT     0
#    define LongSIZE       SIZEOF_SHORT
#    define LongSDIGIT     short
#    define LongSDIGIT2    long
#    undef Long2ISLL
#  endif
#else
#    define LongFORMAT     0
#    define LongSIZE       SIZEOF_SHORT
#    define LongSDIGIT     short
#    define LongSDIGIT2    long
#    undef Long2ISLL
#endif

#define LongDIGIT  unsigned LongSDIGIT
#define LongDIGIT2 unsigned LongSDIGIT2

#define LongMAXLEN (INT_MAX / LongSIZE / 16)


/** TLong
	This represents an instance of the Long class.
*/
struct TLong
{
	OBJ_HEAD;
	LongDIGIT *digits; /* the digits, LSB first */
	long       len;    /* the number of digits */
};


/** LongPredefined
	Predefined system objects.

	See ={LongStatic} to create your own.
*/
extern Long Long0;   /* 0 */
extern Long Long1;   /* 1 */
extern Long Long2;   /* 2 */
extern Long Long_1;  /* -1 */
extern Long Long_2;  /* -2 */
extern Long Long10;  /* 10 */


/** LongFlag
	Flags used by Long objects.

	Do not access these flags directly.
	Use the accessor macros provided instead.
	If none is defined then it means you must not touch the flag.
*/
#define LongFlagNEG Flag0  /* negative? = BigFlagNEG */
#define LongFlagTMP Flag9  /* on local C stack? = BigFlagTMP */


/** LongNew
	Create a new Long object from various C types or other objects.

	@warn
	You may not pass a NaN or infinity value to LongNewd and LongNewld.
*/
#define     LongNewh(v)  LongNewl  (v)
#define     LongNewuh(v) LongNewul (v)
#define     LongNewi(v)  LongNewl  (v)
#define     LongNewui(v) LongNewul (v)
extern Long LongNewl     (long v);
extern Long LongNewul    (unsigned long v);
#define     LongNewli(v) LongNewl (v)
extern Long LongNewd     (double v);
#ifdef HAVE_LONG_LONG
extern Long LongNewll    (long long v);
extern Long LongNewull   (unsigned long long v);
#endif
#ifdef HAVE_LONG_DOUBLE
extern Long LongNewld    (long double v);
#endif

#define     LongNewInt(x)      LongNewl (IntTol (x))
#define     LongNewLong(x)     (x)
#define     LongNewFloat(x)    LongNewd ((x)->value)
#define     LongNewRational(x) RationalToLong (x)
#define     LongNewBig(x)      BigToLong (x)
#define     LongNewComplex(x)  ComplexToLong (x)
#define     LongNewInteger(x)  IntegerToLong (x)
#define     LongNewDecimal(x)  DecimalToLong (x)
#define     LongNewReal(x)     RealToLong (x)
#define     LongNewNumber(x)   NumberToLong (x)
#define     LongNewBoolean(x)  ((x) == True ? Long1 : Long0)
extern Long LongNewString      (String s);


/** LongStatic
	Create a static Long object (not in GC).
*/
extern Long LongStatic   (const char *s);
extern Long LongStatici  (int v);
extern Long LongStaticui (unsigned int v);
extern Long LongStaticl  (long v);
extern Long LongStaticul (unsigned long v);
extern Long LongToStatic (Long x);



/** LongTo
	Convert a Long to various C types or other objects.

	LongTod returns infinity in case it does not fit in a double so you must
	always check the result before you use it to create Float objects.

	The others C type conversions set v{errno} to 0 on success, to ERANGE on failure
	and return a minimum or maximum value e.g. ULONG_MAX, LONG_LONG_MIN, etc.

	For object conversions, an exception is raised and NULL is returned on failure.
	Only LongToInt and LongToFloat can fail. The others always succeed.
*/
#define                   LongToh(x)  Number_ltoh   (LongTol (x))
#define                   LongTouh(x) Number_ultouh (LongToul (x))
#define                   LongToi(x)  Number_ltoi   (LongTol (x))
#define                   LongToui(x) Number_ultouh (LongToul (x))
#define                   LongToli(x) Number_ltoli  (LongTol (x))
extern long               LongTol     (Long x);
extern unsigned long      LongToul    (Long x);
#ifdef HAVE_LONG_LONG
extern long long          LongToll    (Long x);
extern unsigned long long LongToull   (Long x);
#endif
extern float              LongTof     (Long x);
extern double             LongTod     (Long x);
#ifdef HAVE_LONG_DOUBLE
extern long double        LongTold    (Long x);
#endif

extern Int     LongToInt         (Long x);
extern Decimal LongToDecimal     (Long x);
#define        LongToLong(x)     (x)
#define        LongToFloat(x)    FloatNewLong (x)
#define        LongToBig(x)      BigNewLong (x)
#define        LongToRational(x) RationalNewInteger ((Integer) x)
#define        LongToComplex(x)  ComplexNew1 ((Real) x)
#define        LongToInteger(x)  ((Integer) (x))
#define        LongToReal(x)     ((Real) x)
#define        LongToNumber(x)   ((Number) x)
#define        LongToBoolean(x)  (LongIsZero (x) ? False : True)
extern String  LongToString      (Long x);
extern Array   LongToArray       (Long x);


/** LongSign
	LongSign returns -1, 0, or 1 if the number is negative, 0, or positive respectively.
*/
extern int LongSign (Long x);


/** LongIs
	Identity functions.

	LongIs tests Whether or not an object is a Long.

	LongIsNeg return 1 if the number is negative, 0 otherwise.

	LongIsZero return 1 if the number is zero, 0 otherwise.

	LongIsOne return 1 if the number is -1 or 1, 0 otherwise.

	LongIs1 return 1 if the number is 1, 0 otherwise.

	LongIsWhole always return 1, obviously.

	LongIsFrac always return 0, obviously.

	LongIsOdd return 1 if the number is odd, 0 otherwise.
*/
#define    LongIs(x)      (ClassOf(x) == LongClass)
#define    LongIsNeg(x)   FlagTest (x, LongFlagNEG)
#define    LongIsZero(x)  ((x)->len == 0)
#define    LongIsWhole(x) (1)
#define    LongIsFrac(x)  (0)
extern int LongIsOne      (Long x);
extern int LongIs1        (Long x);
extern int LongIsOdd      (Long x);


/** LongUnary
	Unary functions.

	LongPos, obviously, return the Long itself.

	LongNeg returns a negated value.
	The result is a Long and always succeed.

	LongIncr returns the Long incremented by 1.
	On success, the result is an Int, a Long or ={InfPositive}.
	EBig is thrown if the ={NumberContext} does not allow infinities.

	LongDecr returns the Long decremented by 1.
	On success, the result is an Int, a Long or ={InfNegative}.
	EBig is thrown if the ={NumberContext} does not allow infinities.

	LongInot returns a bit-not (inverted) value.
	The result is either an Int or a Long and always succeed.
*/
#define        LongPos(x)  (x)
extern Long    LongNeg     (Long self);
extern Real    LongIncr    (Long self);
extern Real    LongDecr    (Long self);
extern Integer LongInot    (Long self);


/** LongEq
	Whether the value of a Long is the same as the value of another Number object.
	All return 0 or 1.
*/
extern int LongEqInt            (Long x, Int y);
extern int LongEqLong           (Long x, Long y);
#define    LongEqRational(x, y) RationalEqLong (y, x)
#define    LongEqFloat(x, y)    FloatEqLong (y, x)
#define    LongEqBig(x, y)      BigEqLong (y, x)
#define    LongEqComplex(x, y)  ComplexEqLong (y, x)
#define    LongEqInf(x, y)      (0)
extern int LongEqInteger        (Long x, Integer y);
extern int LongEqDecimal        (Long x, Decimal y);
extern int LongEqReal           (Long x, Real y);
extern int LongEqNumber         (Long x, Number y);


/** LongLike
	Whether the absolute value of a Long is the same as the absolute value of another
	Number object.
	All return 0 or 1.
*/
#define    LongLikeInt(x, y)      (LongCmpAbsInt (x, y) == 0)
#define    LongLikeLong(x, y)     (LongCmpAbsLong (x, y) == 0)
#define    LongLikeRational(x, y) RationalLikeLong (y, x)
#define    LongLikeFloat(x, y)    FloatLikeLong (y, x)
#define    LongLikeBig(x, y)      BigLikeLong (y, x)
#define    LongLikeComplex(x, y)  ComplexLikeLong (y, x)
#define    LongLikeInf(x, y)      (0)
extern int LongLikeInteger        (Long x, Integer y);
extern int LongLikeDecimal        (Long x, Decimal y);
extern int LongLikeReal           (Long x, Real y);
extern int LongLikeNumber         (Long x, Number y);


/** LongCmp
	Compare a Long with another Number object.
	All return -1, 0 or 1 in the usual manner.
*/
extern int LongCmpInt            (Long x, Int y);
extern int LongCmpLong           (Long x, Long y);
#define    LongCmpRational(x, y) (-RationalCmpLong (y, x))
#define    LongCmpFloat(x, y)    (-FloatCmpLong (y, x))
#define    LongCmpBig(x, y)      (-BigCmpLong (y, x))
#define    LongCmpComplex(x, y)  (-ComplexCmpLong (y, x))
#define    LongCmpInf(x, y)      (-InfCmpLong (y, x))
extern int LongCmpInteger        (Long x, Integer y);
extern int LongCmpDecimal        (Long x, Decimal y);
extern int LongCmpReal           (Long x, Real y);
extern int LongCmpNumber         (Long x, Number y);


/** LongCmpAbs
	Compare the absolute value of a Long with the absolute value of another
	Number object.
	All return -1, 0 or 1 in the usual manner.
*/
extern int LongCmpAbsInt            (Long x, Int y);
extern int LongCmpAbsLong           (Long x, Long y);
#define    LongCmpAbsRational(x, y) (-RationalCmpAbsLong (y, x))
#define    LongCmpAbsFloat(x, y)    (-FloatCmpAbsLong (y, x))
#define    LongCmpAbsBig(x, y)      (-BigCmpAbsLong (y, x))
#define    LongCmpAbsComplex(x, y)  (-ComplexCmpAbsLong (y, x))
#define    LongCmpAbsInf(x, y)      (-1)
extern int LongCmpAbsInteger        (Long x, Integer y);
extern int LongCmpAbsDecimal        (Long x, Decimal y);
extern int LongCmpAbsReal           (Long x, Real y);
extern int LongCmpAbsNumber         (Long x, Number y);


/** LongPlus
	Add a Long with another Number object.
*/
extern Real   LongPlusInt            (Long x, Int y);
extern Real   LongPlusLong           (Long x, Long y);
#define       LongPlusRational(x, y) RationalPlusLong (y, x)
#define       LongPlusFloat(x, y)    FloatPlusLong (y, x)
#define       LongPlusBig(x, y)      BigPlusLong (y, x)
#define       LongPlusComplex(x, y)  ComplexPlusLong (y, x)
#define       LongPlusInf(x, y)      InfPlusLong (y, x)
extern Real   LongPlusInteger        (Long x, Integer y);
extern Real   LongPlusDecimal        (Long x, Decimal y);
extern Real   LongPlusReal           (Long x, Real y);
extern Number LongPlusNumber         (Long x, Number y);


/** LongMinus
	Subtract a Long with another Number object.
*/
extern Real   LongMinusInt            (Long x, Int y);
extern Real   LongMinusLong           (Long x, Long y);
#define       LongMinusRational(x, y) RationalRminusLong (y, x)
#define       LongMinusFloat(x, y)    FloatRminusLong (y, x)
#define       LongMinusBig(x, y)      BigRminusLong (y, x)
#define       LongMinusComplex(x, y)  ComplexRminusLong (y, x)
#define       LongMinusInf(x, y)      InfPlusLong (InfNeg (y), x)
extern Real   LongMinusInteger        (Long x, Integer y);
extern Real   LongMinusDecimal        (Long x, Decimal y);
extern Real   LongMinusReal           (Long x, Real y);
extern Number LongMinusNumber         (Long x, Number y);


/** LongRminus
	Subtract another with a Long.
	This is only for efficiency.
*/
extern Real LongRminusInt (Long x, Int y);


/** LongMul
	Multiply a Long with another Number object.
*/
extern Real   LongMulInt            (Long x, Int y);
extern Real   LongMulLong           (Long x, Long y);
#define       LongMulRational(x, y) RationalMulLong (y, x)
#define       LongMulFloat(x, y)    FloatMulLong (y, x)
#define       LongMulBig(x, y)      BigMulLong (y, x)
#define       LongMulComplex(x, y)  ComplexMulLong (y, x)
#define       LongMulInf(x, y)      InfMulLong (y, x)
extern Real   LongMulInteger        (Long x, Integer y);
extern Real   LongMulDecimal        (Long x, Decimal y);
extern Real   LongMulReal           (Long x, Real y);
extern Number LongMulNumber         (Long x, Number y);


/** LongDiv
	Divide a Long with another Number object.
*/
extern Real   LongDivInt            (Long x, Int y);
extern Real   LongDivLong           (Long x, Long y);
#define       LongDivRational(x, y) RationalRdivLong (y, x)
#define       LongDivFloat(x, y)    FloatRdivLong (y, x)
#define       LongDivBig(x, y)      BigRdivLong (y, x)
#define       LongDivComplex(x, y)  ComplexRdivLong (y, x)
extern Int    LongDivInf            (Long x, Inf y);
extern Real   LongDivInteger        (Long x, Integer y);
extern Real   LongDivDecimal        (Long x, Decimal y);
extern Real   LongDivReal           (Long x, Real y);
extern Number LongDivNumber         (Long x, Number y);


/** LongRdiv
	Divide another with a Long.
	This is only for efficiency.
*/
extern Real LongRdivInt (Long x, Int y);


/** LongIdiv
	Integer divison of a Long with another Number object.
*/
extern Integer LongIdivInt            (Long x, Int y);
extern Integer LongIdivLong           (Long x, Long y);
#define        LongIdivRational(x, y) RationalRidivLong (y, x)
#define        LongIdivFloat(x, y)    FloatRidivLong (y, x)
#define        LongIdivBig(x, y)      BigRidivLong (y, x)
#define        LongIdivComplex(x, y)  ComplexRidivLong (y, x)
extern Int     LongIdivInf            (Long x, Inf y);
extern Integer LongIdivInteger        (Long x, Integer y);
extern Integer LongIdivDecimal        (Long x, Decimal y);
extern Integer LongIdivReal           (Long x, Real y);
extern Integer LongIdivNumber         (Long x, Number y);


/** LongRidiv
	Integer divison of another Number object with a Long.
	This is only for efficiency.
*/
extern Integer LongRidivInt (Long x, Int y);


/** LongRem
	Remainder of dividing a Long with another.
*/
extern Real    LongRemInt            (Long x, Int y);
extern Real    LongRemLong           (Long x, Long y);
#define        LongRemRational(x, y) RationalRremLong (y, x)
#define        LongRemFloat(x, y)    FloatRremLong (y, x)
#define        LongRemBig(x, y)      BigRremLong (y, x)
#define        LongRemComplex(x, y)  ComplexRremLong (y, x)
extern Integer LongRemInf            (Long x, Inf y);
extern Real    LongRemInteger        (Long x, Integer y);
extern Real    LongRemDecimal        (Long x, Decimal y);
extern Real    LongRemReal           (Long x, Real y);
extern Number  LongRemNumber         (Long x, Number y);


/** LongRrem
	Remainder of dividing a Long with another Number object.
	This is only for efficiency.
*/
extern Real LongRremInt (Long x, Int y);


/** LongPow
	Raise a Long to the power of another Number object.
*/
extern Real   LongPowInt            (Long x, Int y);
extern Real   LongPowLong           (Long x, Long y);
#define       LongPowRational(x, y) RationalRpowLong (y, x)
#define       LongPowFloat(x, y)    FloatRpowLong (y, x)
#define       LongPowBig(x, y)      BigRpowLong (y, x)
#define       LongPowComplex(x, y)  ComplexRpowLong (y, x)
extern Real   LongPowInf            (Long x, Inf y);
extern Real   LongPowInteger        (Long x, Integer y);
extern Real   LongPowDecimal        (Long x, Decimal y);
extern Real   LongPowReal           (Long x, Real y);
extern Number LongPowNumber         (Long x, Number y);


/** LongRpow
	Raise another Number object to the power of a Long.
	This is only for efficiency.
*/
extern Real LongRpowInt (Long x, Int y);


/** LongPow10
	Return a Long which is the result of raising 10 to the specified power.

	@warn
	v{p} must be a non-negative number.
	If the value is huge then NULL might be returned so be careful.
*/
extern Long LongPow10 (int p);


/** LongIand
	Bit-and a Long.
*/
extern Int     LongIandInt     (Long x, Int y);
extern Integer LongIandLong    (Long x, Long y);
extern Integer LongIandInteger (Long x, Integer y);


/** LongRiand
	Bit-and another with a Long.
	This is only for efficiency.
*/
extern Integer LongRiandInt (Long x, Int y);


/** LongIor
	Bit-or a Long.
*/
extern Integer LongIorInt     (Long x, Int y);
extern Integer LongIorLong    (Long x, Long y);
extern Integer LongIorInteger (Long x, Integer y);


/** LongRior
	Bit-or another with a Long.
	This is only for efficiency.
*/
extern Integer LongRiorInt (Long x, Int y);


/** LongIxor
	Bit-xor a Long.
*/
extern Integer LongIxorInt     (Long x, Int y);
extern Integer LongIxorLong    (Long x, Long y);
extern Integer LongIxorInteger (Long x, Integer y);


/** LongRixor
	Bit-xor another with a Long.
	This is only for efficiency.
*/
extern Integer LongRixorInt (Long x, Int y);


/** LongRshift
	Right shift a Long.
*/
extern Integer LongRshiftInt     (Long x, Int y);
extern Integer LongRshiftLong    (Long x, Long y);
extern Integer LongRshiftInteger (Long x, Integer y);


/** LongRrshift
	Right shift another with a Long.
*/
/* no such thing because we don't need it */


/** LongLshift
	Left shift a Long.
*/
extern Real LongLshiftInt     (Long x, Int y);
extern Real LongLshiftLong    (Long x, Long y);
extern Real LongLshiftInteger (Long x, Integer y);


/** LongRlshift
	Left shift another with a Long.
	This is only for efficiency.
*/
extern Real LongRlshiftInt (Long x, Int y);


/** LongRound
	Rounding functions.

	These are only required by some macros. You should have no reason to call
	them since the result is always the same as the argument.
*/
#define LongCeil(x)        (x)
#define LongFloor(x)       (x)
#define LongTrunc(x)       (x)
#define LongRound(x, prec) (x)


/** LongFrac
	Return the fraction part, which is always Long (0).
*/
#define LongFrac(x) Long0


/** LongPrint
	Print a Long to v{stdout}.

	Note that this might throw an exception because writing to v{stdout} may
	fail for various reasons.
*/
extern Null LongPrint (Long self);


/** LongFormat
	Format a Long.

	See ={Format} for details.
*/
extern String LongFormat (Long self, format_s *f);


/** LongScan
	Scan an Long number.

	v{s} is the start of a nil terminated string.

	On success, a Long or ={Int} object is returned and v{s} is updated to the next
	cursor position after the number.

	On failure NULL is returned.

	LongScanStr scans a base-10 integer and throws ESyntax on failure.
*/
extern Long LongScan       (char **s, int neg);
extern Long LongScanHex    (char **s, int neg);
extern Long LongScanOct    (char **s, int neg);
extern Long LongScanBinary (char **s, int neg);
extern Long LongScanString (String s);


/** LongBin LongOct LongHex
	Return binary, octal and hexidecimal representation of the Long.
*/
extern String LongBin (Long x);
extern String LongOct (Long x);
extern String LongHex (Long x);


/** LongLoad LongStore
	Load or store a Long to a Stream.
*/
extern Long LongLoad  (Stream x);
extern Null LongStore (Long self, Stream x);


/** LongHash
	Return the hash value of a Long.
*/
extern Int LongHash (Long self);


/** LongDigitSum
	Return the sum of the digits.
*/
extern Integer LongDigitSum (Long x);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Long class */
extern void LongSetup ();
extern void LongInit  ();

#endif /*_H_Long_*/
