/* Module.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Module_
#define _H_Module_ 1


/** Module
	This is the final Module class.

	The main program, modules, and k{eval} are all Module objects.

	Once successfuly compiled, module objects (except for k{eval}) live in
	memory forever. We never compile a module twice.

	@warn
	This behavior may change in the future so do not rely on it.
*/
extern Class ModuleClass;


/** ModuleIs
	Whether or not the object is a Module object.
*/
#define ModuleIs(o) (ClassOf(o) == ModuleClass)


/** TModule
	This represents an instance of the Module class.

	All fields are read-only.
*/
struct TModule
{
	OBJ_HEAD;
	String  name;     /* Absolute path with extension, or name in case of Sys or eval */
	Module  parent;   /* For eval modules: the module where eval is, 0 otherwise */
	Set     literals; /* Literals, may be shared */
	Set     use;      /* Used module (of Module objects) */
	void   *dlhandle; /* External library handle */
	Code    code;     /* Code object initially 0, see Code.h */
	String  source;   /* Source string, during compilation only */
	Ast     tree;     /* AST root node, during compilation only */
	Dict    doc;      /* Documentation, may be 0 */
};


/** ModuleFlag
	Flags used by Module objects.

	Do not access these flags directly.
	Use the accessor macros provided instead.
*/
#define ModuleFlagDOC        SymbolFlagDOC  /* documentation */
#define ModuleFlagMAIN       Flag10         /* main program */
#define ModuleFlagRESIDENT   Flag11         /* module */
#define ModuleFlagEVAL       Flag12         /* eval */
#define ModuleFlagTRUSTED    Flag13         /* trusted module? */

/* for interp */
#define ModuleFlagFINALIZED  Flag14         /* __destroy called? */

/* Status */
#define ModuleFlagPARSED     Flag18         /* parsed */
#define ModuleFlagCOMPILED   Flag19         /* compiled */
#define ModuleFlagREADY      Flag20         /* ready to run */
#define ModuleFlagINIT       Flag21         /* initialized */
#define ModuleFlagERROR      Flag22         /* failed */


/** ModuleIsTrusted
	Whether this is "trusted" module.

	If not, then it is restricted to certain things such as reading files, etc.

	The flag ModuleFlagTRUSTED is set by the interpreter.
	Once set it must never be unset.
*/
#define ModuleIsTrusted(self)  FlagTest (self, ModuleFlagTRUSTED)
#define ModuleSetTrusted(self) FlagSet  (self, ModuleFlagTRUSTED)


/** ModuleDoc
	The ModuleFlagDOC is set by the compiler.
	It indicates whether the module was compiled as a documentation module.

	Once set it may never be unset.
*/
#define ModuleIsDoc(self)  FlagTest (self, ModuleFlagDOC)
#define ModuleSetDoc(self) FlagSet  (self, ModuleFlagDOC)


/** ModuleIsMain
	Whether or not this is the main module.

	The ModuleFlagMAIN is set by ModuleNewMain and may only be queried.
*/
#define ModuleIsMain(self) FlagTest (self, ModuleFlagMAIN)


/** ModuleIsEval
	Whether or not this is an eval module.

	The ModuleFlagEVAL is set by ModuleNewEval and may only be queried.
*/
#define ModuleIsEval(self) FlagTest (self, ModuleFlagEVAL)


/** ModuleIsResident
	Whether or not this module is kept in memory all time.

	The ModuleFlagRESIDENT is Module.c and may only be queried.
*/
#define ModuleIsResident(self) FlagTest (self, ModuleFlagRESIDENT)


/** ModuleParse
	The ModuleFlagPARSED is set/unset by the compiler to keep track of whether
	or not the module has been parsed.
*/
#define ModuleParsed(self)      FlagTest  (self, ModuleFlagPARSED)
#define ModuleSetParsed(self)   FlagSet   (self, ModuleFlagPARSED)
#define ModuleUnsetParsed(self) FlagUnset (self, ModuleFlagPARSED)


/** ModuleCompile
	The ModuleFlagCOMPILED is set/unset by the compiler to keep track of whether
	or not the module has been compiled.

	This flag may only be set after a successful parsing
	(the flag ModuleFlagPARSED must be set).
*/
#define ModuleCompiled(self)       FlagTest  (self, ModuleFlagCOMPILED)
#define ModuleSetCompiled(self)    FlagSet   (self, ModuleFlagCOMPILED)
#define ModuleUnsetCompiled(self)  FlagUnset (self, ModuleFlagCOMPILED)


/** ModuleReady
	The ModuleFlagREADY is set by the compiler to indicate whether or not the module
	is ready to run by the interpreter.

	This flag must only be set after a successful compilation
	(the flag ModuleFlagCOMPILED is set).
*/
#define ModuleIsReady(self)  FlagTest (self, ModuleFlagREADY)
#define ModuleSetReady(self) FlagSet  (self, ModuleFlagREADY)


/** ModuleError
	The ModuleFlagERROR is set and used by the compiler to indicate that
	a module has an error (can not be properly compiled).

	Once set this flag must never be unset.
*/
#define ModuleHasError(self)    FlagTest (self, ModuleFlagERROR)
#define ModuleSetHasError(self) FlagSet  (self, ModuleFlagERROR)


/** ModuleInitialize
	The ModuleFlagINIT flag is set by the interpreter after the module has been
	initialized. Once set, this flag may never be unset.
*/
#define ModuleInitialized(self)    FlagTest (self, ModuleFlagINIT)
#define ModuleSetInitialized(self) FlagSet  (self, ModuleFlagINIT)


/** ModuleDir ModuleExt ModuleDlExt
	ModuleDir is the installed library directory. Generally C{/opt/qu/lib/}
	(with the trailing slash).

	ModuleExt is our module file extension, e.g. C{".qm"} (with the dot).

	ModuleDlExt is the shared library file extension, e.g. C{".so"} (with the dot).
*/
extern String ModuleDir;
extern String ModuleExt;
extern String ModuleDlExt;


/** ModuleMain ModuleNewMain ModuleNewEval
	ModuleNewMain creates the main program module ModuleMain. Only the main program
	(C{qu.c}) may call this function.

	ModuleNewEval creates an k{eval} module. Only the interpreter should call
	this function. See C{interp.c}.
*/
extern Module ModuleMain;
extern Module ModuleNewMain (String path);
extern Module ModuleNewEval (Module self, String src);


/** ModuleUse
	For the compiler, called for each k{use} statements.

	On success, if v{path} is a directory then an ={Array} of Module objects is
	returned, otherwise a Module object is returned. In case of a directory
	then v{as} must be 0 otherwise an exception is thrown.
*/
extern Obj ModuleUse (Module self, String use, String as, int package);


/** ModuleFailed
	For the compiler, when compilation failed. This ensures that failed
	modules are not kept in memory.
*/
extern void ModuleFailed (Module self);


/** ModuleGetAttr ModuleSetAttr
	For the compiler, to get/set module attributes.
*/
extern Symbol ModuleGetAttr (Module self, String name);
extern void   ModuleSetAttr (Module self, String name, Symbol thing);


/** ModuleImport ModuleImportAll
	For the compiler, to import symbols from another module.
	See C{Code.c}.
*/
extern int ModuleImport    (Module self, Module u, String name);
extern int ModuleImportAll (Module self, Module u);


/** ModuleLoaded
	If the module is loaded then return the Module object, NULL otherwise.
	v{path} must be an absolute path.
*/
extern Module ModuleLoaded (String path);


/** ModuleLoad ModuleLoadMath
	ModuleLoad loads a module.

	ModuleLoadMath loads the Math module.
*/
extern Module ModuleLoad     (String path);
extern Module ModuleLoadMath ();


/** ModuleAddSub
	Add a new Sub.

	ModuleAddSub creates a module level function.
*/
extern Sub ModuleAddSub (Module self, const char *name, void *func, int argc, long flags);

#define ModuleAddFunction(self, namez, faddr, argc) \
	ModuleAddSub (self, namez, faddr, argc, SubFlagCFUNC)


/** ModuleAddVar
	Add a new variable.
*/
extern Var ModuleAddVar (Module self, const char *name, Any value);


/** ModuleAddConst ModuleAddFloat ModuleAddInt ModuleAddString
	Add a new constant.
*/
extern Var ModuleAddConst  (Module self, const char *name, Any value);
extern Var ModuleAddFloat  (Module self, const char *name, double val);
extern Var ModuleAddInt    (Module self, const char *name, long val);
extern Var ModuleAddString (Module self, const char *name, const char *val);


/** ModuleAddAlias
	Add an alias of an existing symbol.
*/
extern Symbol ModuleAddAlias (Module self, const char *name, const char *alias);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Module class. */
extern void ModuleSetup ();
extern void ModuleInit  ();


#endif /*_H_Module_*/
