/* Number.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Number_
#define _H_Number_ 1


/** Number
	This is the abstract parent class of all number objects.
*/
extern Class NumberClass;


/** NumberPredefined
	Predefined system objects.
*/
#define Number0  ((Number) Int0)   /*  0 */
#define Number1  ((Number) Int1)   /*  1 */
#define Number2  ((Number) Int2)   /*  2 */
#define Number_1 ((Number) Int_1)  /* -1 */
#define Number_2 ((Number) Int_2)  /* -2 */


/** NumberContext
	Qu provides a global number context that can be set by an application.

	NumberContext is a combination of the NUMBER_CONTEXT flags.

	@warn
	All number classes must respect the setting.
*/
extern volatile long NumberContext;

#define NUMBER_CONTEXT_ALLOW_INF       0x01 /* throw EInf or EBig or return an infinity */
#define NUMBER_CONTEXT_ALLOW_TINY      0x02 /* throw ETiny or return 0 */
#define NUMBER_CONTEXT_ALLOW_HUGE      0x04 /* throw EHuge or return an infinity */
#define NUMBER_CONTEXT_ALLOW_ZERODIV   0x08 /* throw EDiv or return an infinity */
#define NUMBER_CONTEXT_ALLOW_COMPLEX   0x10 /* throw EComplex if result is Complex */
#define NUMBER_CONTEXT_REALIFY_COMPLEX 0x20 /* return Real if imaginary part is 0 */
#define NUMBER_CONTEXT_EXACT           0x40 /* try to return "exact" values */

/* this is the default setting */
#define NUMBER_CONTEXT_DEFAULT \
	(NUMBER_CONTEXT_ALLOW_TINY \
	| NUMBER_CONTEXT_ALLOW_HUGE \
	| NUMBER_CONTEXT_ALLOW_ZERODIV \
	| NUMBER_CONTEXT_ALLOW_COMPLEX)

/* test macros */
#define NUMBER_ALLOW_INF       (NumberContext & NUMBER_CONTEXT_ALLOW_INF)
#define NUMBER_ALLOW_HUGE      (NumberContext & NUMBER_CONTEXT_ALLOW_HUGE)
#define NUMBER_ALLOW_TINY      (NumberContext & NUMBER_CONTEXT_ALLOW_TINY)
#define NUMBER_ALLOW_ZERODIV   (NumberContext & NUMBER_CONTEXT_ALLOW_ZERODIV)
#define NUMBER_ALLOW_COMPLEX   (NumberContext & NUMBER_CONTEXT_ALLOW_COMPLEX)
#define NUMBER_REALIFY_COMPLEX (NumberContext & NUMBER_CONTEXT_REALIFY_COMPLEX)
#define NUMBER_EXACT           (NumberContext & NUMBER_CONTEXT_EXACT)

/* ensure macros */
#define NUMBER_ENSURE_ALLOW_INF      REQUIRE (EInf    , NUMBER_ALLOW_INF)
#define NUMBER_ENSURE_ALLOW_HUGE     REQUIRE (EHuge   , NUMBER_ALLOW_HUGE)
#define NUMBER_ENSURE_ALLOW_TINY     REQUIRE (ETiny   , NUMBER_ALLOW_TINY)
#define NUMBER_ENSURE_ALLOW_ZERODIV  REQUIRE (EDiv    , NUMBER_ALLOW_ZERODIV)
#define NUMBER_ENSURE_ALLOW_COMPLEX  REQUIRE (EComplex, NUMBER_ALLOW_COMPLEX)


/** __NumberContext
	Sometimes it is necessary to temporarily change the context to perform some
	calculations. Use these macros for that.

	@xmp
		int saved_context;
		__NumberContextBegin (saved_context, NumberContext | NUMBER_ALLOW_INF)
		  ...
		__NumberContextEnd (saved_context);

	@block
	Since any thread can change the context at any time then it might be necessary
	to use ={__Atomic}Begin/End.

	@xmp
		int saved_context;
		__AtomicBegin;
		__NumberContextBegin (saved_context, NUMBER_CONTEXT_DEFAULT)
		  ...
		__NumberContextEnd (saved_context);
		__AtomicEnd;
*/
#define __NumberContextBegin(var, new) \
BEGIN \
	var = NumberContext; \
	NumberContext = new; \
END

#define __NumberContextEnd(var) \
BEGIN \
	NumberContext = var; \
END


/** NumberNew
	Create a Number object from C types or other objects.

	These are just convenient wrappers around ={RealNew} and ={ComplexNew}.
*/
#define NumberNewh(v)   ((Number) RealNewh   (v))
#define NumberNewuh(v)  ((Number) RealNewuh  (v))
#define NumberNewi(v)   ((Number) RealNewi   (v))
#define NumberNewui(v)  ((Number) RealNewui  (v))
#define NumberNewl(v)   ((Number) RealNewl   (v))
#define NumberNewul(v)  ((Number) RealNewul  (v))
#define NumberNewli(v)  ((Number) RealNewli  (v))
#define NumberNewd(v)   ((Number) RealNewd   (v))
#ifdef HAVE_LONG_LONG
#define NumberNewll(v)  ((Number) RealNewll  (v))
#define NumberNewull(v) ((Number) RealNewull (v))
#endif
#ifdef HAVE_LONG_DOUBLE
#define NumberNewld(v)  ((Number) RealNewui  (v))
#endif

#define NumberNewInt(x)      ((Number) (x))
#define NumberNewFloat(x)    ((Number) (x))
#define NumberNewRational(x) ((Number) (x))
#define NumberNewBig(x)      ((Number) (x))
#define NumberNewComplex(x)  ((Number) (x))
#define NumberNewInteger(x)  ((Number) (x))
#define NumberNewDecimal(x)  ((Number) (x))
#define NumberNewReal(x)     ((Number) (x))
#define NumberNewNumber(x)   (x)


/** NumberTo
	Functions to convert any object into C types or other Number objects.

	For C type conversions, the functions set v{errno} to zero on success.
	On failure, v{errno} is set to EDOM if v{x} is conversion is not possible,
	to ERANGE if the value falls out of the C type range.
	The I{unsigned} versions fail with ERANGE if the value is negative.

	For object conversions, an exception is thrown and NULL is returned on failure.

	@note
	Qu uses a global abbreviation convention for C types.
	@code
		h     short
		uh    unsigned short
		i     int
		ui    unsigned int
		i16   int 2 bytes
		ui16  unsigned int 2 bytes
		i32   int 4 bytes
		ui32  unsigned int 4 bytes
		l     long
		ul    unsigned long
		li    long in the Int range
		d     double
		ll    long long
		ull   unsigned long long
		ld    long double

	@block
	These are just convenient wrappers around ={RealTo} and ={ComplexTo}.
*/
#define                   NumberTob(x)    Number_ltob     (NumberTol  (x))
#define                   NumberToub(x)   Number_ultob    (NumberToul (x))
#define                   NumberToh(x)    Number_ltoh     (NumberTol  (x))
#define                   NumberTouh(x)   Number_ultouh   (NumberToul (x))
#define                   NumberToi(x)    Number_ltoi     (NumberTol  (x))
#define                   NumberToui(x)   Number_ultoui   (NumberToul (x))
#define                   NumberToi16(x)  Number_ltoi16   (NumberTol  (x))
#define                   NumberToui16(x) Number_ultoui16 (NumberToul (x))
#define                   NumberToi32(x)  Number_ltoi32   (NumberTol  (x))
#define                   NumberToui32(x) Number_ultoui32 (NumberToul (x))
extern long               NumberTol       (Any x);
extern unsigned long      NumberToul      (Any x);
extern long               NumberToli      (Any x);
#define                   NumberTouli(x)  Number_ltouli   (NumberTol (x))
#define                   NumberTof(x)    Number_dtof     (NumberTod  (x))
extern double             NumberTod       (Any x);
#ifdef HAVE_LONG_LONG
extern long long          NumberToll      (Any x);
extern unsigned long long NumberToull     (Any x);
#endif
#ifdef HAVE_LONG_DOUBLE
extern long double        NumberTold      (Any x);
#endif

extern Int      NumberToInt      (Any x);
extern Float    NumberToFloat    (Any x);
extern Long     NumberToLong     (Any x);
extern Rational NumberToRational (Any x);
extern Big      NumberToBig      (Any x);
extern Complex  NumberToComplex  (Any x);
extern Integer  NumberToInteger  (Any x);
extern Decimal  NumberToDecimal  (Any x);
extern Real     NumberToReal     (Any x);
extern Number   NumberToNumber   (Any x);


/** NumberSign
	NumberSign returns -1, 0, 1 if the value is negative, 0, positive respectively.

	This is just a convenient wrapper around ={RealSign} and ={ComplexSign}.
*/
extern int NumberSign (Number x);


/** NumberIs
	Identity functions.

	NumberIs tests whether or not an object is a Number.

	NumberIsNeg returns 1 if the value is negative.
	Returns 0 otherwise.

	NumberIsZero returns 1 if the value is 0.
	Returns 0 otherwise.

	NumberIsOne returns 1 if the value is 1 or -1.
	Returns 0 otherwise.

	NumberIs1 return 1 if the value is 1.
	Returns 0 otherwise.

	NumberIsWhole return 1 if the value is a whole number.
	Returns 0 otherwise.

	NumberIsFrac return 1 if 0 < |x| < 1.
	Returns 0 otherwise.

	NumberIsOdd return 1 if the value is a whole and odd number.
	Returns 0 otherwise.

	These are just convenient wrappers around ={RealIs} and ={ComplexIs}.
*/
extern int NumberIs      (Any x);
extern int NumberIsZero  (Number x);
extern int NumberIsNeg   (Number x);
extern int NumberIsOne   (Number x);
extern int NumberIs1     (Number x);
extern int NumberIsWhole (Number x);
extern int NumberIsFrac  (Number x);
extern int NumberIsOdd   (Number x);


/** NumberUnary
	Unary functions.

	NumberPos returns the number itself, obviously.

	NumberNeg returns a negated value.

	NumberIncr returns the number incremented by 1.

	NumberDecr returns the number decremented by 1.

	These are just convenient wrappers around
	={RealPos} ={RealNeg} ={RealIncr} ={RealDecr}
	={ComplexPos} ={ComplexNeg} ={ComplexIncr} ={ComplexDecr}.
*/
extern Number NumberPos  (Number x);
extern Number NumberNeg  (Number x);
extern Number NumberIncr (Number x);
extern Number NumberDecr (Number x);


/** NumberEq
	Whether the value of a Number is the same as the value of another Number object.
	All return 0 or 1.

	These are just convenient wrappers around ={RealEq} and ={ComplexEq}.
*/
extern int NumberEqInt      (Number x, Int y);
extern int NumberEqFloat    (Number x, Float y);
extern int NumberEqLong     (Number x, Long y);
extern int NumberEqRational (Number x, Rational y);
extern int NumberEqBig      (Number x, Big y);
extern int NumberEqComplex  (Number x, Complex y);
extern int NumberEqInf      (Number x, Inf y);
extern int NumberEqInteger  (Number x, Integer y);
extern int NumberEqDecimal  (Number x, Decimal y);
extern int NumberEqReal     (Number x, Real y);
extern int NumberEqNumber   (Number x, Number y);


/** NumberLike
	Whether the absolute value of a Number is the same as the absolute value of another
	Number object.
	All return 0 or 1.

	These are just convenient wrappers around ={RealLike} and ={ComplexLike}.
*/
extern int NumberLikeInt      (Number x, Int y);
extern int NumberLikeFloat    (Number x, Float y);
extern int NumberLikeLong     (Number x, Long y);
extern int NumberLikeRational (Number x, Rational y);
extern int NumberLikeBig      (Number x, Big y);
extern int NumberLikeComplex  (Number x, Complex y);
extern int NumberLikeInf      (Number x, Inf y);
extern int NumberLikeInteger  (Number x, Integer y);
extern int NumberLikeDecimal  (Number x, Decimal y);
extern int NumberLikeReal     (Number x, Real y);
extern int NumberLikeNumber   (Number x, Number y);


/** NumberCmp
	Compare a Number with another Number object.
	All return -1, 0, or -1 in the usual manner.

	These are just convenient wrappers around ={RealCmp} and ={ComplexCmp}.
*/
extern int NumberCmpInt      (Number x, Int y);
extern int NumberCmpFloat    (Number x, Float y);
extern int NumberCmpLong     (Number x, Long y);
extern int NumberCmpRational (Number x, Rational y);
extern int NumberCmpBig      (Number x, Big y);
extern int NumberCmpComplex  (Number x, Complex y);
extern int NumberCmpInf      (Number x, Inf y);
extern int NumberCmpInteger  (Number x, Integer y);
extern int NumberCmpDecimal  (Number x, Decimal y);
extern int NumberCmpReal     (Number x, Real y);
extern int NumberCmpNumber   (Number x, Number y);


/** NumberCmpAbs
	Compare the absolute value of a Number with the absolute value of another
	Number object.
	All return -1, 0, or -1 in the usual manner.

	These are just convenient wrappers around ={RealCmpAbs} and ={ComplexCmpAbs}.
*/
extern int NumberCmpAbsInt      (Number x, Int y);
extern int NumberCmpAbsFloat    (Number x, Float y);
extern int NumberCmpAbsLong     (Number x, Long y);
extern int NumberCmpAbsRational (Number x, Rational y);
extern int NumberCmpAbsBig      (Number x, Big y);
extern int NumberCmpAbsComplex  (Number x, Complex y);
extern int NumberCmpAbsInf      (Number x, Inf y);
extern int NumberCmpAbsInteger  (Number x, Integer y);
extern int NumberCmpAbsDecimal  (Number x, Decimal y);
extern int NumberCmpAbsReal     (Number x, Real y);
extern int NumberCmpAbsNumber   (Number x, Number y);


/** NumberPlus
	Add a Number with another Number object.

	These are just convenient wrappers around ={RealPlus} and ={ComplexPlus}.
*/
extern Number  NumberPlusInt      (Number x, Int y);
extern Number  NumberPlusFloat    (Number x, Float y);
extern Number  NumberPlusLong     (Number x, Long y);
extern Number  NumberPlusRational (Number x, Rational y);
extern Number  NumberPlusBig      (Number x, Big y);
extern Complex NumberPlusComplex  (Number x, Complex y);
extern Number  NumberPlusInf      (Number x, Inf y);
extern Number  NumberPlusInteger  (Number x, Integer y);
extern Number  NumberPlusDecimal  (Number x, Decimal y);
extern Number  NumberPlusReal     (Number x, Real y);
extern Number  NumberPlusNumber   (Number x, Number y);


/** NumberMinus
	Subtract a Number with another Number object.

	These are just convenient wrappers around ={RealMinus} and ={ComplexMinus}.
*/
extern Number  NumberMinusInt      (Number x, Int y);
extern Number  NumberMinusFloat    (Number x, Float y);
extern Number  NumberMinusLong     (Number x, Long y);
extern Number  NumberMinusRational (Number x, Rational y);
extern Number  NumberMinusBig      (Number x, Big y);
extern Complex NumberMinusComplex  (Number x, Complex y);
extern Number  NumberMinusInf      (Number x, Inf y);
extern Number  NumberMinusInteger  (Number x, Integer y);
extern Number  NumberMinusDecimal  (Number x, Decimal y);
extern Number  NumberMinusReal     (Number x, Real y);
extern Number  NumberMinusNumber   (Number x, Number y);


/** NumberMul
	Multiply a Number with another Number object.

	These are just convenient wrappers around ={RealMul} and ={ComplexMul}.
*/
extern Number  NumberMulInt      (Number x, Int y);
extern Number  NumberMulFloat    (Number x, Float y);
extern Number  NumberMulLong     (Number x, Long y);
extern Number  NumberMulRational (Number x, Rational y);
extern Number  NumberMulBig      (Number x, Big y);
extern Complex NumberMulComplex  (Number x, Complex y);
extern Number  NumberMulInf      (Number x, Inf y);
extern Number  NumberMulInteger  (Number x, Integer y);
extern Number  NumberMulDecimal  (Number x, Decimal y);
extern Number  NumberMulReal     (Number x, Real y);
extern Number  NumberMulNumber   (Number x, Number y);


/** NumberDiv
	Divide a Number with another Number object.

	These are just convenient wrappers around ={RealDiv} and ={ComplexDiv}.
*/
extern Number  NumberDivInt      (Number x, Int y);
extern Number  NumberDivFloat    (Number x, Float y);
extern Number  NumberDivLong     (Number x, Long y);
extern Number  NumberDivRational (Number x, Rational y);
extern Number  NumberDivBig      (Number x, Big y);
extern Complex NumberDivComplex  (Number x, Complex y);
extern Number  NumberDivInf      (Number x, Inf y);
extern Number  NumberDivInteger  (Number x, Integer y);
extern Number  NumberDivDecimal  (Number x, Decimal y);
extern Number  NumberDivReal     (Number x, Real y);
extern Number  NumberDivNumber   (Number x, Number y);


/** NumberDivRem
	Divide a Number with another Number object returning the result and remainder.

	These are just convenient wrappers around ={RealDivRem} and ={ComplexDivRem}.
*/
extern Number  NumberDivRemInt      (Number x, Int      y, Number *r);
extern Number  NumberDivRemFloat    (Number x, Float    y, Number *r);
extern Number  NumberDivRemLong     (Number x, Long     y, Number *r);
extern Number  NumberDivRemRational (Number x, Rational y, Number *r);
extern Number  NumberDivRemBig      (Number x, Big      y, Number *r);
extern Complex NumberDivRemComplex  (Number x, Complex  y, Number *r);
extern Number  NumberDivRemInf      (Number x, Inf      y, Number *r);
extern Number  NumberDivRemInteger  (Number x, Integer  y, Number *r);
extern Number  NumberDivRemDecimal  (Number x, Decimal  y, Number *r);
extern Number  NumberDivRemReal     (Number x, Real     y, Number *r);
extern Number  NumberDivRemNumber   (Number x, Number   y, Number *r);


/** NumberIdiv
	Integer division of a Number with another Number object.

	These are just convenient wrappers around ={RealIdiv} and ={ComplexIdiv}.
*/
extern Integer NumberIdivInt      (Number x, Int y);
extern Integer NumberIdivFloat    (Number x, Float y);
extern Integer NumberIdivLong     (Number x, Long y);
extern Integer NumberIdivRational (Number x, Rational y);
extern Integer NumberIdivBig      (Number x, Big y);
extern Integer NumberIdivComplex  (Number x, Complex y);
extern Integer NumberIdivInf      (Number x, Inf y);
extern Integer NumberIdivInteger  (Number x, Integer y);
extern Integer NumberIdivDecimal  (Number x, Decimal y);
extern Integer NumberIdivReal     (Number x, Real y);
extern Integer NumberIdivNumber   (Number x, Number y);


/** NumberIdivRem
	Integer division a Number with another Number object returning the result and remainder.

	These are just convenient wrappers around ={RealIdivRem} and ={ComplexIdivRem}.
*/
extern Integer NumberIdivRemInt      (Number x, Int      y, Number *r);
extern Integer NumberIdivRemFloat    (Number x, Float    y, Number *r);
extern Integer NumberIdivRemLong     (Number x, Long     y, Number *r);
extern Integer NumberIdivRemRational (Number x, Rational y, Number *r);
extern Integer NumberIdivRemBig      (Number x, Big      y, Number *r);
extern Integer NumberIdivRemComplex  (Number x, Complex  y, Number *r);
extern Integer NumberIdivRemInf      (Number x, Inf      y, Number *r);
extern Integer NumberIdivRemInteger  (Number x, Integer  y, Number *r);
extern Integer NumberIdivRemDecimal  (Number x, Decimal  y, Number *r);
extern Integer NumberIdivRemReal     (Number x, Real     y, Number *r);
extern Integer NumberIdivRemNumber   (Number x, Number   y, Number *r);


/** NumberRem
	Remainder of dividing a Number with another Number object.

	These are just convenient wrappers around ={RealRem} and ={ComplexRem}.
*/
extern Number  NumberRemInt      (Number x, Int y);
extern Number  NumberRemFloat    (Number x, Float y);
extern Number  NumberRemLong     (Number x, Long y);
extern Number  NumberRemRational (Number x, Rational y);
extern Number  NumberRemBig      (Number x, Big y);
extern Complex NumberRemComplex  (Number x, Complex y);
extern Number  NumberRemInf      (Number x, Inf y);
extern Number  NumberRemInteger  (Number x, Integer y);
extern Number  NumberRemDecimal  (Number x, Decimal y);
extern Number  NumberRemReal     (Number x, Real y);
extern Number  NumberRemNumber   (Number x, Number y);


/** NumberPow
	Raise a Number to the power of another Number object.

	These are just convenient wrappers around ={RealPow} and ={ComplexPow}.
*/
extern Number  NumberPowInt      (Number x, Int y);
extern Number  NumberPowFloat    (Number x, Float y);
extern Number  NumberPowLong     (Number x, Long y);
extern Number  NumberPowRational (Number x, Rational y);
extern Number  NumberPowBig      (Number x, Big y);
extern Complex NumberPowComplex  (Number x, Complex y);
extern Number  NumberPowInf      (Number x, Inf y);
extern Number  NumberPowInteger  (Number x, Integer y);
extern Number  NumberPowDecimal  (Number x, Decimal y);
extern Number  NumberPowReal     (Number x, Real y);
extern Number  NumberPowNumber   (Number x, Number y);


/** NumberRound
	Rounding functions.

	NumberCeil rounds the number to the nearest integer towards positive infinity.
	EBig is thrown on failure.

	NumberFloor rounds the number to the nearest integer towards negative infinity.
	EBig is thrown on failure.

	NumberTrunc round the number towards zero, that is, it removes the
	fraction part. This function throws EMath if the number is an infinity.

	These are just convenient wrappers around
	={RealFloor} ={RealCeil} ={RealTrunc}
	={ComplexFloor} ={ComplexCeil} ={ComplexTrunc}
*/
extern Number NumberFloor (Number x);
extern Number NumberCeil  (Number x);
extern Number NumberTrunc (Number x);
extern Number NumberFrac  (Number x);
extern Number NumberRound (Number x, int prec);


/** NumberPrint
	Print a Number to v{stdout}.

	Note that this might throw an exception because writing to v{stdout} may
	fail for various reasons.

	This is just a convenient wrapper around ={RealPrint} and ={ComplexPrint}.
*/
extern Null NumberPrint (Number x);


/** NumberFormat
	Format a Number.

	See ={Format} for details.

	This is just a convenient wrapper around ={RealFormat} and ={ComplexFormat}.
*/
extern String NumberFormat (Number x, format_s *f);


/** NumberScan
	Scan number.

	v{s} is the start of the string to scan.

	If v{options} does not contain the NUMBER_SCAN_SKIPSPACE flag then the scanner
	will not skip leading blanks and will not read the sign, that is, the string may
	not contain the sign nor leading spaces.

	If it contains the NUMBER_SCAN_COMPLEX flag then the scanner will treat
	things like C{-123j} and {-123 + 456j} as a complex number.

	On success a Number object is returned, NULL otherwise.

	NumberScanString does the same but throws ESyntax on failure.
*/
extern Number NumberScan       (char **s, int options);
extern Number NumberScanString (String s, int options);

#define NUMBER_SCAN_SKIPSPACE 0x1
#define NUMBER_SCAN_COMPLEX   0x2


/** NUMBER_COMPARE_VALUE
	Compare two C values, yielding -1, 0 or 1 if v{x} is smaller, the same,
	and greater than v{y} respectively.
*/
#define NUMBER_COMPARE_VALUE(x, y)  (((x) > (y)) ? 1 : ((x) < (y)) ? -1 : 0)


/** NUMBER_COMPARE_SIGN
	Compare two signs and issues the appropriate return statements.
	This is generally used on top of Cmp functions.
*/
#define NUMBER_COMPARE_SIGN(sx, sy) \
BEGIN \
	if ((sx) > 0) \
	{ \
		if ((sy) <= 0) \
			return 1; \
	} \
	else if ((sx) < 0) \
	{ \
		if ((sy) >= 0) \
			return -1; \
	} \
	else \
	{ \
		return -(sy); \
	} \
END
/* ... here you must compare the values with the same sign */



/** NUMBER_BINOP_Integer
	Make a binop function with an Integer, e.g:
	@code
		Bar FooOpInteger (Foo x, Integer y)

	@block
	Foo must provide the functions:
	@code
		Bar FooOpInt  (Foo x, Int  y) ...
		Bar FooOpLong (Foo x, Long y) ...
*/
#define NUMBER_BINOP_Integer(xtype, name, rtype) \
rtype xtype##name##Integer (xtype x, Integer y) \
{ \
	return IntIs (y) \
		? (rtype) xtype##name##Int  (x, (Int ) y) \
		: (rtype) xtype##name##Long (x, (Long) y); \
}


/** NUMBER_BINOP_Decimal
	Make a binop function with a Decimal, e.g:
	@code
		Bar FooOpDecimal (Foo x, Decimal y)

	@block
	Foo must provide the functions:
	@code
		Bar FooOpFloat (Foo x, Float y) ...
		Bar FooOpBig   (Foo x, Big   y) ...
*/
#define NUMBER_BINOP_Decimal(xtype, name, rtype) \
rtype xtype##name##Decimal (xtype x, Decimal y) \
{ \
	return FloatIs (y) \
		? (rtype) xtype##name##Float (x, (Float) y) \
		: (rtype) xtype##name##Big   (x, (Big  ) y); \
}


/** NUMBER_BINOP_Real
	Make a binop function with a Real, e.g:
	@code
		Bar FooOpReal (Foo x, Real y)

	@block
	Foo must provide the functions:
	@code
		Bar FooOpInt      (Foo x, Int   y) ...
		Bar FooOpLong     (Foo x, Long  y) ...
		Bar FooOpRational (Foo x, Long  y) ...
		Bar FooOpFloat    (Foo x, Float y) ...
		Bar FooOpBig      (Foo x, Big   y) ...
		Bar FooOpInf      (Foo x, Inf   y) ...
*/
#define NUMBER_BINOP_Real(xtype, name, rtype) \
rtype xtype##name##Real (xtype x, Real y) \
{ \
	switch (ClassId (y)) \
	{ \
		case CLASS_ID_Int:      return (rtype) xtype##name##Int      (x, (Int     ) y); \
		case CLASS_ID_Long:     return (rtype) xtype##name##Long     (x, (Long    ) y); \
		case CLASS_ID_Rational: return (rtype) xtype##name##Rational (x, (Rational) y); \
		case CLASS_ID_Float:    return (rtype) xtype##name##Float    (x, (Float   ) y); \
		case CLASS_ID_Big:      return (rtype) xtype##name##Big      (x, (Big     ) y); \
		case CLASS_ID_Inf:      return (rtype) xtype##name##Inf      (x, (Inf     ) y); \
	} \
	FATAL ("y is not a Real object"); \
	return 0; /*never*/ \
}


/** NUMBER_BINOP_Number
	Make a binop function with a Number, e.g:
	@code
		Bar FooOpReal (Foo x, Number y)

	@block
	Foo must provide the functions:
	@code
		Bar FooOpInt      (Foo x, Int     y) ...
		Bar FooOpLong     (Foo x, Long    y) ...
		Bar FooOpRational (Foo x, Long    y) ...
		Bar FooOpFloat    (Foo x, Float   y) ...
		Bar FooOpBig      (Foo x, Big     y) ...
		Bar FooOpInf      (Foo x, Real    y) ...
		Bar FooOpComplex  (Foo x, Complex y) ...
*/
#define NUMBER_BINOP_Number(xtype, name, rtype) \
rtype xtype##name##Number (xtype x, Number y) \
{ \
	switch (ClassId (y)) \
	{ \
		case CLASS_ID_Int:      return (rtype) xtype##name##Int      (x, (Int     ) y); \
		case CLASS_ID_Long:     return (rtype) xtype##name##Long     (x, (Long    ) y); \
		case CLASS_ID_Rational: return (rtype) xtype##name##Rational (x, (Rational) y); \
		case CLASS_ID_Float:    return (rtype) xtype##name##Float    (x, (Float   ) y); \
		case CLASS_ID_Big:      return (rtype) xtype##name##Big      (x, (Big     ) y); \
		case CLASS_ID_Inf:      return (rtype) xtype##name##Inf      (x, (Inf     ) y); \
		case CLASS_ID_Complex:  return (rtype) xtype##name##Complex  (x, (Complex ) y); \
	} \
	FATAL ("y is not a Number object"); \
	return 0; /*never*/ \
}


/** NumberHelpers
	C type conversion helpers.

	All must be entered with errno = 0 or ERANGE.
	These function do not change errno.

	If entered with errno == ERANGE then return a minimum/maximum value.
*/
extern char           Number_ltob     (long v);
extern unsigned char  Number_ultoub   (unsigned long v);
extern short          Number_ltoh     (long v);
extern unsigned short Number_ultouh   (unsigned long v);
extern int            Number_ltoi16   (long v);
extern unsigned int   Number_ultoui16 (unsigned long v);
extern long           Number_ltoi32   (long v);
extern unsigned long  Number_ultoui32 (unsigned long v);
extern long           Number_ltoli    (long v);
extern long           Number_ltouli   (long v);
#if SIZEOF_INT != SIZEOF_LONG
extern int            Number_ltoi     (long v);
extern unsigned int   Number_ultoui   (unsigned long v);
#else
#define Number_ltoi(v)   ((int) (v))
#define Number_ultoui(v) ((unsigned int) (v))
#endif
extern float          Number_dtof     (double v);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Number class */
extern void NumberSetup ();
extern void NumberInit  ();

/* for Sys.c */
extern String Number_context (Class self, int argc, Any *argv);


#endif /*_H_Number_*/
