/* Object.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Object_
#define _H_Object_ 1

#include "base.h"


#if (SIZEOF_LONG != SIZEOF_VOID_P)
#error *** We need sizeof (long) = sizeof (void *)
#endif


/** Object OBJ_HEAD
	All object structs must use OBJ_HEAD as the first entry.

	v{klass} must be the first field. You must never change this because
	it is used by the GC to mark an object or memory chunk as being freed.
	It is also the most frequent used field, so this helps performance too.

	v{attrs} is used to store the attributes and is initially set to NULL
	by the GC. It must only be allocated as necessary since many objects
	may never need it. It is not efficient for objects that don't use it
	(e.g. String, Float, etc.) but lets be practical. Most user objects will
	use it anyway.

	v{flags} are used by most builtin objects. Program level objects usually
	don't use it but Qu provides program level methods to access it so this
	is also not entirely wasted. See ={Flag}.
*/
#ifdef QU_WITH_WEAK

#define OBJ_HEAD \
	Class   klass; /* class object, never 0 */ \
	long    flags; /* see Flag */ \
	Set     weak;  /* weak references, may be 0 */ \
	attr_s *attrs  /* attributes, may be 0 */

#else /*QU_WITH_WEAK*/

#define OBJ_HEAD \
	Class   klass; \
	long    flags; \
	attr_s *attrs

#endif /*QU_WITH_WEAK*/

#ifndef ATTR_S_DECLARED
#define ATTR_S_DECLARED
typedef struct attr_s attr_s;  /* see attr.h */
#endif


/** TObj Obj Any
	Predeclare all builtin object structs here.

	v{Any} is used for any object including ={Int} while v{Obj} is used for
	any object excluding Int.
*/
typedef struct TObj       TObj;
typedef struct TArray     TArray;
typedef struct TAst       TAst;
typedef struct TBig       TBig;
typedef struct TBuffer    TBuffer;
typedef struct TClass     TClass;
typedef struct TCode      TCode;
typedef struct TComplex   TComplex;
typedef struct TContainer TContainer;
typedef struct TDict      TDict;
typedef struct TDlib      TDlib;
typedef struct TException TException;
typedef struct TFloat     TFloat;
typedef struct TFrame     TFrame;
typedef struct TFile      TFile;
typedef struct TIterator  TIterator;
typedef struct TLong      TLong;
typedef struct TMethod    TMethod;
typedef struct TModule    TModule;
typedef struct TOpaque    TOpaque;
typedef struct TProc      TProc;
typedef struct TPointer   TPointer;
typedef struct TQueue     TQueue;
typedef struct TRange     TRange;
typedef struct TRational  TRational;
typedef struct TRecord    TRecord;
typedef struct TSet       TSet;
typedef struct TStream    TStream;
typedef struct TString    TString;
typedef struct TStruct    TStruct;
typedef struct TSub       TSub;
typedef struct TThread    TThread;
typedef struct TTokenizer TTokenizer;
typedef struct TVar       TVar;
typedef struct TWeak      TWeak;

/* abstracts - to enforce unique pointer types */
typedef struct TDecimal  TDecimal;
typedef struct TFunction TFunction;
typedef struct TInt      TInt;
typedef struct TInteger  TInteger;
typedef struct TInf      TInf;
typedef struct TNull     TNull;
typedef struct TNumber   TNumber;
typedef struct TReal     TReal;

typedef void        *Any;       /* any object including Int */
typedef TDict       *Arg;
typedef TArray      *Array;
typedef TAst        *Ast;
typedef TBig        *Big;
typedef TObj        *Boolean;
typedef TBuffer     *Buffer;
typedef TClass      *Class;
typedef TCode       *Code;
typedef TComplex    *Complex;
typedef TContainer  *Container; /* mixin */
typedef TObj        *Decimal;
typedef TDict       *Dict;
typedef TDlib       *Dlib;
typedef TException  *Exception;
typedef TFile       *File;
typedef TFloat      *Float;
typedef TFrame      *Frame;
typedef TFunction   *Function;  /* abstract */
typedef TInt        *Int;
typedef TInteger    *Integer;   /* abstract */
typedef TInf        *Inf;
typedef TIterator   *Iterator;
typedef TLong       *Long;
typedef TMethod     *Method;
typedef TModule     *Module;
typedef TNull       *Null;
typedef TNumber     *Number;    /* abstract */
typedef TObj        *Obj;       /* any object excluding Int */
typedef TObj        *Object;    /* instance of Object */
typedef TOpaque     *Opaque;
typedef TProc       *Proc;
typedef TPointer    *Pointer;
typedef TQueue      *Queue;
typedef TRange      *Range;
typedef TRational   *Rational;
typedef TReal       *Real;      /* abstract */
typedef TRecord     *Record;
typedef TSet        *Set;
typedef TStream     *Stream;
typedef TString     *String;
typedef TStruct     *Struct;
typedef TSub        *Sub;
typedef TThread     *Thread;
typedef TTokenizer  *Tokenizer;
typedef TVar        *Var;
typedef TWeak       *Weak;

struct TObj
{
	OBJ_HEAD;
};



/** Flag
	List of flags. The first four are strictly for use by the GC.
	The others may be used by classes.

	Observe that some classes share the same of the flags.

	@warn
	Flags may not be used with ={Int} objects, of course.
*/

/* Used by the GC */
#define FlagMARK     0x1000000  /* GC: marked? */
#define FlagFIN      0x2000000  /* GC: finalizer already called? */
#define FlagDONTGC   0x4000000  /* GC: dont GC? */
#define FlagGCSTATIC 0x8000000  /* GC: never GC'ed */

/* Usable by classes */
#define Flag0   0x1
#define Flag1   0x2
#define Flag2   0x4
#define Flag3   0x8
#define Flag4   0x10
#define Flag5   0x20
#define Flag6   0x40
#define Flag7   0x80
#define Flag8   0x100
#define Flag9   0x200
#define Flag10  0x400
#define Flag11  0x800
#define Flag12  0x1000
#define Flag13  0x2000
#define Flag14  0x4000
#define Flag15  0x8000
#define Flag16  0x10000
#define Flag17  0x20000
#define Flag18  0x40000
#define Flag19  0x80000
#define Flag20  0x100000
#define Flag21  0x200000
#define Flag22  0x400000
#define Flag23  0x800000


/** FlagTest FlagSet FlagUnset
	These are the 3 macros to test, set and unset a flag.
	See ={Flag}.

	You should generally not use these macros and use the flag accessors
	provided by the corresponding classes. That is, ideally these macros should
	only be used within classes.
*/
#define FlagTest(o, f)   ((((Obj)(o))->flags & (f)) != 0)
#define FlagSet(o, f)    BEGIN ((Obj)(o))->flags |= (f); END
#define FlagUnset(o, f)  BEGIN ((Obj)(o))->flags &= ~(f); END


/** OBJECT_MAXIDENTLEN
	The maximum number of characters in any identifiers.

	This limit is used because in many cases we can use the C stack
	to build strings and save memory allocation.
*/
#define OBJECT_MAXIDENTLEN 64


/** free_f mark_f
	Low-level instance routine prototypes, see ={Class}.

	Notice the argument type is Obj because it must not be an Int.
*/
typedef void (*free_f) (Obj o);
typedef void (*mark_f) (Obj o);


/** unop_f binop_f trip_f quartop_f
	C method function prototypes.
*/
typedef Any (*unop_f   ) (Any self);
typedef Any (*binop_f  ) (Any self, Any x);
typedef Any (*triop_f  ) (Any self, Any x, Any y);
typedef Any (*quartop_f) (Any self, Any x, Any y, Any z);

/** subn_f sub3_f sub2_f sub1_f sub0_f subcn_f subc3_f subc2_f subc1_f subc0_f
	C method and function prototypes.
*/
typedef Any (*subn_f) (Any self, int argc, Any *argv);
typedef Any (*sub4_f) (Any self, Any a1, Any a2, Any a3, Any a4);
typedef Any (*sub3_f) (Any self, Any a1, Any a2, Any a3);
typedef Any (*sub2_f) (Any self, Any a1, Any a2);
typedef Any (*sub1_f) (Any self, Any a1);
typedef Any (*sub0_f) (Any self);

typedef Any (*subcn_f) (int argc, Any *argv);
typedef Any (*subc4_f) (Any a1, Any a2, Any a3, Any a4);
typedef Any (*subc3_f) (Any a1, Any a2, Any a3);
typedef Any (*subc2_f) (Any a1, Any a2);
typedef Any (*subc1_f) (Any a1);
typedef Any (*subc0_f) ();


/***/
/*----------------------------------------------------------------------------
	Symbol = Module, Class, Sub, Var
----------------------------------------------------------------------------*/

/** SYMBOL_HEAD TSymbol Symbol
	Module, Class, Sub and Var objects must use SYMBOL_HEAD instead of ={OBJ_HEAD}.
*/
#define SYMBOL_HEAD \
	OBJ_HEAD; \
	String name       /* symbol name */


typedef struct TSymbol TSymbol;
struct TSymbol
{
	SYMBOL_HEAD;
};

typedef struct TSymbol *Symbol;


/** SymbolModuleOf SymbolClassOf
	Return the Module or Class object where a symbol is declared.
	Return 0 if none.
*/
extern Module SymbolModuleOf (Symbol self);
extern Class  SymbolClassOf  (Symbol self);


/** SymbolFlag
	Flags shared by Module, Class, Sub, and Var.
*/
#define SymbolFlagEXTERN    Flag0  /* Class/Sub: C (not program level) */
#define SymbolFlagABSTRACT  Flag1  /* Class/Sub: abstract */
#define SymbolFlagPRIVATE   Flag2  /* Class/Sub/Var : known to block & parent block */
#define SymbolFlagFINAL     Flag3  /* Class/Sub/Var: cant override */
#define SymbolFlagLOCAL     Flag4  /* Sub/Var: symbol known to block only */
#define SymbolFlagSTATIC    Flag5  /* Sub/Var: is static (self can be anything) */
#define SymbolFlagCLASS     Flag6  /* Sub/Var: is class attribute */
#define SymbolFlagINST      Flag7  /* Sub/Var: is instance attribute */
#define SymbolFlagDOC       Flag8  /* Module/Class: compiled for documentation? */

#define SymbolIsExtern(x)   FlagTest (x, SymbolFlagEXTERN)
#define SymbolIsAbstract(x) FlagTest (x, SymbolFlagABSTRACT)
#define SymbolIsPrivate(x)  FlagTest (x, SymbolFlagPRIVATE)
#define SymbolIsFinal(x)    FlagTest (x, SymbolFlagFINAL)
#define SymbolIsLocal(x)    FlagTest (x, SymbolFlagLOCAL)
#define SymbolIsStatic(x)   FlagTest (x, SymbolFlagSTATIC)
#define SymbolIsClass(x)    FlagTest (x, SymbolFlagCLASS)
#define SymbolIsInst(x)     FlagTest (x, SymbolFlagINST)
#define SymbolIsDoc(x)      FlagTest (x, SymbolFlagDOC)


/***/
/*----------------------------------------------------------------------------
	Object
----------------------------------------------------------------------------*/

/** ObjectClass
	This is the U{not} base ={Class} of all classes.

	The Object class simply provides default methods such as __call, __inst, __cmp,
	etc. See the program documentation for details.
*/
extern Class ObjectClass;


/** ObjectAs
	A convenient function to be called from f{new} and __as methods.
*/
extern Any ObjectAs (Any self, Any x);


/** ObjectHash
	Return the hash value of an object.

	ObjectHash simply calls object__hash and convert it into a long.
*/
extern long ObjectHash   (Any self);
extern Int  object__hash (Any self);


/** ObjectSelf ObjectTrue ObjectFalse ObjectInt0
	Methods that always return self, false, true, etc. can just point
	to these methods.
*/
extern Any     ObjectSelf  (Any self);
extern Boolean ObjectTrue  (Any self);
extern Boolean ObjectFalse (Any self);
extern Null    ObjectNil   (Any self);
extern Int     ObjectInt0  (Any self);
extern Any     ObjectEMath (Any self);


/** ObjectEImplement
	Methods that always throw EImplement.
*/
extern Any ObjectEImplement0 (Any self);
extern Any ObjectEImplement1 (Any self, Any a1);
extern Any ObjectEImplement2 (Any self, Any a1, Any a2);
extern Any ObjectEImplement3 (Any self, Any a1, Any a2, Any a3);
extern Any ObjectEImplement4 (Any self, Any a1, Any a2, Any a3, Any a4);
extern Any ObjectEImplementn (Any self, int argc, Any *argv);


/** ObjectDefines
	Whether or not an object defines a specified attribute.

	This function is provided for efficiency over C{Object.__defines}.
*/
extern int ObjectDefines (Any self, String name);


/** ObjectIterable
	Whether an object is iterable.
*/
extern int ObjectIterable (Any self);


/** ObjectCallable
	Whether the object is a callable: Sub, Method or Class.
*/
extern int ObjectCallable (Any self);


/** ObjectCall
	ObjectCall call a method by name where v{name} must be a ={String}.

	ObjectCallx automatically make the call base on what v{self} is.
	v{self} may be a Sub, Method or Class, and not be an instance method.
	Otherwise ECall is thrown.

	ObjectCallEx is a flexible call.
	v{how} may be a String, Sub, Method or Class.
	If v{how} is as String then call C{self.(how)(*arg)}
	otherwise call C{how (self, *arg)} setting self as ={InterpSelf}.
*/
extern Any ObjectCalln   (Any self, String name, int argc, Any *argv);
extern Any ObjectCall4   (Any self, String name, Any a1, Any a2, Any a3, Any a4);
extern Any ObjectCall3   (Any self, String name, Any a1, Any a2, Any a3);
extern Any ObjectCall2   (Any self, String name, Any a1, Any a2);
extern Any ObjectCall1   (Any self, String name, Any a1);
extern Any ObjectCall0   (Any self, String name);

extern Any ObjectCallxn  (Any self, int argc, Any *argv);
extern Any ObjectCallx4  (Any self, Any a1, Any a2, Any a3, Any a4);
extern Any ObjectCallx3  (Any self, Any a1, Any a2, Any a3);
extern Any ObjectCallx2  (Any self, Any a1, Any a2);
extern Any ObjectCallx1  (Any self, Any a1);
extern Any ObjectCallx0  (Any self);

extern Any ObjectCallExn (Any self, Any how, int argc, Any *argv);
extern Any ObjectCallEx3 (Any self, Any how, Any a1, Any a2, Any a3);
extern Any ObjectCallEx2 (Any self, Any how, Any a1, Any a2);
extern Any ObjectCallEx1 (Any self, Any how, Any a1);
extern Any ObjectCallEx0 (Any self, Any how);


/** ObjectRank
	Return the rank of an object.

	The value must not be taken literally as it may change in the future.
	The only purpose of this function is for sorting.
*/
extern int ObjectRank (Any self);


/** ObjectIsa ObjectInstOf
	ObjectIsa is the "is" operator. It works for any object including classes.
	Both v{self} and v{x} may be anything. If they are not classes then ={ClassOf}
	is used. Be careful, this is a very flexible operator.

	@xmp
		"foo"  is String -> true
		Int    is Number -> true
		String is Class  -> true

	@block
	ObjectInstOf checks whether self is an instance of Class v{x} or a class
	derived from Class v{x}.

	@xmp
		"foo"       instof  StringClass -> true
		IntClass    instof  NumberClass -> false
		StringClass instof  ClassClass  -> true
*/
extern int ObjectIsa (Any self, Any x);
#define ObjectInstOf(o, klass) ClassIsa (ClassOf(o), klass)


/** ObjectSpecificProp ObjectSpecificMethod ObjectSpecificDelete
	For modules, to create or delete instance specific attribute.

	We do not directly support specific attributes in program level for a number
	of reasons (security, documentability, etc.). C classes will have to implement
	it themselves as necessary using these functions.

	Instance specifics must be avoided if possible. Generally you would only use
	them if you know there are not a lot of instances that will be created.

	@warn
	The class must have the ClassFlagDYNAMIC flag set otherwise a ={FATAL} error
	is issued.

	Class objects may not have specific attributes.
	v{self} may not be a class otherwise a ={FATAL} error is issued.

	Never assign methods from foreign classes. You risk the danger of calling a
	C method with the wrong instance/class.

	v{func} must be a C function address.
*/
extern Any  ObjectSpecificProp   (Obj self, String name, Any value);
extern Sub  ObjectSpecificMethod (Obj self, String name, void *func, int argc, long flags);
extern Null ObjectSpecificDelete (Obj self, String name);


/** ObjectWhereStr
	Use to print symbols (={Module}, ={Class}, ={Sub} and ={Var}) so we have
	a consistent form.
*/
extern String ObjectWhereStr (Any sym, const char *prefix, const char *suffix);


/** ObjectValidType ObjectFixType ObjectCompatType ObjectCheckValue ObjectValidate
	These are I{validator} routines.

	ObjectValidType return 1 if v{x} is a valid validator, 1 otherwise.
	Valid validators: Class, Struct, Range, Array, Set, Dict, static Sub,
	and the special ={VarIsValidator} variable.

	ObjectFixType fixes a compile-time generated type which is still in the form
	of a Var into one of VarIsValidator, Struct or Dict (enum).

	ObjectCompatType return 1 if v{othertype} is a compatible validator for
	v{type}. For example, if both are frozen Arrays containing the same objects
	then they are compatible. If v{othertype} is a subclass of v{type} then
	return 1 but not the other way around.

	ObjectCheckValue is designed for compile-time. It validates the value but does
	not call the Sub if k{type} is a Sub.

	ObjectValidate is designed for run-time. It validates the value and call Sub
	validators as necessary.
*/
extern int  ObjectValidType  (Any x);
extern Any  ObjectFixType    (Any x);
extern int  ObjectCompatType (Any type, Any othertype);
extern int  ObjectCheckValue (Any type, Any value);
extern Any  ObjectValidate   (Any self, Any type, Any value);


/** ObjectDefaultString
	Return a default representation String of an object.

	@warn
	This does not call the __string method.
	It returns a default value.
*/
extern String ObjectDefaultString (Any x);


/** ObjectDefaultCmp
	Return default comparisson value of two objects.

	This function return -1, 0, and 1.
	Not just arbitrary integer.

	@warn
	This does not call the __cmp method.
	It returns a default value.
*/
extern int ObjectDefaultCmp    (Any self, Any x);
extern Int ObjectDefaultCmpInt (Any self, Any x);


/** ObjectDefaultPrint
	Default __print method.
*/
extern Null ObjectDefaultPrint (Any self);


/** ObjectModule
	Get the Module where the object came from.
*/
extern Module ObjectModule (Any self);


/** ObjectClassName
	Get the class name of an object.
*/
extern String ObjectClassName (Any self);


/** ObjectDefaultEq
	Whether self and x are the very same object.
*/
extern Boolean ObjectDefaultEq (Any self, Any x);


/** OBJECT_COPY OBJECT_RCOPY OBJECT_SETVAL
	Some handy tools.

	OBJECT_COPY should be faster than f{memcpy}. If not then it should be
	translated into a macro.
*/
inline static void OBJECT_COPY (register Any *dst, register Any *src, register long n)
{
	while (n--)
		*dst++ = *src++;
}

inline static void OBJECT_RCOPY (register Any *dst, register Any *src, register long n)
{
	dst += n;
	src += n;
	while (n--)
		*--dst = *--src;
}

inline static void OBJECT_SETVAL (register Any *dst, register Any v, register long n)
{
	while (n--)
		*dst++ = v;
}


/***/
/*--------------------------------------------------------------------------*/

/* This is called by Class.c to initialize the Object class */
extern void ObjectSetup ();

#endif /*_H_Object_*/
