/* Opaque.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Opaque_
#define _H_Opaque_ 1


/** Opaque
	The Opaque class provides a convenient way to wrap C objects.
*/
extern Class OpaqueClass;


/** opaquemark_f opaquefree_f
	The mark and free method prototypes.
*/
typedef void (*opaquemark_f) (Opaque o);
typedef void (*opaquefree_f) (Opaque o);


/** OPAQUE_HEAD OPAQUE_HEADEX
	To create an Opaque object, use this instead of ={OBJ_HEAD}.

	@warn
	The f{opaquefree} function must B{not} free the object itself because
	that is done internally by Opaque.
*/
#define OPAQUE_HEAD \
	OBJ_HEAD; \
	opaquemark_f  mark;     /* called when marked, may be 0 */ \
	opaquefree_f  free      /* called before freed, may be 0 */

#define OPAQUE_HEADEX \
	OBJ_HEAD; \
	opaquemark_f  mark;     /* called when marked, may be 0 */ \
	opaquefree_f  free;     /* called before freed, may be 0 */ \
	sub1_f        getitem;  /* to implement a C struct */ \
	sub2_f        setitem   /* -ditto- */


/** TOpaque
	This represents an instance of the Opaque class.
*/
struct TOpaque
{
	OPAQUE_HEAD;
};


/** OpaqueNew OpaqueNewEx
	Create a new Opaque instance.

	@args
	.z  is the size of the struct.
	.g  is the __getitem C method.
	.s  is the __setitem C method.
	.m  is the mark C function to call to mark referred objects.
	.f  is the free C function to free memory.

	@block
	Use the macro for it is easier to use than Opaque_New.
*/
#define OpaqueNew(m, f, z) \
	((Any) Opaque_New ((opaquemark_f) m, (opaquefree_f) f, z))

#define OpaqueNewEx(m, f, g, s, z) \
	((Any) Opaque_New ((opaquemark_f) m, (opaquefree_f) f, z, (sub1_f) g, (sub2_f) s))

extern Opaque Opaque_New   (opaquemark_f m, opaquefree_f f, size_t z);
extern Opaque Opaque_NewEx (opaquemark_f m, opaquefree_f f, size_t z, sub1_f g, sub2_f s);


/** OpaqueIs
	OpaqueIs checks whether or not an object is an Opaque object.

	OPAQUE_CHECK checks whether or not an Opaque object is indeed your
	Opaque object. You can use either OPAQUE_CHECK_MARK or OPAQUE_CHECK_FREE.
*/
#define OpaqueIs(o) (ClassOf(o) == OpaqueClass)
#define OPAQUE_CHECK_MARK(o, markfunc) (((Opaque) o)->mark == (opaquemark_f) markfunc)
#define OPAQUE_CHECK_FREE(o, freefunc) (((Opaque) o)->free == (opaquefree_f) freefunc)


/** OpaqueFlag
	Flags used by Opaque objects.

	@warn
	v{Flag0} through v{Flag7} are reserved for Opaque.
	You are free to use the others.
*/
#define OpaqueEX       Flag0 /* uses OPAQUE_HEADEX */
#define OpaqueSTRFIELD Flag1 /* only call setitem/getitem if index is a String */


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Opaque class */
extern void OpaqueSetup ();
extern void OpaqueInit  ();

#endif /*_H_Opaque_*/
