/* Queue.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Queue_
#define _H_Queue_ 1

#include "Container.h"


/** Queue
	The final Queue class.
*/
extern Class QueueClass;


/** TQueue
	This represents an instance of the Queue class.

	See ={CONTAINER_HEAD}.
*/
typedef struct queueitem_s queueitem_s;
struct queueitem_s
{
	queueitem_s *next;   /* next item */
	Any          obj;    /* the object */
};

struct TQueue
{
	CONTAINER_HEAD;
	long         size;   /* max items */
	queueitem_s *first;  /* first item */
	queueitem_s *last;   /* last item */
};


/** QueueIs
	Whether the object is a Queue.

	Always use this macro to test whether or not an object is a Queue.
	In the future, if we ever decide to let the Queue class be suclassed then the
	macro will change to C{ObjectInstOf (o, QueueClass)}.
*/
#define QueueIs(o) (ClassOf (o) == QueueClass)


/** QueueFlag
	Flags used by Queue objects.

	Never set/unset these flags directly.
	Use the available flag accessors instead.

	See ={QueueIsFifo}.
*/
#define QueueFlagPROTECTED ContainerFlagPROTECTED /* protected by program level */
#define QueueFlagFROZEN    ContainerFlagFROZEN    /* immutable? */
#define QueueFlagFIFO      Flag10                 /* first in first out */
#define QueueFlagLIFO      Flag11                 /* last in last out */


/** QueueIsFrozen QueueFreeze QueueUnfreeze QueueIsFixed
	QueueIsFrozen checks whether or not the Queue is immutable.

	QueueFreeze must only be used after you have created a new Queue and
	don't want the Queue to be modified by others.

	QueueUnfreeze must only be used on an Queue you have created.
	Normally, to modify the Queue then freeze it again.
*/
#define QueueIsFrozen(self)  ContainerIsFrozen (self)
#define QueueFreeze(self)    ContainerFreeze   (self)
#define QueueUnfreeze(self)  ContainerUnfreeze (self)


/** QueueIsFifo
	Whether this is a FIFO queue (LIFO otherwise).

	You may only query the QueueFlagLIFO and QueueFlagFIFO flags.
	Setting/unsetting may only be done by Queue.c.
*/
#define QueueIsFifo(self) FlagTest ((self), QueueFlagFIFO)


/** QueueNew
	Create a new Queue.
*/
extern Queue QueueNew (long flags, long size);
#define QueueNewFifo(size) QueueNew(QueueFlagFIFO, size)
#define QueueNewLifo(size) QueueNew(QueueFlagLIFO, size)


/** QueueClear
	Remove everything from the Queue.

	Throws if the Queue is immutable.
*/
extern Queue QueueClear (Queue self);


/** QueueCopy
	Make a shallow copy.

	This function always succeed.
*/
extern Queue QueueCopy (Queue self);


/** QueuePop
	Pop item from the Queue.

	Throws if the Queue is empty or immutable.
*/
extern Any QueuePop (Queue self);


/** QueuePush
	Push an object into the queue.

	Throws if the hard limit is reached or the Queue is immutable.
*/
extern Queue QueuePush (Queue self, Any x);


/** QueuePutBack
	Put back an object into the queue.

	No questions asked.
	Always succeed except if the hard limit is reached.
*/
extern Queue QueuePutBack (Queue self, Any x);


/** QueueRotate
	Rotate the items.

	This function always succeed.
*/
extern Queue QueueRotate (Queue self);


/** QueuePrint
	Print the Queue to v{stdout}.

	Throws on failure.
*/
extern Null QueuePrint (Queue self);


/** QueueString
	Get a String representation.

	This function always succeed.
*/
extern String QueueString (Queue self);


/** QueueTrue
	Whether or not the Queue is empty.
*/
#define QueueTrue(self) ((self)->len != 0)



/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Queue class. */
extern void QueueSetup ();
extern void QueueInit  ();


#endif /*_H_Queue_*/
