/* Rational.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Rational_
#define _H_Rational_ 1


/** Rational
	This is the final Rational class.
*/
extern Class RationalClass;


/** TRational
	This represents an instance of the Rational class.

	The denominator is never 0 and the sign is carried by the numerator.
*/
struct TRational
{
	OBJ_HEAD;
	Integer num;  /* numerator - carries the sign */
	Integer den;  /* denominator */
};


/** RationalPredefined
	Predefined system Rational objects.
*/
extern Rational Rational0;  /*  0 / 1 */
extern Rational Rational1;  /*  1 / 1 */
extern Rational Rational_1; /* -1 / 1 */


/** RationalNew
	Create a new Rational object.

	You may not pass 0 to v{den} otherwise EValue is thrown and NULL is returned.

	RationalNewEx returns an Integer if the numerator is an exact multiplication
	of the denominator, a Rational otherwise.
*/
extern Rational RationalNewl           (long num, long den);
extern Rational RationalNewr           (Integer num, Integer den);
extern Rational RationalNew1           (Integer num);
extern Rational RationalNew_1          (Integer den);
extern Real     RationalNewEx          (Integer num, Integer den);
extern Rational RationalNewd           (double v);
#define         RationalNewInt(x)      RationalNew1 ((Integer) x)
#define         RationalNewLong(x)     RationalNew1 ((Integer) x)
#define         RationalNewRational(x) (x)
#define         RationalNewFloat(x)    RationalNewd ((x)->value)
#define         RationalNewBig(x)      BigToRational (x)
#define         RationalNewComplex     ComplexToRational (x)
#define         RationalNewInteger(x)  RationalNew1 (x)
#define         RationalNewDecimal(x)  DecimalToRational (x)
#define         RationalNewReal(x)     RealToRational (x)
#define         RationalNewNumber(x)   NumberToRational (x)
#define         RationalNewBoolean(x)  (((x) == True) ? Rational1 : Rational0)
extern Rational RationalNewString      (String x);


/** RationalStatic
	Create a system static Rational object, i.e. not in GC.

	@warn
	The denominator may not be zero. No checks are made!
*/
extern Rational RationalStatic   (Integer n, Integer d);
extern Rational RationalStaticl  (long n, long d);
extern Rational RationalToStatic (Rational x);


/** RationalTo
	Conversions.

	For C type conversions, errno is set to 0 on success, non-zero otherwise.

	For object conversions, NULL is returned and an excpetion thrown on failure.
	Only RationalToInt and RationalToFloat can fail. The others always suceed.
*/
#define                   RationalToh(x)  Number_ltoh   (RationalTol (x))
#define                   RationalTouh(x) Number_ultouh (RationalToul (x))
#define                   RationalToi(x)  Number_ltoi   (RationalTol (x))
#define                   RationalToui(x) Number_ultoui (RationalToul (x))
#define                   RationalToli(x) Number_ltoli  (RationalTol (x))
extern long               RationalTol     (Rational x);
extern unsigned long      RationalToul    (Rational x);
extern float              RationalTof     (Rational x);
extern double             RationalTod     (Rational x);
#ifdef HAVE_LONG_LONG
extern long long          RationalToll    (Rational x);
extern unsigned long long RationalToull   (Rational x);
#endif
#ifdef HAVE_LONG_DOUBLE
extern long double        RationalTold    (Rational x);
#endif

extern Int     RationalToInt          (Rational x);
extern Long    RationalToLong         (Rational x);
#define        RationalToRational(x)  (x)
#define        RationalToFloat(x)     FloatNewRational (x)
#define        RationalToBig(x)       BigNewRational(x)
extern Integer RationalToInteger      (Rational x);
extern Decimal RationalToDecimal      (Rational x);
#define        RationalToReal(x)      ((Real) (x))
#define        RationalToNumber(x)    ((Number) (x))
#define        RationalToComplex(x)   ComplexNew1 ((Real) (x))
extern String  RationalToString       (Rational x);
#define        RationalToBoolean(x)   (RationalIsZero (x) ? False : True)


/** RationalSign
	RationalSign returns -1, 0, 1 if the value is negative, 0, positive respectively.
*/
#define RationalSign(x) IntegerSign ((x)->num)


/** RationalIs
	Identity functions.

	RationalIs tests whether or not an object is a Rational.

	RationalIsNeg returnss 1 if the value is negative.
	Returns 0 otherwise.

	RationalIsZero return 1 if the value is zero.
	Returns 0 otherwise.

	RationalIsOne returns 1 if the value is 1 or -1.
	Returns 0 otherwise.

	RationalIsWhole returns 1 if there is no fraction part.
	Returns 0 otherwise.

	RationalIsFrac returns 1 if 0 < |x| < 1.
	Returns 0 otherwise.

	RationalIsOdd returns 1 if the number is whole and odd.
	Returns 0 otherwise.
*/
#define    RationalIs(x)      (ClassOf (x) == RationalClass)
#define    RationalIsNeg(x)   IntegerIsNeg ((x)->num)
#define    RationalIsZero(x)  IntegerIsZero ((x)->num)
#define    RationalIsWhole(x) ((x)->den == (Integer) Int1)
extern int RationalIsFrac     (Rational x);
extern int RationalIsOne      (Rational x);
extern int RationalIs1        (Rational x);
extern int RationalIsOdd      (Rational x);


/** RationalUnary
	Unary functions.

	RationalPos returns the number itself, obviously.

	RationalNeg returns a negated value.
	The result is a Rational and always succeed.

	RationalIncr returns the Rational incremented by 1.
	The result is a Rational or NULL on failure.

	RationalDecr returns the Rational decremented by 1.
	The result is a Rational or NULL on failure.
*/
#define         RationalPos(x) (x)
extern Rational RationalNeg    (Rational self);
extern Real     RationalIncr   (Rational self);
extern Real     RationalDecr   (Rational self);


/** RationalEq
	Whether the value of a Rational is the same as the value of another Number object.
	All return 1 or 0.
*/
extern int RationalEqInt           (Rational x, Int y);
extern int RationalEqLong          (Rational x, Long y);
extern int RationalEqRational      (Rational x, Rational y);
#define    RationalEqFloat(x, y)   FloatEqRational (y, x)
#define    RationalEqBig(x, y)     BigEqRational (y, x)
#define    RationalEqComplex(x, y) ComplexEqRational (y, x)
#define    RationalEqInf(x, y)     (0)
extern int RationalEqInteger       (Rational x, Integer y);
extern int RationalEqDecimal       (Rational x, Decimal y);
extern int RationalEqReal          (Rational x, Real y);
extern int RationalEqNumber        (Rational x, Number y);



/** RationalLike
	Whether the absolute value of a Rational is the same as the absolute value of another
	Number object.
	All return int 1 or 0.
*/
extern int RationalLikeInt           (Rational x, Int y);
extern int RationalLikeLong          (Rational x, Long y);
extern int RationalLikeRational      (Rational x, Rational y);
#define    RationalLikeFloat(x, y)   FloatLikeRational (y, x)
#define    RationalLikeBig(x, y)     BigLikeRational (y, x)
#define    RationalLikeComplex(x, y) ComplexLikeRational (y, x)
#define    RationalLikeInf(x, y)     (0)
extern int RationalLikeInteger       (Rational x, Integer y);
extern int RationalLikeDecimal       (Rational x, Decimal y);
extern int RationalLikeReal          (Rational x, Real y);
extern int RationalLikeNumber        (Rational x, Number y);


/** RationalCmp
	Compare a Rational with another Number object.
	All return -1, 0, 1 in the usual maner.
*/
extern int RationalCmpInt           (Rational x, Int y);
extern int RationalCmpLong          (Rational x, Long y);
extern int RationalCmpRational      (Rational x, Rational y);
#define    RationalCmpFloat(x, y)   (-FloatCmpRational (y, x))
#define    RationalCmpBig(x, y)     (-BigCmpRational (y, x))
#define    RationalCmpComplex(x, y) (-ComplexCmpRational (y, x))
#define    RationalCmpInf(x, y)     (-InfCmpRational (y, x))
extern int RationalCmpInteger       (Rational x, Integer y);
extern int RationalCmpDecimal       (Rational x, Decimal y);
extern int RationalCmpReal          (Rational x, Real y);
extern int RationalCmpNumber        (Rational x, Number y);


/** RationalCmpAbs
	Compare the absolute value of a Rational with the absolute value of another
	Number object.
	All return -1, 0, 1 in the usual maner.
*/
extern int RationalCmpAbsInt           (Rational x, Int y);
extern int RationalCmpAbsLong          (Rational x, Long y);
extern int RationalCmpAbsRational      (Rational x, Rational y);
#define    RationalCmpAbsFloat(x, y)   (-FloatCmpAbsRational (y, x))
#define    RationalCmpAbsBig(x, y)     (-BigCmpAbsRational (y, x))
#define    RationalCmpAbsComplex(x, y) (-ComplexCmpAbsRational (y, x))
#define    RationalCmpAbsInf(x, y)     (-InfCmpAbsRational (y, x))
extern int RationalCmpAbsInteger       (Rational x, Integer y);
extern int RationalCmpAbsDecimal       (Rational x, Decimal y);
extern int RationalCmpAbsReal          (Rational x, Real y);
extern int RationalCmpAbsNumber        (Rational x, Number y);


/** RationalPlus
	Add a Rational with another Number object.
*/
extern Real   RationalPlusInt           (Rational x, Int y);
extern Real   RationalPlusLong          (Rational x, Long  y);
extern Real   RationalPlusRational      (Rational x, Rational y);
#define       RationalPlusFloat(x, y)   FloatPlusRational (y, x)
#define       RationalPlusBig(x, y)     BigPlusRational (y, x)
#define       RationalPlusComplex(x, y) ComplexPlusRational (y, x)
#define       RationalPlusInf(x, y)     InfPlusRational (y, x)
extern Real   RationalPlusInteger       (Rational x, Integer y);
extern Real   RationalPlusDecimal       (Rational x, Decimal y);
extern Real   RationalPlusReal          (Rational x, Real y);
extern Number RationalPlusNumber        (Rational x, Number y);


/** RationalMinus
	Subtract a Rational with another Number object.
*/
extern Real   RationalMinusInt           (Rational x, Int y);
extern Real   RationalMinusLong          (Rational x, Long y);
extern Real   RationalMinusRational      (Rational x, Rational y);
#define       RationalMinusFloat(x, y)   FloatRminusRational (y, x)
#define       RationalMinusBig(x, y)     BigRminusRational (y, x)
#define       RationalMinusComplex(x, y) ComplexRminusRational (y, x)
#define       RationalMinusInf(x, y)     InfPlusRational (InfNeg (y), x)
extern Real   RationalMinusInteger       (Rational x, Integer y);
extern Real   RationalMinusDecimal       (Rational x, Decimal y);
extern Real   RationalMinusReal          (Rational x, Real y);
extern Number RationalMinusNumber        (Rational x, Number y);


/** RationalRminus
	Subtract another with a Rational Number object.
	This is only for efficiency.
*/
extern Real RationalRminusInt   (Rational x, Int y);
extern Real RationalRminusLong  (Rational x, Long y);


/** RationalMul
	Multiply a Rational with another Number object.
*/
extern Real   RationalMulInt           (Rational x, Int y);
extern Real   RationalMulLong          (Rational x, Long y);
extern Real   RationalMulRational      (Rational x, Rational y);
#define       RationalMulFloat(x, y)   FloatMulRational (y, x)
#define       RationalMulBig(x, y)     BigMulRational (y, x)
#define       RationalMulComplex(x, y) ComplexMulRational (y, x)
#define       RationalMulInf(x, y)     InfMulRational (y, x)
extern Real   RationalMulInteger       (Rational x, Integer y);
extern Real   RationalMulDecimal       (Rational x, Decimal y);
extern Real   RationalMulReal          (Rational x, Real y);
extern Number RationalMulNumber        (Rational x, Number y);


/** RationalDiv
	Divide a Rational with another Number object.
*/
extern Real   RationalDivInt           (Rational x, Int y);
extern Real   RationalDivLong          (Rational x, Long y);
extern Real   RationalDivRational      (Rational x, Rational y);
#define       RationalDivFloat(x, y)   FloatRdivRational (y, x)
#define       RationalDivBig(x, y)     BigRdivRational (y, x)
#define       RationalDivComplex(x, y) ComplexRdivRational (y, x)
extern Int    RationalDivInf           (Rational x, Inf y);
extern Real   RationalDivInteger       (Rational x, Integer y);
extern Real   RationalDivDecimal       (Rational x, Decimal y);
extern Real   RationalDivReal          (Rational x, Real y);
extern Number RationalDivNumber        (Rational x, Number y);


/** RationalRdiv
	Divide another Number object with a Rational.
	This is only for efficiency.
*/
extern Real RationalRdivInt  (Rational x, Int y);
extern Real RationalRdivLong (Rational x, Long y);


/** RationalIdiv
	Integer division of a Rational with another Number object.
*/
extern Integer RationalIdivInt           (Rational x, Int y);
extern Integer RationalIdivLong          (Rational x, Long y);
extern Integer RationalIdivRational      (Rational x, Rational y);
#define        RationalIdivFloat(x, y)   FloatRidivRational (y, x)
#define        RationalIdivBig(x, y)     BigRidivRational (y, x)
#define        RationalIdivComplex(x, y) ComplexRidivRational (y, x)
extern Integer RationalIdivInf           (Rational x, Inf y);
extern Integer RationalIdivInteger       (Rational x, Integer y);
extern Integer RationalIdivDecimal       (Rational x, Decimal y);
extern Integer RationalIdivReal          (Rational x, Real y);
extern Integer RationalIdivNumber        (Rational x, Number y);


/** RationalRidiv
	Integer division of another Number object with a Rational.
	This is only for efficiency.
*/
extern Integer RationalRidivInt  (Rational x, Int y);
extern Integer RationalRidivLong (Rational x, Long y);


/** RationalRem
	Remainder of dividing a Rational with another Number object.
*/
extern Real     RationalRemInt           (Rational x, Int y);
extern Real     RationalRemLong          (Rational x, Long y);
extern Real     RationalRemRational      (Rational x, Rational y);
#define         RationalRemFloat(x, y)   FloatRremRational (y, x)
#define         RationalRemBig(x, y)     BigRremRational (y, x)
#define         RationalRemComplex(x, y) ComplexRremRational (y, x)
extern Rational RationalRemInf           (Rational x, Inf y);
extern Real     RationalRemInteger       (Rational x, Integer y);
extern Real     RationalRemDecimal       (Rational x, Decimal y);
extern Real     RationalRemReal          (Rational x, Real y);
extern Number   RationalRemNumber        (Rational x, Number y);


/** RationalRrem
	Remainder of dividing another Number object with a Rational.
	This is only for efficiency.
*/
extern Real RationalRremInt  (Rational x, Int y);
extern Real RationalRremLong (Rational x, Long y);


/** RationalPow
	Raise a Rational to the power of another Number object.
*/
extern Real   RationalPowInt           (Rational x, Int y);
extern Real   RationalPowLong          (Rational x, Long y);
extern Number RationalPowRational      (Rational x, Rational y);
#define       RationalPowFloat(x, y)   FloatRpowRational (y, x)
#define       RationalPowBig(x, y)     BigRpowRational (y, x)
#define       RationalPowComplex(x, y) ComplexRpowRational (y, x)
extern Real   RationalPowInf           (Rational x, Inf y);
extern Real   RationalPowInteger       (Rational x, Integer y);
extern Real   RationalPowDecimal       (Rational x, Decimal y);
extern Real   RationalPowReal          (Rational x, Real y);
extern Number RationalPowNumber        (Rational x, Number y);


/** RationalRpow
	Raise another Number object to the power of a Rational.
	This is only for efficiency.
*/
extern Number RationalRpowInt  (Rational x, Int y);
extern Number RationalRpowLong (Rational x, Long y);


/** RationalRound
	Rounding routines.

	RationalCeil rounds to the nearest integer towards positive infinity.
	Thus it is possible to get an EInf.

	RationalFloor rounds to the nearest integer towards negative infinity.
	Thus it is possible to get an EInf.

	RationalTrunc round the number towards zero, that is, it removes the
	fraction part. This function always succeed.

	RationalCeilEx, RationalFloorEx, and RationalTruncEx do the same but return
	a Rational instead of an Integer.
*/
extern Rational RationalCeil  (Rational x);
extern Rational RationalFloor (Rational x);
extern Rational RationalTrunc (Rational x);
extern Rational RationalRound (Rational x, int prec);


/** RationalFrac
	Return the fraction part.
	The result is exact and always succeed.
*/
extern Rational RationalFrac (Rational x);


/** RationalPrint
	Print a Rational to v{stdout}.

	Note that this might throw an exception because writing to v{stdout} may
	fail for various reasons.
*/
extern Null RationalPrint (Rational self);


/** RationalFormat
	Format a Rational.

	See ={Format} for details.
*/
extern String RationalFormat (Rational x, format_s *f);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Rational class */
extern void RationalSetup ();
extern void RationalInit  ();

/* for interp.c and Sys.c */
extern Rational RationalNewRatio (Any x, Any y);

#endif /*_H_Rational_*/
