/* Real.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Real_
#define _H_Real_ 1


/** Real
	This is the abstract parent class of ={Integer},  ={Decimal} and ={Inf}.
*/
extern Class RealClass;


/** RealPredefined
	Predefined system objects.
*/
#define Real0  ((Real) Int0)   /*  0 */
#define Real1  ((Real) Int1)   /*  1 */
#define Real2  ((Real) Int2)   /*  2 */
#define Real_1 ((Real) Int_1)  /* -1 */
#define Real_2 ((Real) Int_2)  /* -2 */


/** RealNew
	Create a Real object from C types or other objects.

	These are just convenient wrappers around ={IntegerNew} and ={DecimalNew}.
*/
#define RealNewh(v)   ((Real) IntegerNewh   (v))
#define RealNewuh(v)  ((Real) IntegerNewuh  (v))
#define RealNewi(v)   ((Real) IntegerNewi   (v))
#define RealNewui(v)  ((Real) IntegerNewui  (v))
#define RealNewl(v)   ((Real) IntegerNewl   (v))
#define RealNewul(v)  ((Real) IntegerNewul  (v))
#define RealNewli(v)  ((Real) IntegerNewli  (v))
#define RealNewd(v)   ((Real) DecimalNewd   (v))
#ifdef HAVE_LONG_LONG
#define RealNewll(v)  ((Real) IntegerNewll  (v))
#define RealNewull(v) ((Real) IntegerNewull (v))
#endif
#ifdef HAVE_LONG_DOUBLE
#define RealNewld(v)  ((Real) DecimalNewld (v))
#endif

#define     RealNewInt(x)      ((Real) (x))
#define     RealNewFloat(x)    ((Real) (x))
#define     RealNewRational(x) ((Real) (x))
#define     RealNewBig(x)      ((Real) (x))
#define     RealNewComplex(x)  ComplexToReal (x)
#define     RealNewInf(x)      ((Real) (x))
#define     RealNewInteger(x)  ((Real) (x))
#define     RealNewDecimal(x)  ((Real) (x))
#define     RealNewReal(x)     (x)
extern Real RealNewNumber      (Number x);


/** RealTo
	Functions to convert any object into C types or other Real objects.

	For C type conversions, the functions set v{errno} to zero on success.
	On failure, v{errno} is set to EDOM if v{x} is conversion is not possible,
	to ERANGE if the value falls out of the C type range.
	The I{unsigned} versions fail with ERANGE if the value is negative.

	For object conversions, an exception is thrown and NULL is returned on failure.
*/
extern short              RealToh   (Any x);
extern unsigned short     RealTouh  (Any x);
extern int                RealToi   (Any x);
extern unsigned int       RealToui  (Any x);
extern long               RealTol   (Any x);
extern unsigned long      RealToul  (Any x);
extern long               RealToli  (Any x);
extern float              RealTof   (Any x);
extern double             RealTod   (Any x);
#ifdef HAVE_LONG_LONG
extern long long          RealToll  (Any x);
extern unsigned long long RealToull (Any x);
#endif
#ifdef HAVE_LONG_DOUBLE
extern long double        RealTold  (Any x);
#endif

extern Int      RealToInt       (Any x);
extern Float    RealToFloat     (Any x);
extern Long     RealToLong      (Any x);
extern Rational RealToRational  (Any x);
extern Big      RealToBig       (Any x);
extern Complex  RealToComplex   (Any x);
extern Integer  RealToInteger   (Any x);
extern Decimal  RealToDecimal   (Any x);
extern Real     RealToReal      (Any x);
extern Number   RealToNumber    (Any x);


/** RealSign
	RealSign returns -1, 0, 1 if the value is negative, 0, positive respectively.

	This is just a convenient wrapper around ={IntegerSign} ={DecimalSign} and ={InfSign}.
*/
extern int RealSign (Real x);


/** RealIs
	Identity functions.

	RealIs tests whether an object is a Real.

	RealIsNeg returns 1 if the value is negative.
	Returns 0 otherwise.

	RealIsZero returns 1 if the value is 0.
	Returns 0 otherwise.

	RealIsOne returns 1 if the value is 1 or -1.
	Returns 0 otherwise.

	RealIs1 return 1 if the value is 1.
	Returns 0 otherwise.

	RealIsWhole return 1 if the value is a whole number.
	Returns 0 otherwise.

	RealIsFrac return 1 if 0 < |x| < 1.
	Returns 0 otherwise.

	RealIsOdd return 1 if the value is a whole and odd number.
	Returns 0 otherwise.

	These are just convenient wrappers around
	={IntegerIs} ={DecimalIs} and ={InfIs}.
*/
extern int RealIs      (Any x);
extern int RealIsZero  (Real x);
extern int RealIsNeg   (Real x);
extern int RealIsOne   (Real x);
extern int RealIs1     (Real x);
extern int RealIsWhole (Real x);
extern int RealIsFrac  (Real x);
extern int RealIsOdd   (Real x);


/** RealUnary
	Unary functions.

	RealPos returns the number itself, obviously.

	RealNeg returns a negated value.

	RealIncr returns the number incremented by 1.

	RealDecr returns the number decremented by 1.

	These are just convenient wrappers around
	={IntegerPos} ={IntegerNeg} ={IntegerIncr} ={IntegerDecr}
	={DecimalPos} ={DecimalNeg} ={DecimalIncr} ={DecimalDecr}
	={InfPos}     ={InfNeg}     ={InfIncr}     ={InfDecr}
*/
extern Real RealPos  (Real x);
extern Real RealNeg  (Real x);
extern Real RealIncr (Real x);
extern Real RealDecr (Real x);


/** RealEq
	Whether the value of a Real is the same as the value of another Number object.
	All return 0 or 1.

	These are just convenient wrappers around ={IntegerEq} ={DecimalEq} and ={InfEq}.
*/
extern int RealEqInt      (Real x, Int y);
extern int RealEqFloat    (Real x, Float y);
extern int RealEqLong     (Real x, Long y);
extern int RealEqRational (Real x, Rational y);
extern int RealEqBig      (Real x, Big y);
extern int RealEqComplex  (Real x, Complex y);
extern int RealEqInf      (Real x, Inf y);
extern int RealEqInteger  (Real x, Integer y);
extern int RealEqDecimal  (Real x, Decimal y);
extern int RealEqReal     (Real x, Real y);
extern int RealEqNumber   (Real x, Number y);


/** RealLike
	Whether the absolute value of a Real is the same as the absolute value of another
	Number object.
	All return 0 or 1.

	These are just convenient wrappers around ={IntegerLike} ={DecimalLike} and ={InfLike}.
*/
extern int RealLikeInt      (Real x, Int y);
extern int RealLikeFloat    (Real x, Float y);
extern int RealLikeLong     (Real x, Long y);
extern int RealLikeRational (Real x, Rational y);
extern int RealLikeBig      (Real x, Big y);
extern int RealLikeComplex  (Real x, Complex y);
extern int RealLikeInf      (Real x, Inf y);
extern int RealLikeInteger  (Real x, Integer y);
extern int RealLikeDecimal  (Real x, Decimal y);
extern int RealLikeReal     (Real x, Real y);
extern int RealLikeNumber   (Real x, Number y);


/** RealCmp
	Compare a Real with another Number object.
	All return -1, 0, or -1 in the usual manner.

	These are just convenient wrappers around ={IntegerCmp} ={DecimalCmp} and ={InfCmp}.
*/
extern int RealCmpInt      (Real x, Int y);
extern int RealCmpFloat    (Real x, Float y);
extern int RealCmpLong     (Real x, Long y);
extern int RealCmpRational (Real x, Rational y);
extern int RealCmpBig      (Real x, Big y);
extern int RealCmpComplex  (Real x, Complex y);
extern int RealCmpInf      (Real x, Inf y);
extern int RealCmpInteger  (Real x, Integer y);
extern int RealCmpDecimal  (Real x, Decimal y);
extern int RealCmpReal     (Real x, Real y);
extern int RealCmpNumber   (Real x, Number y);


/** RealCmpAbs
	Compare the absolute value of a Real with the absolute value of another
	Number object.
	All return -1, 0, or -1 in the usual manner.

	These are just convenient wrappers around ={IntegerCmpAbs} ={DecimalCmpAbs} and ={InfCmpAbs}.
*/
extern int RealCmpAbsInt      (Real x, Int y);
extern int RealCmpAbsFloat    (Real x, Float y);
extern int RealCmpAbsLong     (Real x, Long y);
extern int RealCmpAbsRational (Real x, Rational y);
extern int RealCmpAbsBig      (Real x, Big y);
extern int RealCmpAbsComplex  (Real x, Complex y);
extern int RealCmpAbsInf      (Real x, Inf y);
extern int RealCmpAbsInteger  (Real x, Integer y);
extern int RealCmpAbsDecimal  (Real x, Decimal y);
extern int RealCmpAbsReal     (Real x, Real y);
extern int RealCmpAbsNumber   (Real x, Number y);


/** RealPlus
	Add a Real with another Number object.

	These are just convenient wrappers around ={IntegerPlus} ={DecimalPlus} and ={InfPlus}.
*/
extern Real    RealPlusInt      (Real x, Int y);
extern Real    RealPlusFloat    (Real x, Float y);
extern Real    RealPlusLong     (Real x, Long y);
extern Real    RealPlusRational (Real x, Rational y);
extern Real    RealPlusBig      (Real x, Big y);
extern Complex RealPlusComplex  (Real x, Complex y);
extern Real    RealPlusInf      (Real x, Inf y);
extern Real    RealPlusInteger  (Real x, Integer y);
extern Real    RealPlusDecimal  (Real x, Decimal y);
extern Real    RealPlusReal     (Real x, Real y);
extern Number  RealPlusNumber   (Real x, Number y);


/** RealMinus
	Subtract a Real with another Number object.

	These are just convenient wrappers around ={IntegerMinus} ={DecimalMinus} and ={InfMinus}.
*/
extern Real    RealMinusInt      (Real x, Int y);
extern Real    RealMinusFloat    (Real x, Float y);
extern Real    RealMinusLong     (Real x, Long y);
extern Real    RealMinusRational (Real x, Rational y);
extern Real    RealMinusBig      (Real x, Big y);
extern Complex RealMinusComplex  (Real x, Complex y);
extern Real    RealMinusInf      (Real x, Inf y);
extern Real    RealMinusInteger  (Real x, Integer y);
extern Real    RealMinusDecimal  (Real x, Decimal y);
extern Real    RealMinusReal     (Real x, Real y);
extern Number  RealMinusNumber   (Real x, Number y);


/** RealMul
	Multiply a Real with anothe Number objectr.

	These are just convenient wrappers around ={IntegerMul} ={DecimalMul} and ={InfMul}.
*/
extern Real    RealMulInt      (Real x, Int y);
extern Real    RealMulFloat    (Real x, Float y);
extern Real    RealMulLong     (Real x, Long y);
extern Real    RealMulRational (Real x, Rational y);
extern Real    RealMulBig      (Real x, Big y);
extern Complex RealMulComplex  (Real x, Complex y);
extern Real    RealMulInf      (Real x, Inf y);
extern Real    RealMulInteger  (Real x, Integer y);
extern Real    RealMulDecimal  (Real x, Decimal y);
extern Real    RealMulReal     (Real x, Real y);
extern Number  RealMulNumber   (Real x, Number y);


/** RealDiv
	Divide a Real with another Number object.

	These are just convenient wrappers around ={IntegerDiv} ={DecimalDiv} and ={InfDiv}.
*/
extern Real    RealDivInt      (Real x, Int y);
extern Real    RealDivFloat    (Real x, Float y);
extern Real    RealDivLong     (Real x, Long y);
extern Real    RealDivRational (Real x, Rational y);
extern Real    RealDivBig      (Real x, Big y);
extern Complex RealDivComplex  (Real x, Complex y);
extern Real    RealDivInf      (Real x, Inf y);
extern Real    RealDivInteger  (Real x, Integer y);
extern Real    RealDivDecimal  (Real x, Decimal y);
extern Real    RealDivReal     (Real x, Real y);
extern Number  RealDivNumber   (Real x, Number y);


/** RealDivRem
	Divide a Real with another Number object returning the result and remainder.

	These are just convenient wrappers around
	={IntegerDivRem} ={DecimalDivRem} and ={InfDivRem}.
*/
extern Real    RealDivRemInt      (Real x, Int      y, Number *r);
extern Real    RealDivRemFloat    (Real x, Float    y, Number *r);
extern Real    RealDivRemLong     (Real x, Long     y, Number *r);
extern Real    RealDivRemRational (Real x, Rational y, Number *r);
extern Real    RealDivRemBig      (Real x, Big      y, Number *r);
extern Complex RealDivRemComplex  (Real x, Complex  y, Number *r);
extern Real    RealDivRemInf      (Real x, Inf      y, Number *r);
extern Real    RealDivRemInteger  (Real x, Integer  y, Number *r);
extern Real    RealDivRemDecimal  (Real x, Decimal  y, Number *r);
extern Real    RealDivRemReal     (Real x, Real     y, Number *r);
extern Number  RealDivRemNumber   (Real x, Number   y, Number *r);


/** RealIdiv
	Integer division of a Real with another Number object.

	These are just convenient wrappers around ={IntegerIdiv} ={DecimalIdiv} and ={InfIdiv}.
*/
extern Integer RealIdivInt      (Real x, Int y);
extern Integer RealIdivFloat    (Real x, Float y);
extern Integer RealIdivLong     (Real x, Long y);
extern Integer RealIdivRational (Real x, Rational y);
extern Integer RealIdivBig      (Real x, Big y);
extern Integer RealIdivComplex  (Real x, Complex y);
extern Integer RealIdivInf      (Real x, Inf y);
extern Integer RealIdivInteger  (Real x, Integer y);
extern Integer RealIdivDecimal  (Real x, Decimal y);
extern Integer RealIdivReal     (Real x, Real y);
extern Integer RealIdivNumber   (Real x, Number y);


/** RealIdivRem
	Integer division of a Real with another Number object returning the result and remainder.

	These are just convenient wrappers around
	={IntegerIdivRem} ={DecimalIdivRem} and ={InfIdivRem}.
*/
extern Integer RealIdivRemInt      (Real x, Int      y, Number *r);
extern Integer RealIdivRemFloat    (Real x, Float    y, Number *r);
extern Integer RealIdivRemLong     (Real x, Long     y, Number *r);
extern Integer RealIdivRemRational (Real x, Rational y, Number *r);
extern Integer RealIdivRemBig      (Real x, Big      y, Number *r);
extern Integer RealIdivRemComplex  (Real x, Complex  y, Number *r);
extern Integer RealIdivRemInf      (Real x, Inf      y, Number *r);
extern Integer RealIdivRemInteger  (Real x, Integer  y, Number *r);
extern Integer RealIdivRemDecimal  (Real x, Decimal  y, Number *r);
extern Integer RealIdivRemReal     (Real x, Real     y, Number *r);
extern Integer RealIdivRemNumber   (Real x, Number   y, Number *r);


/** RealRem
	Remainder of dividing a Real with another Number object.

	These are just convenient wrappers around ={IntegerRem} ={DecimalRem} and ={InfRem}.
*/
extern Real    RealRemInt      (Real x, Int y);
extern Real    RealRemFloat    (Real x, Float y);
extern Real    RealRemLong     (Real x, Long y);
extern Real    RealRemRational (Real x, Rational y);
extern Real    RealRemBig      (Real x, Big y);
extern Complex RealRemComplex  (Real x, Complex y);
extern Real    RealRemInf      (Real x, Inf y);
extern Real    RealRemInteger  (Real x, Integer y);
extern Real    RealRemDecimal  (Real x, Decimal y);
extern Real    RealRemReal     (Real x, Real y);
extern Number  RealRemNumber   (Real x, Number y);


/** RealPow
	Raise a Real to the power of another Number object.

	These are just convenient wrappers around ={IntegerPow} ={DecimalPow} and ={InfPow}.
*/
extern Real    RealPowInt      (Real x, Int y);
extern Real    RealPowFloat    (Real x, Float y);
extern Real    RealPowLong     (Real x, Long y);
extern Real    RealPowRational (Real x, Rational y);
extern Real    RealPowBig      (Real x, Big y);
extern Complex RealPowComplex  (Real x, Complex y);
extern Real    RealPowInf      (Real x, Inf y);
extern Real    RealPowInteger  (Real x, Integer y);
extern Real    RealPowDecimal  (Real x, Decimal y);
extern Real    RealPowReal     (Real x, Real y);
extern Number  RealPowNumber   (Real x, Number y);


/** RealRound
	Rounding functions.

	RealCeil round the number to the nearest integer towards positive infinity.
	EHuge is thrown on failure.

	RealFloor round the number to the nearest integer towards negative infinity.
	EHuge is thrown on failure.

	RealTrunc round the number towards zero, that is, it removes the
	fraction part. This function throws EMath if the number is an infinity.

	These are just convenient wrappers around ={IntegerRound} ={DecimalRound} and ={InfRound}.
*/
extern Real RealFloor (Real x);
extern Real RealCeil  (Real x);
extern Real RealTrunc (Real x);
extern Real RealFrac  (Real x);
extern Real RealRound (Real x, int prec);


/** RealFormat
	Format a Real.

	See ={Format} for details;

	This is just a convenient wrapper around ={IntegerFormat} ={DecimalFormat} and ={InfFormat}.
*/
extern String RealFormat (Real x, format_s *f);


/** RealPrint
	Print a Real to v{stdout}.

	This is just a convenient wrapper around ={IntegerPrint} ={DecimalPrint} and ={InfPrint}.
*/
extern Null RealPrint (Real x);


/** __Tiny__ __Huge__ __Inf__ __Zerodiv__
	__Tiny__ throws ETiny if the ={NumberContext} does not allow rounding
	to zero. You must supply an Integer or Decimal  zero value, i.e. Int0,
	Long0, Rational0, Float0 or Big0.

	__Huge__ throws EHuge if the ={NumberContext} does not allow infinities to
	be returned. v{sign} must be -1 for ={InfNegative}, anything else for ={InfPositive}.

	__Inf__ returns an infinity or throw EInf if the number context does not allow
	infinities to be returned. v{sign} must be -1 for ={InfNegative}, anything
	else for ={InfPositive}.

	__Zerodiv__ throws EDiv if the ={NumberContext} does not allow division
	by zero, returns an infinity otherwise.
*/
extern Real __Tiny__    (Real zero);
extern Inf  __Huge__    (int sign);
extern Inf  __Inf__     (int sign);
extern Inf  __Zerodiv__ (int sign);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Real class */
extern void RealSetup ();
extern void RealInit  ();

#endif /*_H_Real_*/
