/* Record.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Record_
#define _H_Record_ 1

#include "Container.h"


/** Record
	This is the final Record class.
*/
extern Class RecordClass;


/** TRecord
	This represents an instance of the Record class.

	See ={CONTAINER_HEAD}.
*/
struct TRecord
{
	CONTAINER_HEAD;
	Any    *objs;  /* values are allocated */
	Struct  struc; /* the associated Struct object */
};


/** RecordFlag
	Flags used by Record objects.

	You must not access these flags directly.
	Use the accessor macros provided instead.
*/
#define RecordFlagPROTECTED ContainerFlagPROTECTED /* protected by program level */
#define RecordFlagFROZEN    ContainerFlagFROZEN    /* immutable? */


/** RecordIsFrozen RecordFreeze RecordUnfreeze
	RecordIsFrozen checks whether or not the Record is immutable.

	RecordFreeze must only be used after you have created a new Record and
	don't want the Record to be modified by others.

	RecordUnfreeze must only be used on a Record you have created.
	Normally, to modify the Record then freeze it again.
*/
#define RecordIsFrozen(self)  ContainerIsFrozen (self)
#define RecordFreeze(self)    ContainerFreeze   (self)
#define RecordUnfreeze(self)  ContainerUnfreeze (self)


/** RecordIs
	Whether or not an object is a Record.
*/
#define RecordIs(x) (ClassOf(x) == RecordClass)


/** RecordOf
	Whether or not an object is a Record of a specified ={Struct}.

	@warn
	Make sure v{u} is a ={Struct}.
*/
#define RecordOf(x, u) (RecordIs (x) && ((Record) x)->struc == (u))


/** RecordNew
	Create a new Record object.

	@warn
	If you supply the wrong number of arguments, you get an EArgCount.
	If you supply the wrong type of arguments, you get EArgValue.

	RecordNewEx does not ask any question, so be careful.
*/
extern Record RecordNew   (Struct u, int argc, Any *argv);
extern Record RecordNewEx (Struct u, ...);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Record class */
extern void RecordSetup ();
extern void RecordInit  ();


#endif /*_H_Record_*/
