/* Set.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Set_
#define _H_Set_ 1

#include "Container.h"


/** Set
	This is the final Set class.
*/
extern Class SetClass;


/** TSet
	This represents an instance of the SetClass.

	The v{set_entry_s} struct is intentionally hidden.
	See C{Set.c} if you are curious.

	See also ={CONTAINER_HEAD}.
*/
typedef struct set_entry_s set_entry_s;
struct TSet
{
	CONTAINER_HEAD;
	long          mask;    /* current mask */
	Any           type;    /* type/validator, maybe 0 */
	set_entry_s **buckets; /* the buckets */
};


/** SetEMPTY
	This is a predefined system object frozen empty Set.

	C{const {}} in scripts simply return this object.
*/
extern Set SetEMPTY;


/** SetIs
	Whether or not the objet is a Set.

	Always use this macro to test whether or not an object is a Set.
	In the future, if we ever decide to let the Set class be suclassed then the
	macro will change to C{ObjectInstOf (o, SetClass)}.
*/
#define SetIs(o) (ClassOf(o) == SetClass)


/** SetFlag
	Flags used by Set objects.

	Never access these flags directly.
	Use the accessor macros provided instead.
*/
#define SetFlagPROTECTED ContainerFlagPROTECTED /* program level protected? */
#define SetFlagFROZEN    ContainerFlagFROZEN    /* mutable? */
#define SetFlagENUM      Flag10                 /* built from enum statement? */


/** SetIsFrozen
	Whether or not the Set is mutable.

	SetFreeze must only be used after you have created a new Set and
	don't want the Set to be modified by others.

	SetUnfreeze must only be used on an Set you have created.
	Normally, to modify the Set then freeze it again.
*/
#define SetIsFrozen(self)  ContainerIsFrozen (self)
#define SetFreeze(self)    ContainerFreeze   (self)
#define SetUnfreeze(self)  ContainerUnfreeze (self)


/** SetNew
	Create a new Set with an initial capacity.

	SetRoot creates a root Set object, i.e. it is always marked by the GC.

	SetStatic creates a static Set object, i.e. it is never marked by the GC,
	thus you must make sure that all of the items are also static objects.
	No checks are made.

	@warn
	Make sure v{len} <= #{GC_MAX_ITEMS}.
*/
extern Set SetNew    (long len);
extern Set SetRoot   (long len);
extern Set SetStatic (long len);


/** SetAdd SetAddIf SetAddz SetAddFloat
	Include an object in the Set.

	SetAdd obediently add the object. It is the responsibility of the caller
	to make sure that the object is not in the Set. Be careful. You must have
	good knowledge of the object's __eq method behavior.

	SetAddIf returns the same object in the Set if one exist otherwise add
	the object and return that object.

	SetAddIfEx does the same as SetAddIf but the object's class is compared
	for equality. This allows you to store, e.g. a Big and a Float with the same
	value as two different items.

	SetAddz is for efficiency. If the same string is already in the Set
	then it is returned, created otherwise. This may save creating a ={String}
	object. SetAddFloat does the same for a ={Float} object.

	@warn
	All of these functions return 0 if the set becomes too big.
*/
extern Any    SetAdd      (Set self, Any o);
extern Any    SetAddIf    (Set self, Any o);
extern Any    SetAddIfEx  (Set self, Any o);
extern String SetAddz     (Set self, const char *z, long len);
extern Float  SetAddFloat (Set self, double v);


/** SetWalk
	Walk through the objects.

	On first call v{b} and v{i} must be set to 0.

	@xmp
		long b = 0, i = 0;
		Any  o;
		while ((o = SetWalk (MyFooSet, &b, &i)) != 0)
		{
			... do something with o ...
		}
*/
extern Any SetWalk (Set self, long *b, long *i);


/** SetContains
	Whether the Set contains a specified object.
*/
extern int SetContains (Set self, Any x);


/** SetOf
	Set type of items.
*/
extern Set SetOf (Set self, Any type);


/** SetPush
	Push an item.
*/
extern Set SetPush (Set self, Any x);


/** SetDelete
	Remove an object from the Set.
*/
extern Set SetDelete (Set self, Any x);


/** SetClear
	Clear the Set.
*/
extern Set SetClear (Set self);


/** SetExchange
	Exchange the items in the two Sets.
*/
extern void SetExchange (Set self, Set x);


/** SetLike
	Whether two sets are alike.
*/
#define SetLike(self, x) (set__like (self, x) == True)


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Set class */
extern void SetSetup ();
extern void SetInit  ();

/* for Gc */
#ifdef QU_WITH_WEAK
extern int  SetAddWeak (Set self, Weak w);
extern void SetDelWeak (Set self, Weak w);
#endif

/* published for efficiency */
extern Set     Set_new   (Class self, int argc, Any *argv);
extern Boolean set__like (Set self, Any x);


#endif /*_H_Set_*/
