/* Stream.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Stream_
#define _H_Stream_ 1


/** Stream
	This the base class of all Stream classes (File, Socket, etc.).
*/
extern Class StreamClass;


/** StreamStdErr StreamStdIn StreamStdOut StreamNull
	These are the Stream objects of v{stderr}, v{stdin}, v{stdout}, and k{null}.
*/
extern Stream StreamStdErr;
extern Stream StreamStdIn;
extern Stream StreamStdOut;
extern Stream StreamNull;


/** fstream_s
	Stream functions.

	See ={TStream}.
*/
typedef	ssize_t (*stream_rdr_f)   (int fd, char *buf, size_t len);
typedef	ssize_t (*stream_wrt_f)   (int fd, const char *buf, size_t len);
typedef	ssize_t (*stream_ordr_f)  (Any obj, char *buf, size_t len);
typedef	ssize_t (*stream_owrt_f)  (Any obj, const char *buf, size_t len);
typedef	Stream  (*stream_acc_f)   (Any obj);
typedef	Null    (*stream_con_f)   (Any obj, Any addr);
typedef	ssize_t (*stream_seek_f)  (int fd, off_t pos, int whence);
typedef	ssize_t (*stream_oseek_f) (Any obj, off_t pos, int whence);

typedef struct fstream_s fstream_s;
struct fstream_s
{
	stream_rdr_f   rdr;   /* read bytes from descriptor */
	stream_wrt_f   wrt;   /* write bytes to descriptor */
	stream_ordr_f  ordr;  /* read bytes from object */
	stream_owrt_f  owrt;  /* write bytes to object */

	/* for the benefit of Socket/SSL */
	stream_acc_f   acc;   /* accept a new client Stream */
	stream_con_f   con;   /* connect to an address */

	stream_seek_f  seek;  /* move file pointer of descriptor */
	stream_oseek_f oseek; /* move file pointer of object */
};


/** StreamDefaultFuncs
	The default readers/writers.

	@args
	.rdr    read
	.wrt    write
	.ordr   internal, always return EBADF
	.owrt   -ditto-

	@warn
	Read-only! You must never change the content.
*/
extern fstream_s StreamDefaultFuncs;


/** TStream STREAM_HEAD
	This represents an instance of the Stream class.

	All Stream subclasses must ust STREAM_HEAD as the header instead of OBJ_HEAD.

	@warn
	The readers/writers must not take care of errors, including EINTR, EAGAIN, etc.
	Stream already takes care of those.

	Note that v{obj} must be a Qu object. It is marked by stream__mark.

	If v{obj} is NULL then rdr/wrt/snd/rcv are used, otherwise ordr/owrt/osnd/orcv
	are used thus they may not be NULL.
*/
#define STREAM_HEAD \
	OBJ_HEAD; \
	int        fd;       /* file descriptor  */ \
	String     br;       /* linebreak string, may not be 0 */ \
	double     timeout;  /* when to timeout  */ \
	fstream_s *funcs;    /* readers/writers */ \
	Any        obj;      /* object to pass to ordr/owrt/acc/con, may be NULL */ \
	long       sid       /* unique ID */

struct TStream
{
	STREAM_HEAD;
};


/** StreamFlag
	Flags used by Stream objects.

	Do not access these flags directly.
	Use the accessor macros provided instead.

	For C classes, these flags should be used directly when an instance is
	created, e.g. C{inst->flags = StreamFlagCANEOF | StreamFlagCANCLOSE}.
	Once created, the flags must never be modified.
*/
/* 0-9 reserved for Stream */
#define StreamFlagCANEOF    Flag0  /* EOF is relevant? */
#define StreamFlagCANSELECT Flag2  /* select is not relevant */
#define StreamFlagCANBLOCK  Flag3  /* can set blocking mode? */
#define StreamFlagCANCLOSE  Flag4  /* can be closed? */


/** StreamCanClose StreamCanBlock StreamCanEof StreamCanSelect
	StreamCanClose checks whether the Stream can be closed.
	If not then the f{close} method should do nothing.

	StreamCanBlock checks whether the Stream can be set/unset to blocking mode.
	If not then the f{blocking} method should do nothing.

	StreamCanEof checks whether the Stream has EOF (end of file) notion.
	This is used by StreamRead et.al. to stop immediately when zero bytes
	are read from the Stream. A socket does not have EOF. A regular file does.

	StreamCanSelect checks whether the Stream can be tested for readyness
	using the system f{select} call. If not then the Stream is assumed to be
	always ready.
*/
#define StreamCanClose(self)  FlagTest (self, StreamFlagCANCLOSE)
#define StreamCanBlock(self)  FlagTest (self, StreamFlagCANBLOCK)
#define StreamCanEof(self)    FlagTest (self, StreamFlagCANEOF)
#define StreamCanSelect(self) FlagTest (self, StreamFlagCANSELECT)


/** StreamIs
	Whether or not an object is a Stream.
*/
#define StreamIs(o) ObjectInstOf (o, StreamClass)


/** StreamNew
	Allocate a new Stream object.

	The following fields are set:
	@arg
	.fd      -1
	.br      StringBYTE ('\n')
	.buf     NULL
	.timeout maximum value of time_t
	.funcs   ={StreamDefaultFuncs}
	.obj     NULL.

	@warn
	If you later set v{obj} then you must also set v{ordr}, v{owrt}, v{osnd} and v{orcv}.
*/
extern Stream StreamNew (Class self);


/** StreamPrint
	These are called by the builtin -{print} functions.
	See ={SysPrint}.

	On success ={Nil} is returned, otherwise an exception is thrown and
	0 is returned.
*/
extern Null StreamPrints  (Stream self, String s);
extern Null StreamPrint   (Stream self, Any x);
extern Null StreamPrintln (Stream self, Any x);
extern Null StreamPrintz  (Stream self, const char *s, long len);


/** StreamSelect
	Get Stream status.

	Returns what the system f{select} function returns.

	As long as the Stream is valid, then the only possible v{errno} is
	ENOMEM. Note that EINTR is already handled by StreamSelect.

	v{timeout} is in microseconds. A negative value means to use the
	default value (1000 microsecs).

	v{how} is one of:
	@args
	.r ready for reading
	.w ready for writing
	.e has error

	@block
	Calling this prior to do a read/write may improve performance a great
	deal.

	@warn
	={StreamRead} and ={StreamWrite} already perform this so you must not
	do it yourself.

	f{StreamSelect} does not wrap itself in ={__Blocking}Begin/End. You must
	do it yourself if necessary.

	@block
	ENOMEM should have never been possible in the first place but ...
*/
extern int StreamSelect (Stream self, char how, long timeout);


/** StreamNow
	Return the current time since epoch as a double.
*/
extern double StreamNow ();


/** StreamRead
	Read from Stream.

	In any case v{len} must be <= SSIZE_MAX.

	StreamRead is the read primitive. It return the length read on success,
	throw and return -1 otherwise.

	StreamReadln read until linebreak or v{max}. It return the length read on success,
	throw and return -1 otherwise.

	StreamReads and StreamReadsln do the same but return a new String.

	StreamRecv and StreamRecvln do the same but read into a Buffer.
*/
extern long   StreamRead     (Stream self, char *buf, long len, int all);
extern long   StreamReadln   (Stream self, char *buf, long max);
extern String StreamReads    (Stream self, long n, int all);
extern String StreamReadsln  (Stream self, long max);
extern long   StreamRecv     (Stream self, Buffer b, long n, int all);
extern long   StreamRecvln   (Stream self, Buffer b, long max);


/** StreamWrite
	Write to Stream.

	In any case v{len} must be <= SSIZE_MAX.

	All return the number of bytes written on success, -1 on error.

	StreamWrite is the write primitive.
	StreamWriteln does the same then writes the linbreak.

	StreamWrites writes a String.
	StreamWritesln does the same then writes the linebreak.

	StreamSend does the same. It consumes the Buffer.
*/
extern long StreamWrite    (Stream self, const char *buf, long len, int all);
extern long StreamWriteln  (Stream self, const char *buf, long len);
extern long StreamWrites   (Stream self, String x, long ofs, long len, int all);
extern long StreamWritesln (Stream self, String x, long ofs, long len);
extern long StreamSend     (Stream self, Buffer x, long len, int all);


/** StreamTransfer
	StreamTransfer read v{len} bytes from self and write it to v{dst}.

	If v{len} is negative then read until self reaches EOF. Return 1 on success
	0 otherwise.
*/
extern int StreamTransfer (Stream self, Stream dst, long len);


/** StreamClose
	Close the Stream.
*/
extern Null StreamClose (Stream self);


/** StreamContains
	Keep reading from Stream until the string is found or EOF is reached or
	an exception is thrown.
*/
extern Boolean StreamContains (Stream self, String x);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize Stream. */
extern void StreamSetup ();
extern void StreamInit  ();

/* published for the benefit of C subclasses */
extern void stream__free (Stream self);
extern void stream__mark (Stream self);

#endif /*_H_Stream_*/
