/* String.h - Copyright (c) Marc Krisnanto */

#ifndef _H_String_
#define _H_String_ 1


/** String
	This is the final String class.

	String objects are immutable byte based strings.
	The string may contain nils and is nil terminated just like C strings.

	Thus if you pass a string to a C function and the string happens to contain
	a nil somewhere in the middle then C will interpret it as the end of the
	string.
*/
extern Class StringClass;


/** StringBYTE
	StringBYTELIST is the list of system predefined single byte String objects.
	You access a particular object with the StringBYTE macro.
*/
extern String StringBYTELIST [256];
#define StringBYTE(c) StringBYTELIST [(c) & 255]


/** StringPredefined
	Predefined strings.
*/
extern String StringEMPTY;  /* empty string */


/** TString
	This represents an instance of the String class.
*/
#define STRING_HEAD \
	OBJ_HEAD; \
	long   len;   /* content length */ \
	long   hash;  /* cached hash value, -1 = not cached */ \
	char  *str    /* content is always nil terminated */ \

struct TString
{
	STRING_HEAD;
};


/** StringIs
	Whether or not the object is a String.
**/
#define StringIs(o) (ClassOf(o) == StringClass)


/** StringFlag
	Flags used by String objects.

	Never access the flags directly.
	Use the provided macros instead.

	If there are no accessor macros, it is because you are not suppose to
	touch the flag.
*/
#define StringFlagPROTECTED Flag0  /* \__ match Buffer/Container */
#define StringFlagFROZEN    Flag1  /* /                          */
#define StringFlagSTATIC    Flag7  /* static memory? */
#define StringFlagUTF8      Flag3  /* UTF-8? (currently not used) */

/*
	10..end are reserved for Qu modules
	These may only be touched by the corresonding modules.
*/
#define StringFlagNETADDR   Flag11  /* reserved for the Net module */
#define StringFlagSOCKADDR  Flag12  /* -ditto- */
#define StringFlagREGEX     Flag13  /* reserved for the Regex module */


/** StringIsStatic
	Whether or not the String is a static String.

	A static String is not GC-ed and it's v{str} field points to static memory
	or permanent GcMemMalloc-ed memory.
*/
#define StringIsStatic(self)  FlagTest (self, StringFlagSTATIC)


/** StringHash
	Get the string hash value.

	@warn
	Prefer StringHASH over StringHash but notice the macro side effect.
*/
extern long StringHash   (String self);
#define StringHASH(self) (((self)->hash == -1) ? StringHash (self) : (self)->hash)


/** StringEQ
	Whether or not a String equals another object.

	This macro is provided for efficiency.
	v{self} must be a String object while v{x} may be any object.
*/
#define StringEQ(self, x) \
	((String) (self) == (String) (x) || StringEq (self, x))

extern int StringEq (String self, String x);


/** StringNew
	Various string constructors.

	StringNew takes a memory location and a length.

	StringNewz takes a C nil terminated string.

	StringNewl creates an uninitialized string with the v{hash} field set to -1.
	Use this function to set the content yourself.

	StringNewf creates a string using arguments ala C f{printf}.

	StringNewc creates a string by repreating a character.

	StringNewTrim does the same as StringNew except that the leading and trailing
	spaces are removed.

	StringSource creates string from source code.
	This function is used by the compiler. It translates escape sequences.

	StringDup creates a new string with the same content.
	The v{hash} field is set to -1 since you generally would call this
	function to modify the content. This is actually the same as calling
	StringNewl then copy the content.

	StringCat concatenate all strings.
	All arguments must be String objects.
	The last argument must be NULL to indicate termination.

	StringStatic creates a new string from a static C string.
	The string will not be included in the GC thus will never be freed.
	If one already exists in the static string table then no new string is created.

	StringStaticDup does the same but reallocate the C string.
	StringStaticDupz does the same and calculates the length.

	StringStaticz does the same as StringStatic and computes f{strlen} all by
	itself.

	StringGetStatic returns an existing string from the static string table
	if one exists, v{s} otherwise. This function is for efficiency to avoid
	creating duplicate strings, that is, if an existing String exist then
	forget v{s}.

	StringStaticFindz is a version of StringGetStatic.
	It returns NULL if no match is found.
	An existing String otherwise.

	@warn
	Make sure v{len} is within C{0..}={GC_MEM_MAXSIZE}.
	No checks are made.

	The argument v{z} to StringStatic and StringStaticz must be a nil terminated
	C static string e.g. C{StringStaticz ("foo bar")}.
*/
extern String StringNew         (const char *z, long len);
extern String StringNewz        (const char *z);
extern String StringNewl        (long len);
extern String StringNewf        (const char *format, ...);
extern String StringNewc        (char c, long n);
extern String StringNewTrim     (const char *s, long len);
extern String StringDup         (String s);
extern String StringSource      (const char *z, long len);
extern String StringCat         (String s, ...);
extern String StringStatic      (const char *z, long len);
extern String StringStaticDup   (const char *z, long len);
extern String StringStaticDupz  (const char *z);
extern String StringStaticz     (const char *z);
extern String StringStaticv     (const char *fmt, va_list va);
extern String StringStaticf     (const char *fmt, ...);
extern String StringGetStatic   (String s);
extern String StringStaticFindz (const char *z, long len);


/** StringPrint
	Print a String to v{stdout}.

	This function works with both mutable and immutable Strings.
	To print the String unquoted, set v{quot} to 0.
	Otherwise v{quot} must be one of C{'}, C{`}, or C{"}.

	Note that this might throw an exception because writing to v{stdout} may
	fail for various reasons.
*/
extern Null StringPrint (String self, char quot);


/** StringFormat
	Format this string.

	See ={Format} for details.
*/
extern String StringFormat (String self, format_s *f);


/** StringQuote
	Return a quoted string.
*/
#define StringQuote(self) StrzQuote ((self)->str, (self)->len, '\'')


/***/
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	C string
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

/** Strz
	Functions dealing with C strings.

	StrzHash computes the string hash value.
	Note that there is a cut-off implemented for reasonable performance.

	StrzEq return 1 if the strings are equal, 0 otherwise.
	StrzEqnc does the same in case insensitive way.
	The functions are intentionally taking only one length argument to force you to check for
	the same length before calling the functions.

	StrzChr returns the position of a character in the string, NULL if not found.
	StrzChrnc does the same in case insensitive way.
	StrzInChar does the same and return 1 if found, 0 otherwise.

	StrzStr returns the position of a string in the string, NULL if not found.
	StrzStrnc does the same in case insensitive way.

	StrzCharTest return 1 if all characters are in the a specified 256 characters set,
	0 otherwise.

	StrzSetCset fills a 256 character buffer with characters from a string.
	v{cset} is not cleared and the values are set to 1 if the character exists,
	unchanged otherwise.
	v{cmin} and v{cmax} are set to the lowest and highest character value.

	StrzSetCsetnc does the same except that it also sets the lowercase and uppercase
	characters in the set for all alphabet characters.

	StrzSlice return a Str from a slice of the string.

	StrzIndex find the index of a string in a string.
	Return -1 if not found.

	StrzStartWith test whether the string starts with v{what} while
	StrzEndWith test whether the string ends with v{what}.
	They return the matched item or NULL if none found.
	v{what} may be a String or an iterable that yields String objects.
	In case of a Dict then the keys are used and the corresponding value is stored
	in v{value} which may be anything.
	Non String keys are ignored.

	StrzTranslate translates a character.
	v{what} may be a String or an iterable that yields String objects.
	In case of a Dict then the keys are used and the corresponding value is stored
	in v{value} which may be anything.
	Non String keys are ignored.
	Return NULL if no translation occured.
*/
extern long    StrzHash      (const char *z, long l);
extern int     StrzEq        (const char *z, const char *x, long l);
extern int     StrzEqnc      (const char *z, const char *x, long l);
extern char   *StrzChr       (const char *z, long lz, char c);
extern char   *StrzChrnc     (const char *z, long lz, char c);
extern int     StrzCmp       (const char *z, long lz, const char *x, long lx);
extern char   *StrzStr       (const char *z, long lz, const char *x, long lx);
extern char   *StrzStrnc     (const char *z, long lz, const char *x, long lx);
extern long    StrzCopy      (const char *z, long lz, char *x, long len);
extern String  StrzSlice     (const char *z, long lz, long i, long n);
extern long    StrzIndex     (const char *z, long lz, const char *x, long lx, long ofs);
extern long    StrzIndexnc   (const char *z, long lz, const char *x, long lx, long ofs);
extern long    StrzRindex    (const char *z, long lz, const char *x, long lx, long ofs);
extern long    StrzRindexnc  (const char *z, long lz, const char *x, long lx, long ofs);
extern String  StrzStartWith (const char *z, long lz, int nocase, Any what, Any *value);
extern String  StrzEndWith   (const char *z, long lz, int nocase, Any what, Any *value);
extern void    StrzReverse   (char *z, long lz);
extern String  StrzTranslate (unsigned char c, Any x);
extern String  StrzQuote     (const char *z, long lz, char quot);

extern void    StrzSetCset   (const unsigned char *z, long lz, unsigned char *cset256,
                              unsigned char *cmin, unsigned char *cmax);

extern void    StrzSetCsetnc (const unsigned char *z, long lz, unsigned char *cset256,
                              unsigned char *cmin, unsigned char *cmax);

#define StrzEQ(za, la, zb, lb)   (((la) == (lb)) && StrzEq   (za, zb, la))
#define StrzEQNC(za, la, zb, lb) (((la) == (lb)) && StrzEqnc (za, zb, la))

#define StringIndex(sa, sb, ofs)   StrzIndex  ((sa)->str, (sa)->len, (sb)->str, (sb)->len, ofs)
#define StringRindex(sa, sb, ofs)  StrzRindex ((sa)->str, (sa)->len, (sb)->str, (sb)->len, ofs)
#define StringGetSlice(sa, ofs, n) StrzSlice  ((sa)->str, (sa)->len, ofs, n)


/***/
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	buffer
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

/** StringBuffer
	String buffer used by SysPrint etc. See C{Sys.c} for examples.

	@warn
	StringBuffer uses the v{hash} field to store the current allocated capacity.
	You must never touch this field.

	Make sure to call StringBufferReturn when the String is ready to be returned
	to program level.

	@block
	StringBufferNew creates a new string buffer.

	StringBufferPut append a string to the buffer.

	StringBufferClear clears the buffer.

	StringBufferReverse reverses the content.

	StringBufferSetlen sets the content length. Use this only if you are
	injecting the content directly. Be careful!

	StringBufferMakeSpace makes sure there is enough room to put more bytes,
	growing the buffer if necessary.

	StringBufferReturn must be called prior to returning the string to program
	level. This puts the nil terminator and sets the v{hash} field to -1.
*/
extern String StringBufferNew       (long cap);
extern int    StringBufferPuts      (String self, String s);
extern int    StringBufferPut       (String self, const char *z, long len);
extern int    StringBufferPutz      (String self, const char *z);
extern int    StringBufferPutc      (String self, char c);
extern void   StringBufferClear     (String self);
extern void   StringBufferReverse   (String self);
extern String StringBufferReturn    (String self);
extern void   StringBufferSetlen    (String self, long len);
extern int    StringBufferMakeSpace (String self, long more);
extern long   StringBufferAvail     (String self);
extern String StringBufferReturn    (String self);


/***/
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	Methods
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

/** StringCmpInt
	Compare two Strings.
*/
extern Int StringCmpInt (String self, Any x);


/** StringDelete
	Return a new String by replace occurences of v{what}.
*/
extern String StringDelete (String self, String what);


/** StringPad
	Return a new String by padding v{fill} to the left and right.
*/
extern String StringPad (String self, long left, long right, char fill);


/** StringTrim
	Return a new String by trimming spaces.
*/
extern String StringLtrim (String self);
extern String StringRtrim (String self);
extern String StringTrim  (String self);


/** StringInterval
	Return a new String by inserting v{delim} at v{intvl} intervals.
*/
extern String StringInterval (String self, String delim, long intvl);


/** StringCFuncName
	Convert C function name to something that makes sense.
*/
void StringCFuncName (const char *func, char *buf, size_t size);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the String class */
extern void StringSetup ();
extern void StringInit  ();

/* To avoid duplicate codes in Buffer */
extern Boolean string_isalnum  (String self, int argc, Any *argv);
extern Boolean string_isalpha  (String self, int argc, Any *argv);
extern Boolean string_isbdigit (String self, int argc, Any *argv);
extern Boolean string_isblank  (String self, int argc, Any *argv);
extern Boolean string_iscntrl  (String self, int argc, Any *argv);
extern Boolean string_isdigit  (String self, int argc, Any *argv);
extern Boolean string_isgraph  (String self, int argc, Any *argv);
extern Boolean string_ishi     (String self, int argc, Any *argv);
extern Boolean string_isident  (String self, int argc, Any *argv);
extern Boolean string_isidnum  (String self, int argc, Any *argv);
extern Boolean string_islo     (String self, int argc, Any *argv);
extern Boolean string_islower  (String self, int argc, Any *argv);
extern Boolean string_isname   (String self, int argc, Any *argv);
extern Boolean string_isodigit (String self, int argc, Any *argv);
extern Boolean string_ispath   (String self, int argc, Any *argv);
extern Boolean string_isprint  (String self, int argc, Any *argv);
extern Boolean string_ispwd    (String self, int argc, Any *argv);
extern Boolean string_ispunct  (String self, int argc, Any *argv);
extern Boolean string_isspace  (String self, int argc, Any *argv);
extern Boolean string_istext   (String self, int argc, Any *argv);
extern Boolean string_isupper  (String self, int argc, Any *argv);
extern Boolean string_isxdigit (String self, int argc, Any *argv);
extern Boolean string_iswhite  (String self, int argc, Any *argv);
extern Boolean string_isword   (String self, int argc, Any *argv);

extern Int string_asc (String self);
extern Int string_ord (String self);

#endif /*_H_String_*/
