/* Struct.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Struct_
#define _H_Struct_ 1


/** Struct
	This is the final Struct class.
*/
extern Class StructClass;


/** TStruct
	This represents an instance of the Struct class.
*/
typedef struct struct_entry_s struct_entry_s;
struct struct_entry_s
{
	String name; /* field name */
	Any    type; /* field type Class, Var, Sub, or Struct */
};

struct TStruct
{
	OBJ_HEAD; /* attrs used for specialized methods, 0 if none */

	/* never access these fields
		they are implementation specific */

	attr_s         *indexes;  /* list of indexes (key=name, value=index) */
	struct_entry_s *items;    /* items are allocated */
};


/** StructFlag
	Flags used by Struct objects.
*/
/* none but reserve all */


/** StructIs
	Whether or not an object is a Struct.
*/
#define StructIs(o) (ClassOf(o) == StructClass)


/** StructLEN
	Get the number of items in the Struct.
*/
#define StructLEN(self) AttrLen ((self)->indexes)


/** StructNew StructStatic
	Create a new Struct object.

	StructStatic creates a Struct which is never garbage collected.

	@xmp
	Struct mystruct = StructNew (3, IntClass, "foo", FloatClass, "bar", Nil, "baz");
*/
extern Struct StructNew    (int nelt, ...);
extern Struct StructStatic (int nelt, ...);


/** StructSetMethodz
	A wrapper arount f{struct_setmethod}.

	@warn
	v{namez} must be a static C string, e.g. C{"foo"}.
	It must not be a dynamic pointer.
*/
#define StructSetMethodz(self, namez, f) struct_setmethod (self, StringStatic (namez), f)


/** StructIndex StructIndexEx
	StructIndex finds the index of a field by name.

	StructIndexEx takes any object. If it is a String then StructIndex is called.
	If it is an Integer then validates the index. Otherwise returns -1.

	If not found, -1 is returned.
*/
extern long StructIndex   (Struct self, String x);
extern long StructIndexEx (Struct self, Any x);


/** StrucTypeList
	Lists the Qu "type", C type size and C type alignment for each type characters.

	See ={StructTypeToChar} for a list of type characters.
*/
typedef struct structtype_s structtype_s;
struct structtype_s
{
	Class       klass;  /* class */
	Sub         type;   /* validator */
	int         align;  /* alignment */
	int         size;   /* C size */
	const char *name;   /* C name */
};
extern structtype_s StructTypeList [256];
#define StructTypeInfo(c) (&StructTypeList[(unsigned char) (c)])


/** STRUCT_VALID_SPEC
	Check for a valid specification character.
*/
#define STRUCT_VALID_SPEC(c) (StructTypeList[(unsigned char) (c)].type != NULL)


/** StructValidSpec
	Check for a valid String specification.
*/
extern int StructValidSpec (char *s, long len);


/** StrucTypeToChar
	StructTypeToChar translates a type object into the corresponding character
	specification.

	The following table shows the list of character types and the corresponding
	C type, Qu "type", and Qu object.

	@table
	.| a | char              | Sys_char               | String
	.| y | byte              | Sys_byte               | Int
	.| Y | unsigned byte     | Sys_ubyte              | Int
	.| h | short             | Sys_short              | Int
	.| H | unsigned short    | Sys_ushort             | Int
	.| i | int               | Sys_int                | Int or Long
	.| I | unsigned int      | Sys_uint               | Int or Long
	.| j | int16             | Sys_int16              | Int
	.| J | unsigned int16    | Sys_uint16             | Int
	.| k | int32             | Sys_int32              | Int or Long
	.| K | unsigned int32    | Sys_uint32             | Int or Long
	.| l | long              | Sys_long               | Int or Long
	.| L | unsigned long     | Sys_ulong              | Int or Long
	.| o | long long         | Sys_longlong           | Int or Long
	.| O | unsigned long long| Sys_ulonglong          | Int or Long
	.| g | float             | Sys_float              | Float
	.| f | double            | Sys_double or Float    | Float
	.| F | long double       | Sys_longdouble         | Float or Big
	.| z | ssize_t           | Sys_ssize_t            | Int or Long
	.| Z | size_t            | Sys_size_t             | Int or Long
	.| e | off_t             | Sys_off_t              | Int or Long
	.| n | NULL              | Sys_Null or nil        | Pointer or Null
	.| b | int               | Sys_Boolean or Boolean | Boolean
	.| q | long              | Sys_Int or Int         | Int
	.| p | void *            | Sys_Pointer or Pointer | Pointer
	.| s | char *            | Sys_String or String   | String
	.| S | String            | Sys_String or String   | String
	.| A | Array             | Array                  | Array
	.| D | Dict              | Dict                   | Dict
	.| E | Set               | Set                    | Set
	.| R | Range             | Range                  | Range
	.| C | Class             | Class                  | Class
	.| Q | Queue             | Queue                  | Queue
	.| T | Stream            | Stream                 | Stream
	.| P | Proc              | Proc                   | Proc
	.| x | Any               | Any                    | any Qu object as is
*/
extern char StructTypeToChar (Any type);

#define STRUCT_ID_char       'a'
#define STRUCT_ID_byte       'y'
#define STRUCT_ID_ubyte      'Y'
#define STRUCT_ID_short      'h'
#define STRUCT_ID_ushort     'H'
#define STRUCT_ID_int        'i'
#define STRUCT_ID_uint       'I'
#define STRUCT_ID_int16      'j'
#define STRUCT_ID_uint16     'J'
#define STRUCT_ID_int32      'k'
#define STRUCT_ID_uint32     'K'
#define STRUCT_ID_long       'l'
#define STRUCT_ID_ulong      'L'
#define STRUCT_ID_longlong   'o'
#define STRUCT_ID_ulonglong  'O'
#define STRUCT_ID_float      'g'
#define STRUCT_ID_double     'f'
#define STRUCT_ID_longdouble 'F'
#define STRUCT_ID_ssize_t    'z'
#define STRUCT_ID_size_t     'Z'
#define STRUCT_ID_off_t      'e'
#define STRUCT_ID_Null       'n'
#define STRUCT_ID_Boolean    'b'
#define STRUCT_ID_Int        'q'
#define STRUCT_ID_Pointer    'p'
#define STRUCT_ID_String     's'
#define STRUCT_ID_Array      'A'
#define STRUCT_ID_Dict       'D'
#define STRUCT_ID_Set        'E'
#define STRUCT_ID_Range      'R'
#define STRUCT_ID_Class      'C'
#define STRUCT_ID_Function   'U'
#define STRUCT_ID_Sub        'v'
#define STRUCT_ID_Method     'w'
#define STRUCT_ID_Queue      'Q'
#define STRUCT_ID_Stream     'T'
#define STRUCT_ID_Proc       'P'
#define STRUCT_ID_Any        'x'

#define STRUCT_IDS_char       "a"
#define STRUCT_IDS_byte       "y"
#define STRUCT_IDS_ubyte      "Y"
#define STRUCT_IDS_short      "h"
#define STRUCT_IDS_ushort     "H"
#define STRUCT_IDS_int        "i"
#define STRUCT_IDS_uint       "I"
#define STRUCT_IDS_int16      "j"
#define STRUCT_IDS_uint16     "J"
#define STRUCT_IDS_int32      "k"
#define STRUCT_IDS_uint32     "K"
#define STRUCT_IDS_long       "l"
#define STRUCT_IDS_ulong      "L"
#define STRUCT_IDS_longlong   "o"
#define STRUCT_IDS_ulonglong  "O"
#define STRUCT_IDS_float      "g"
#define STRUCT_IDS_double     "f"
#define STRUCT_IDS_longdouble "F"
#define STRUCT_IDS_ssize_t    "z"
#define STRUCT_IDS_size_t     "Z"
#define STRUCT_IDS_off_t      "e"
#define STRUCT_IDS_Null       "n"
#define STRUCT_IDS_Boolean    "b"
#define STRUCT_IDS_Int        "q"
#define STRUCT_IDS_Pointer    "p"
#define STRUCT_IDS_String     "s"
#define STRUCT_IDS_Array      "A"
#define STRUCT_IDS_Dict       "D"
#define STRUCT_IDS_Set        "E"
#define STRUCT_IDS_Range      "R"
#define STRUCT_IDS_Class      "C"
#define STRUCT_IDS_Function   "U"
#define STRUCT_IDS_Sub        "v"
#define STRUCT_IDS_Method     "w"
#define STRUCT_IDS_Queue      "Q"
#define STRUCT_IDS_Stream     "T"
#define STRUCT_IDS_Proc       "P"
#define STRUCT_IDS_Any        "x"


/** StructOfs
	Find the type and offset of a C value if the Struct were a C struct.

	StructOfsEx does the same but v{x} may be a Struct or a String specification,
	e.g. "ssfi".
*/
extern long StructOfs   (Struct self, long i, char *type);
extern long StructOfsEx (Any x, long i, char *type);


/** StructSize
	Return the size of the Struct if it were a C struct.

	StructSizeEx does the same but v{x} may be a Struct or a String specifiction,
	e.g. "ssfi".
*/
extern long StructSize   (Struct self);
extern long StructSizeEx (Any x);


/** StructGetItem
	Get an item from the Struct.
*/
extern Any StructGetItem (Struct self, Any x);


/** StructLike
	Whether two Structs are alike.
*/
#define StructLike(self, x) (struct__like (self, x) == True)



/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Struct class */
extern void StructSetup ();
extern void StructInit  ();

/* This one is called by SysInit */
extern void StructSysInit ();

/* for interp.c and Code.c */
extern Sub StructMethodNew;

/* for Reduce.c */
extern Struct StructCodeNew (Ast n);

/* published for efficiency */
extern Boolean struct__like (Struct self, Struct x);


#endif /*_H_Struct_*/
