/* Sub.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Sub_
#define _H_Sub_ 1


/** Sub
	This is the final Sub class.
*/
extern Class SubClass;


/** SUB_ALLOW_ASSIGN
	Whether or not calling Sub can be done via the assignment operator.

	This must always be defined.
	The macro only serves as a placeholder so tracking is easier.
*/
#define SUB_ALLOW_ASSIGN 1


/** TSub
	This represents an instance of the Sub class.

	Methods and functions are all Sub objects.

	We have three types:
	@args
	.external  C level function/method, SubFlagEXTERN set
	.regular   script level function/method, SubFlagEXTERN not set
	.closure   script level closures, SubFlagCLOSURE set, SubFlagEXTERN not set

	@warn
	Some fields are not allocated depending on the type.

	@table
	.| SubFlagEXTERN              | SUB_HEAD only
	.| SubFlagEXTERN  SubFlagJIT  | SUB_HEAD + SUB_SCRIPT_FIELDS
	.| SubFlagCLOSURE             | all
	.| others                     | SUB_HEAD + SUB_SCRIPT_FIELDS
*/
#define SUB_HEAD \
	OBJ_HEAD; \
	String     name;    /* Name, must be the first field */ \
	Symbol     parent;  /* Block where declared (Module, Class, or Sub) */ \
	Any        value;   /* Code object or C function address or "once" value or JIT code */ \
	int        argc;    /* Minimum number of arguments, -1 means any */ \
	Any        type;    /* Return type, maybe 0 */ \
	Any       *ltype;   /* List of lexical types, 0 or set by Sub_reindex */ \
	Dict       tag;     /* The tag entries, maybe 0 */ \
	subcall_s *call     /* How to call the Sub --- set by Sub.c based on flags */

#define SUB_SCRIPT_FIELDS \
	Sub        root;    /* Root sub, self for root */ \
	int        narg;    /* Actual number of arguments */ \
	int        nlocal;  /* Number of lexicals */ \
	int        first    /* Index of first lexical */ \

#define SUB_CLOSURE_FIELDS \
	Array      clex;   /* Closure lexicals */ \
	Sub        corg    /* The Sub object used to create the closure */


typedef Any (*subcalln_f) (Sub self, Any theself, int argc, Any *argv);
typedef Any (*subcall4_f) (Sub self, Any theself, Any arg1, Any arg2, Any arg3, Any arg4);
typedef Any (*subcall3_f) (Sub self, Any theself, Any arg1, Any arg2, Any arg3);
typedef Any (*subcall2_f) (Sub self, Any theself, Any arg1, Any arg2);
typedef Any (*subcall1_f) (Sub self, Any theself, Any arg);
typedef Any (*subcall0_f) (Sub self, Any theself);

typedef struct subcall_s subcall_s;
struct subcall_s
{
	subcalln_f fn;
	subcall4_f f4;
	subcall3_f f3;
	subcall2_f f2;
	subcall1_f f1;
	subcall0_f f0;
};

struct TSub
{
	SUB_HEAD;

	/* allocated for script subs only */
	SUB_SCRIPT_FIELDS;

	/* allocated for closures only */
	SUB_CLOSURE_FIELDS;
};


/** SubIs
	Whether or not the object is a Sub.
*/
#define SubIs(o) (ClassOf(o) == SubClass)


/** SubFlag
	Flags use by Sub objects.

	Never access these flags directly.
	Use the accessor macros provided instead.
*/
#define SubFlagEXTERN     SymbolFlagEXTERN   /* C or script? */
#define SubFlagPRIVATE    SymbolFlagPRIVATE  /* private? */
#define SubFlagLOCAL      SymbolFlagLOCAL    /* local? */
#define SubFlagSTATIC     SymbolFlagSTATIC   /* self may be anything? */
#define SubFlagCLASS      SymbolFlagCLASS    /* class method? */
#define SubFlagINST       SymbolFlagINST     /* instance method? */
#define SubFlagFINAL      SymbolFlagFINAL    /* cant override? */
#define SubFlagABSTRACT   SymbolFlagABSTRACT /* must override? */
#define SubFlagVARARG     Flag10             /* last arg is vararg? e.g. foo (a, *b) */
#define SubFlagARGMIN     Flag11             /* has default arguments? (argc is minimum) */
#define SubFlagARGCHECK   Flag12             /* must check argument type? */
#define SubFlagCFUNC      Flag13             /* C function? (don't pass self) */
#define SubFlagCLOSURE    Flag14             /* is closure? */
#define SubFlagGENERATOR  Flag15             /* is generator? */
#define SubFlagUNSAFE     Flag16             /* __unsafe? */
#define SubFlagONCE       Flag17             /* __once? */
#define SubFlagACCUM      Flag18             /* __accumulator? */
#define SubFlagENSURE     Flag19             /* property validator? (ensure) */
#define SubFlagRETURNS    Flag20             /* issues return? */
#define SubFlagJIT        Flag21             /* jit-ed? */
#define SubFlagINTERFACE  Flag22             /* __interface? */
#define SubFlagARGS       Flag23             /* __args? */


/** SubIsExtern
	Whether this is an external function (C or whatever) or program level Sub.

	Note that JIT might turn a program level Sub to "external".
*/
#define SubIsExtern(self) FlagTest (self, SubFlagEXTERN)


/** SubIsClass
	Whether this is a class method.

	@xmp
		class Foo
			sub class bar  <-
*/
#define SubIsClass(self) FlagTest (self, SubFlagCLASS)


/** SubIsInst
	Whether this is an instance method.

	@xmp
		class Foo
			sub bar  <-
*/
#define SubIsInst(self) FlagTest (self, SubFlagINST)


/** SubIsStatic
	Whether self in this Sub may have any value (unbound).

	This has nothing to do with the word "static" as known in other language such
	as C, Java, etc.

	All module level functions and child Sub (Sub within Sub) are "static".

	@xmp
		sub foo             <-
			sub bar         <-

		class Foo
			sub static foo  <-
			sub bar
				sub baz     <-
*/
#define SubIsStatic(self) FlagTest (self, SubFlagSTATIC)


/** SubIsPrivate
	Whether this is a private function known to block/childs, or method known
	to class/subclasses.

	@xmp
		sub foo
		sub _bar            <-
		sub baz __private   <-
*/
#define SubIsPrivate(self) FlagTest (self, SubFlagPRIVATE)


/** SubIsLocal
	Whether it is local to the class (unknown to subclasses).

	@xmp
		class Foo
			sub local bar   <-
*/
#define SubIsLocal(self) FlagTest (self, SubFlagLOCAL)


/** SubIsFinal
	Whether this is a final method (can not override).

	@xmp
		class Foo
			sub bar __final  <-
*/
#define SubIsFinal(self) FlagTest (self, SubFlagFINAL)


/** SubIsAbstract
	Whether this is an abstract method (must be defined by real subclasses).

	@xmp
		class Foo __abstract
			sub bar __abstract  <-
*/
#define SubIsAbstract(self) FlagTest (self, SubFlagABSTRACT)


/** SubIsVararg
	Whether the last argument of the Sub is a variable argument.

	@xmp
	sub foo (a, b, *c)
	               |
	               +--- makes foo SubIsVararg
*/
#define SubIsVararg(self) FlagTest (self, SubFlagVARARG)


/** SubIsArgmin
	Whether the Sub has default arguments, thus the field 'argc' is the minimum
	number of arguments that must be passed to the Sub.

	@xmp
	sub foo (a, b = 5, c = true)
	              |
	              +--- makes foo SubIsArgmin (argc = 1)
*/
#define SubIsArgmin(self) FlagTest (self, SubFlagARGMIN)


/** SubNeedArgCheck
	Whether the one or more arguments are typed, i.e. the Sub needs to validate
	arguments passed to it.

	@xmp
	sub foo (a: Boolean, b, c)
	          |
	          +--- makes foo SubNeedArgCheck
*/
#define SubNeedArgCheck(self) FlagTest (self, SubFlagARGCHECK)


/** SubIsCfunc
	Whether or not calling the C function Sub must not pass self as the first
	argument.

	@xmp
		static Int ModuleFoo_bar (Any self, String a, Array b)  <- no
		static Int ModuleFoo_bar (String a, Array b)            <- yes
*/
#define SubIsCfunc(self) FlagTest (self, SubFlagCFUNC)


/** SubIsClosure
	Whether this is a run-time created closure.

	@xmp
		sub foo (x)
			return lambda (x) x * 2   <-
*/
#define SubIsClosure(self) FlagTest (self, SubFlagCLOSURE)


/** SubReturns
	Whether the Sub has a k{return} statement.

	This is only used by compile.c to track yield and return.
*/
#define SubReturns(self) FlagTest (self, SubFlagRETURNS)


/** SubIsGenerator
	Whether the Sub issues k{yield} or is tagged __generator.

	@xmp
		sub foo (x)
			yield x

		sub foo (x) __generator
*/
#define SubIsGenerator(self) FlagTest (self, SubFlagGENERATOR)


/** SubIsAccum
	Whether this is an accumulator.

	An accummulator Sub preserves the value of it's lexical over calls.

	@xmp
		sub make (x)
			return lambda (z) __accumulator x + z
		;;
*/
#define SubIsAccum(self) FlagTest (self, SubFlagACCUM)


/** SubIsEnsure
	Whether this is an k{ensure} method.

	@xmp
		var foo
			ensure
				...
			;;

	@warn
	The SubFlagENSURE flag must always be coupled with SubFlagPRIVATE.
*/
#define SubIsEnsure(self) FlagTest (self, SubFlagENSURE)


/** SubIsOnce
	Whether the Sub may only be called once.

	If it is then after the first call the field v{value} will have the result
	and subsequent calls will just return this result.

	@xmp
		sub foo (x) __once
			return x
		;;

	@block
	Note that C functions may also set this flag.
*/
#define SubIsOnce(self) FlagTest (self, SubFlagONCE)


/** SubIsUnsafe
	Whether this is an __unsafe Sub.

	If it is, the compiler generates some extra prolog code to make sure that
	the caller does not come from an untrusted Module.

	@xmp
		sub foo (x) __unsafe
*/
#define SubIsUnsafe(self) FlagTest (self, SubFlagUNSAFE)


/** SubIsJit
	Whether the Sub is JIT-ed, i.e. the code is compiled into native code
	thus the field v{value} is the native code and must be GcMemFree-ed when
	the Sub is destroyed.
*/
#define SubIsJit(self) FlagTest (self, SubFlagJIT)



/** SubNew
	Create a new Sub.

	This is called by the compiler, ={ClassMethod} etc.

	@warn
	Do not call directly unless you know exactly what you are doing.
*/
extern Sub SubNew (Symbol block, String name, Any value, int argc, long flags);


/** Sub_reindex SubEndDecl SubSetOnce
	For the compiler, to reindex locals and to finalize declaration.

	For Sub_reindex, the Sub must be the root Sub.

	SubEndDecl frees all local Var objects and only keep the types,
	if any, in the v{args} field.

	SubSetOnce sets the necessary __once properties.
	It is called by C{Interp.c}.
*/
extern void Sub_reindex (Sub self);
extern void SubEndDecl  (Sub self);
extern void SubSetOnce  (Sub self, Any result);


/** SubJit
	These are for the JIT engine.

	See ={CodeJit} for more info.
*/
extern void SubJitSet       (Sub self);
extern void SubJitSetExtern (Sub self);


/** SubModule
	Get the Module object where the Sub is declared.
*/
extern Module SubModule (Sub self);


/** SubClassOf
	Get the Class object where the Sub is declared.

	If the Sub is a function (not a method, i.e. not in a Class block)
	then 0 is returned, no exception thrown.
*/
extern Class SubClassOf (Sub self);


/** SubIsRoot SubIsParent
	Whether a Sub is a nested Sub or the parent of a Sub.
*/
extern int SubIsRoot   (Sub self);
extern int SubIsParent (Sub self, Sub p);


/** SubClosure
	For the interpreter, to create a new closure.
*/
extern Sub SubClosure (Sub u, Frame f);


/** SubGetAttr SubSetAttr
	SubGetAttr find a local symbol in the Sub.

	SubSetAttr sets a local symbol in the Sub where v{thing} must be a
	Sub or a ={Var} object.

	This is designed for the compiler.
	Do not call unless you know exactly what you are doing.
*/
extern Symbol SubGetAttr (Sub self, String name);
extern void   SubSetAttr (Sub self, String name, Symbol attr);


/** SubGetTag SubSetTag
	SubGetTag returns a value in the C{__tag} property.
	Returns 0 if none.

	SubSetTag set a value in the C{__tag} property.

	SubGetTagDict return all in a new Dict.
	Return 0 if none.
*/
extern Any  SubGetTag     (Sub self, String name);
extern void SubSetTag     (Sub self, String name, Any value);
extern Dict SubGetTagDict (Sub self);


/** SubArgCheck
	Checks for valid arguments.

	If v{runtime} is true then validators are actually called. Otherwise
	only a type match is performed (e.g. used by the compiler).
*/
extern Sub SubArgCheck (Sub self, int argc, Any *argv, int runtime);


/** SubCode
	Get the Sub's code or throw ENone if none.
*/
extern Code SubCode (Sub self);


/** SubString
	Get a String representation of the Sub.
*/
extern String SubString (Sub self);


/** RETURN
	Use this instead of C{return x} to override the function name in the exception
	message to the name of this function.

	As a rule of thumb, always use RETURN instead of k{return} unless you
	know exactly that the called function never fails.

	@xmp
		static Int _do_something ()
		{
			...
			THROW (EFoo);
				--> sets Thread->eblock to "_do_something"
			...
			return Int0;
		}

		static Int Foo_bar (String baz)
		{
			...
			RETURN (_do_something ());
				--> if _do_something throws the this will override
				    Thread->eblock to "Foo_bar" then return 0
		}

	@block
	If you are certain that C{_do_something} never fails then you can simply
	use k{return} instead of RETURN.
*/
#define RETURN(x) \
BEGIN \
	Any _rEtvAl_ = x; \
	ENSURE (_rEtvAl_); \
	return _rEtvAl_; \
END


/** SubCall
	Convenient macros to call a Sub (function or method).
*/
#define SubCalln(self, theself, argc, argv) \
	((Sub) self)->call->fn ((Sub) self, theself, argc, argv)

#define SubCall4(self, theself, a1, a2, a3, a4) \
	((Sub) self)->call->f4 ((Sub) self, theself, a1, a2, a3, a4)

#define SubCall3(self, theself, a1, a2, a3) \
	((Sub) self)->call->f3 ((Sub) self, theself, a1, a2, a3)

#define SubCall2(self, theself, a1, a2) \
	((Sub) self)->call->f2 ((Sub) self, theself, a1, a2)

#define SubCall1(self, theself, a1) \
	((Sub) self)->call->f1 ((Sub) self, theself, a1)

#define SubCall0(self, theself) \
	((Sub) self)->call->f0 ((Sub) self, theself)


/***/
/*----------------------------------------------------------------------------
	Dealing with arguments (for C functions/methods)
----------------------------------------------------------------------------*/

/** Arg
	Various macros to deal with function/method arguments.

	See a lot of examples in the source code.
*/
extern void ArgThrowBoolean (const char *cfunc);
extern void ArgThrowInteger (const char *cfunc);
extern void ArgThrowFloat   (const char *cfunc);
extern void ArgThrowEnum    (const char *cfunc, Var v);
extern void ArgThrowCount   (const char *cfunc, int argc, int min, int max);
extern void ArgThrowMin     (const char *cfunc, int argc, int min);
extern void ArgThrowMax     (const char *cfunc, int argc, int max);
extern void ArgThrowExact   (const char *cfunc, int argc, int n);

#define ArgTOBOOL(x, v)  BEGIN if (!BooleanIs (x)) { ArgThrowBoolean (__FUNCTION__); return 0; } v = ((Boolean) (x)) == True; END;
#define ArgTOB(x, v)     BEGIN v = IntegerTob   (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOUB(x, v)    BEGIN v = IntegerToub  (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOLI(x, v)    BEGIN v = IntegerToli  (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOULI(x, v)   BEGIN v = IntegerTouli (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOH(x, v)     BEGIN v = IntegerToh   (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOUH(x, v)    BEGIN v = IntegerTouh  (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOI(x, v)     BEGIN v = IntegerToi   (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOUI(x, v)    BEGIN v = IntegerToui  (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOL(x, v)     BEGIN v = IntegerTol   (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOUL(x, v)    BEGIN v = IntegerToul  (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOF(x, v)     BEGIN v = RealTof      (x); if (errno) { ArgThrowFloat   (__FUNCTION__); return 0; } END
#define ArgTOD(x, v)     BEGIN v = RealTod      (x); if (errno) { ArgThrowFloat   (__FUNCTION__); return 0; } END
#ifdef HAVE_LONG_LONG
#define ArgTOLL(x, v)    BEGIN v = IntegerToll  (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#define ArgTOULL(x, v)   BEGIN v = IntegerToull (x); if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } END
#endif
#ifdef HAVE_LONG_DOUBLE
#define ArgTOLD(x, v)    BEGIN v = RealTold     (x); if (errno) { ArgThrowFloat   (__FUNCTION__); return 0; } END
#endif

/* this only works with integer values */
#define ArgTOENUM(x, v, enu) \
BEGIN \
	assert (VarIs (enu) && VarIsEnum (enu)); \
	assert (DictIs ((enu)->value)); \
	v = IntegerToi (x); \
	if (errno) { ArgThrowInteger (__FUNCTION__); return 0; } \
	if (!DictGetItem ((Dict) (enu)->value, x)) { ArgThrowEnum (__FUNCTION__, enu); return 0; } \
END

#define ArgVCOUNT(min, max) BEGIN if ((argc) < (min) || (argc) > (max)) { ArgThrowCount (__FUNCTION__, argc, min, max); return 0; } END
#define ArgVEXACT(count)    BEGIN if ((argc) != (count)) { ArgThrowExact (__FUNCTION__, argc, count); return 0; } END
#define ArgVMIN(min)        BEGIN if ((argc) < (min)) { ArgThrowMin (__FUNCTION__, argc, min); return 0; } END
#define ArgVMAX(max)        BEGIN if ((argc) > (max)) { ArgThrowMax (__FUNCTION__, argc, max); return 0; } END

#define ArgVTOBOOL(i, v)      ArgTOBOOL (argv[i], v)
#define ArgVTOB(i, v)         ArgTOB    (argv[i], v)
#define ArgVTOUB(i, v)        ArgTOUB   (argv[i], v)
#define ArgVTOLI(i, v)        ArgTOLI   (argv[i], v)
#define ArgVTOULI(i, v)       ArgTOULI  (argv[i], v)
#define ArgVTOH(i, v)         ArgTOH    (argv[i], v)
#define ArgVTOUH(i, v)        ArgTOUH   (argv[i], v)
#define ArgVTOI(i, v)         ArgTOI    (argv[i], v)
#define ArgVTOUI(i, v)        ArgTOUI   (argv[i], v)
#define ArgVTOL(i, v)         ArgTOL    (argv[i], v)
#define ArgVTOUL(i, v)        ArgTOUL   (argv[i], v)
#define ArgVTOF(i, v)         ArgTOF    (argv[i], v)
#define ArgVTOD(i, v)         ArgTOD    (argv[i], v)
#ifdef HAVE_LONG_LONG
#define ArgVTOLL(i, v)        ArgTOLL   (argv[i], v)
#define ArgVTOULL(i, v)       ArgTOULL  (argv[i], v)
#endif
#ifdef HAVE_LONG_DOUBLE
#define ArgVTOLD(i, v)        ArgTOLD   (argv[i], v)
#endif
#define ArgVTOENUM(i, v, enu) ArgTOENUM (argv[i], v, enu)


/** SubArgValidate
	Use these macros to validate all arguments passed in v{argv}.
*/
extern int SubArgValidate (Any type, int argc, Any *argv);


/** ArgPARSE
	Parse arguments.

	Throw EArgValue if one or more fields do not match, EArgCount
	if the number of fields is less than required.

	@warn
	You must name the arguments v{argc}, v{argv} and v{argx}.
	Do not invent new names or the macros will not work.

	@xmp
		class Foo
			sub bar (how: Dict , nocase = false) -> Bool


		Boolean Foo_bar (Foo self, int argc, Any *argv)
		{
			static char *argx[] = {"D", "how", ":", "b", "nocase", NULL};

			Dict how;
			int  nocase = 0;

			ArgXPARSE (&how, &nocase);

			...
			...
		}

	@block
	Do not call the functions directly.
	Use the macro instead.

	Note that the list is similiar as ={StrucTypeToChar}.

	@table
	.| : | -        | start optional arguments
	.| a | String   | char
	.| y | Integer  | int (-127 .. 127)
	.| Y | Integer  | int (0 .. 256)
	.| b | Boolean  | int (0 or 1)
	.| h | Integer  | short
	.| H | Integer  | unsigned short
	.| q | Integer  | long (IntMIN .. IntMAX)
	.| i | Integer  | int
	.| I | Integer  | unsigned int
	.| j | Integer  | int16
	.| J | Integer  | unsigned int16
	.| k | Integer  | int32
	.| K | Integer  | unsigned int32
	.| m | Integer  | int64
	.| M | Integer  | unsigned int64
	.| l | Integer  | long
	.| L | Integer  | unsigned long
	.| o | Integer  | long long
	.| O | Integer  | unsigned long long
	.| g | Real     | float
	.| f | Real     | double
	.| F | Real     | long double
	.| z | Integer  | ssize_t
	.| Z | Integer  | size_t
	.| e | Integer  | off_t
	.| p | Pointer  | void *
	.| s | String   | String   (not converted)
	.| A | Array    | Array    (not converted)
	.| D | Dict     | Dict     (not converted)
	.| E | Set      | Set      (not converted)
	.| R | Range    | Range    (not converted)
	.| C | Class    | Class    (not converted)
	.| B | Buffer   | Buffer   (not converted)
	.| Q | Queue    | Queue    (not converted)
	.| T | Stream   | Stream   (not converted)
	.| P | Proc     | Proc     (not converted)
	.| U | Function | Function (not converted)
	.| v | Sub      | Sub      (not converted)
	.| w | Method   | Method   (not converted)
	.| x | Any      | Any      (not converted)
*/
extern int SubArgvParse (const char *funcname, const char *spec, int argc, Any *argv, ...);
#define ArgVPARSE(spec, ...) \
	ENSURE (SubArgvParse (__FUNCTION__, spec, argc, argv, __VA_ARGS__))

extern int SubArgxParse (const char *funcname, char **argx, int argc, Any *argv, ...);
#define ArgXPARSE(...) \
	ENSURE (SubArgxParse (__FUNCTION__, argx, argc, argv, __VA_ARGS__))

extern int SubArgdParse (const char *funcname, char **argx, Dict argd, ...);
#define ArgDPARSE(...) \
BEGIN \
	REQUIRE (EArgValue, DictIs (argd)); \
	ENSURE (SubArgdParse (__FUNCTION__, argx, argd, __VA_ARGS__)); \
END


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Sub class */
extern void SubSetup (void);
extern void SubInit  (void);

/* Special Sub to simulate yield from C code. See Code.c */
extern Sub SubYield;

/* for Method.c */
extern Int sub_argmatch  (Sub self, int argc, Any *argv);
extern Any sub_argnames  (Sub self);
extern Any sub_argtypes  (Sub self);
extern Any sub__getitem  (Sub self, String name);
extern Any sub__getslice (Sub self, String name, Any defval);
extern Any sub__setitem  (Sub self, String name, Any val);

/* for Reduce.c/Emit.c */
extern int SubAccessible (Sub self, Symbol from);

#ifdef DEBUG_SEGV
extern void SubTrace     (Sub u);
extern void SubTraceShow ();
#endif

#endif /*_H_Sub_*/
