/* Sys.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Sys_
#define _H_Sys_ 1


/** SysModule
	This is the builtin Sys module.

	It contains all the builtin functions, operators and printing functions.
*/
extern Module SysModule;


/** Sys__debug__
	Sys__debug__ is the global __debug__ variable.
*/
extern Var Sys__debug__;


/** SysWarning
	Functions and macros to help keep warnings and error messages consistent
	across files.

	Warnings go to v{stderr} but only if ={APP_NO_WARNING} is not set.
*/
#define SysWARNING(...) \
	BEGIN if (!APP_NO_WARNING) SysWarning (__VA_ARGS__); END

#define SysWARNCLASS(self, msgz) \
	BEGIN if (!APP_NO_WARNING) SysWarnClass (self, msgz); END

#define SysWARNRETVAL(self, namez, expectedz) \
	BEGIN if (!APP_NO_WARNING) SysWarnRetval (self, namez, expectedz); END

#define SysEIMPLEMENT(self, namez) \
	BEGIN SysEImplement (self, namez, __FUNCTION__); RETHROW; END

extern void SysWarning    (const char *fmt, ...);
extern void SysWarnClass  (Any self, const char *msg);
extern void SysWarnRetval (Any self, const char *name, const char *expected);
extern void SysEImplement (Any self, const char *name, const char *func);


/** Sys__arg
	This is the global v{__arg__} variable. It is set by C{qu.c} from the
	command line.
*/
extern Var Sys__arg;


/** SysTrace Sys__tracefilter__ Sys__tracehow__
	SysTrace determines whether or not exceptions should be displayed.

	Sys__tracefilter__ holds the list of exception classes that should or should
	not be displayed when trace is on. The value is a ={Set}.

	If Sys__tracehow__ is False, then display those exceptions.
*/
extern int volatile SysTrace;
extern Var Sys__tracefilter__;
extern Var Sys__tracehow__;


/** SysPrint SysPrintz SysPrintf SysSprintf SysPrintBuf SysPrintHtml
	These are the function to call from the __print methods.

	SysPrintz is provided for efficiency. It is generally preferable over
	SysPrint.

	SysPrintHtml does SysPrint but HTML escape the characters.

	SysPrintf and SysSprintf use our formatting standard, not of C.

	@warn
	An object must not call SysPrint to print itself otherwise we fall in
	a deadly infinite loop.

	SysPrintf and SysSprintf can only take SysPRINTFMAXARG arguments max.

	SysPrintb prints with a buffer thus improve performance.
	See Array.c for an example.

	@warn
	SysPrintb buffer size must be SYS_PRINTB_BUFSIZ
*/
#define SysPRINTFMAXARG 256
extern Null   SysPrint     (Any x);
extern Null   SysPrintHtml (Any x);
extern Null   SysPrintz    (const char *s, long len);
extern Null   SysPrintf    (String format, int argc, ...);
extern String SysSprintf   (String format, int argc, ...);


#define SYS_PRINTB_BUFSIZ 1024
typedef struct sysprintb_s sysprintb_s;
struct sysprintb_s
{
	char  buf [SYS_PRINTB_BUFSIZ];
	char *ptr;
};
extern int Sys_Printb (sysprintb_s *b, const char *z, long n);

#define SysPrintbInit(b)   BEGIN b.ptr = b.buf; END
#define SysPrintb(b, z, n) Sys_Printb (&b, z, n)
#define SysPrintbs(b, s)   Sys_Printb (&b, ((String) s)->str, ((String) s)->len)
#define SysPrintbFinal(b)  ((b.ptr > b.buf) ? (SysPrintz (b.buf, b.ptr - b.buf) != 0) : 1)


/** Sys_binop
	Generic binary operators.
*/
extern Any     Sys_binop    (Any self, Any x, int op);
extern Boolean Sys_are      (Any self, Any x);
extern Any     Sys_ddiv     (Any self, Any x);
extern Any     Sys_dminus   (Any self, Any x);
extern Any     Sys_dmul     (Any self, Any x);
extern Any     Sys_dplus    (Any self, Any x);
extern Any     Sys_dpow     (Any self, Any x);
extern Any     Sys_drem     (Any self, Any x);
extern Int     Sys_abscmp   (Any self, Any x);
extern Boolean Sys_abseq    (Any self, Any x);
extern Boolean Sys_absne    (Any self, Any x);
extern Boolean Sys_abslt    (Any self, Any x);
extern Boolean Sys_absle    (Any self, Any x);
extern Boolean Sys_absgt    (Any self, Any x);
extern Boolean Sys_absge    (Any self, Any x);
extern Any     Sys_absdiv   (Any self, Any x);
extern Any     Sys_absidiv  (Any self, Any x);
extern Any     Sys_absminus (Any self, Any x);
extern Any     Sys_absmul   (Any self, Any x);
extern Any     Sys_absplus  (Any self, Any x);
extern Any     Sys_abspow   (Any self, Any x);
extern Any     Sys_absrem   (Any self, Any x);
extern Method  Sys_curry    (Any self, Any x);


/** Sys_operators
	These are the builtin operators.

	Refer the I{operator} section of the manual for details.
*/
extern Any     Sys_as       (Any self, Any x);
extern Any     Sys_and      (Any self, Any x);
extern Boolean Sys_band     (Any self, Any x);
extern Boolean Sys_bor      (Any self, Any x);
extern Any     Sys_clear    (Any self);
extern Int     Sys_cmp      (Any self, Any x);
extern String  Sys_concat   (Any self, Any x);
extern Any     Sys_copy     (Any self);
extern Any     Sys_decr     (Any self);
extern Int     Sys_del      (Any self, Any x);
extern Any     Sys_div      (Any self, Any x);
extern Boolean Sys_eq       (Any self, Any x);
extern Any     Sys_each     (Any self);
extern Any     Sys_format   (Any self, int argc, Any *argv);
extern String  Sys__format  (Any self, Any x);
extern Boolean Sys_ge       (Any self, Any x);
extern Any     Sys_getitem  (Any self, Any x);
extern Any     Sys_getslice (Any self, Any i, Any len);
extern Boolean Sys_gt       (Any self, Any x);
extern Any     Sys_iand     (Any self, Any x);
extern Any     Sys_idiv     (Any self, Any x);
extern Boolean Sys_in       (Any self, Any x);
extern Any     Sys_incr     (Any self);
extern Any     Sys_inot     (Any self);
extern Any     Sys_ior      (Any self, Any x);
extern Boolean Sys_is       (Any self, Any x);
extern Any     Sys_ixor     (Any self, Any x);
extern Boolean Sys_le       (Any self, Any x);
extern Int     Sys_len      (Any self);
extern Boolean Sys_like     (Any self, Any x);
extern Any     Sys_lshift   (Any self, Any x);
extern Boolean Sys_lt       (Any self, Any x);
extern Any     Sys_minus    (Any self, Any x);
extern Any     Sys_mul      (Any self, Any x);
extern Boolean Sys_ne       (Any self, Any x);
extern Any     Sys_neg      (Any self);
extern Boolean Sys_not      (Any self);
extern Any     Sys_of       (Any self, Any x);
extern Any     Sys_or       (Any self, Any x);
extern Any     Sys_plus     (Any self, Any x);
extern Any     Sys_pop      (Any self);
extern Any     Sys_pos      (Any self);
extern Any     Sys_pow      (Any self, Any x);
extern Any     Sys_push     (Any self, Any x);
extern Any     Sys_range    (Any self, Any x);
extern Any     Sys_ratio    (Any self, Any x);
extern Any     Sys_rem      (Any self, Any x);
extern Any     Sys_rep      (Any self, Any x);
extern Boolean Sys_rin      (Any self, Any x);
extern Any     Sys_rshift   (Any self, Any x);
extern Any     Sys_setitem  (Any self, Any x, Any v);
extern Any     Sys_setslice (Any self, Any from, Any len, Any v);
extern Boolean Sys_true     (Any self);
extern String  Sys_string   (Any self);
extern Any     Sys_xchg     (Any self, Any x);

extern String  Sys_syscat  (int argc, Any *argv);
extern String  Sys_sysjoin (int argc, Any *argv);



/** SysGetBinop
	Get the binop function for a specified operator.
*/
extern binop_f SysGetBinop (String op);


/** Sys_functions
	These are the script level builtin functions.

	Refer to the script manual for details: %{Sys}.
*/
extern Null    Sys__abort__   (int argc, Any *argv);
extern Null    Sys__buffer__  ();
extern Null    Sys__clear__   ();
extern String  Sys__collect__ ();
extern Any     Sys__err__     ();
extern Module  Sys__main__    ();
extern Null    Sys_print      (int argc, Any *argv);
extern Null    Sys_printc     (int argc, Any *argv);
extern Null    Sys_printf     (int argc, Any *argv);
extern String  Sys_prints     (int argc, Any *argv);
extern Null    Sys_println    (int argc, Any *argv);
extern Null    Sys_printfln   (int argc, Any *argv);
extern Null    Sys_printcln   (int argc, Any *argv);
extern String  Sys_printsln   (int argc, Any *argv);
extern Any     Sys_raise      (int argc, Any *argv);
extern String  Sys__remark__  (int argc, Any *argv);
extern Null    Sys_require    (int argc, Any *argv);
extern Boolean Sys_sysfilter  (Any x, Any ap);
extern Any     Sys_sysmap     (Any x, Any ap);
extern Boolean Sys__trace__   (int argc, Any *argv);
extern Null    Sys__verbose__ (int argc, Any *argv);
extern String  Sys__warning__ (int argc, Any *argv);
extern String  Sys__where__   ();

/** Sys__ignore
	The global __ read/write variable.
	This is specially used in for-each loops.
*/
extern Var Sys__ignore;


/** Sys__keyboard__
	This variable holds the keyboard interrupt handler (a static Sub)
	or nil if none is set.
*/
extern Var Sys__keyboard__;


/** Sys__quit__
	This variable holds the EExit exception handler (a static Sub)
	or nil if none is set.
*/
extern Var Sys__quit__;


/** Format
	format_s is used for formatting objects.

	FormatParse parses a format. It returns non-zero on success, throws EFormat
	and return 0 on failure.

	FormatP handles the 'p' mode format.
	You must always use this function instead of providing your own.

	FormatY handles the 'y' mode format.
	You must always use this function instead of providing your own.

	FormatS handles the 's' mode format.
	You must always use this function instead of providing your own.

	FormatNumber does the final arrangements. The number formatter must provide
	the integral and fractional part, whether the number is negative, and
	the exponent value.

	FORMAT_MAX_WIDTH is the maximum allowed width of a formatted string.
	We consider the maximum length of a string and a "reasonable" value.
	We also make it possible to check for the maximum value I{after} a calculation
	such as: C{width = width * 10 + (*i - '0')}

	FormatAlign does the final alignment.

	Note that for the exponent, 'E' and 'e' must always be used. All number
	parsers and formatters use this convention.
*/
typedef struct format_s format_s;
struct format_s
{
	char mode;    /* mode: 'g', 'f', etc. */
	int  align;   /* 0: align right, otherwise: align left */
	int  sign;    /* 0:"+"/"-" 1:"-" 2:" "/"-" -1:dont print sign */
	int  zero;    /* pad 0 on front */
	int  group;   /* number of characters to group, -1 means none */
	int  width;   /* total available space */
	int  prec;    /* number of decimal digits */
	int  ucase;   /* use uppercase letters */
	char sep;     /* group separator character */
	char dec;     /* decimal point character */
};

#define FORMAT_MAX_WIDTH  (INT_MAX / 256)

extern int    FormatParse  (const char *fmt, format_s *f);
extern String FormatP      (format_s *f, Any x);
extern String FormatS      (format_s *f, Any x);
extern String FormatY      (format_s *f, Any x);
extern String FormatNumber (format_s *f, String s, int neg, int expo);
extern String FormatAlign  (format_s *f, String s);


/** SysValidators
	These are C type validators.
*/
extern Sub Sys_char;
extern Sub Sys_byte;
extern Sub Sys_ubyte;
extern Sub Sys_short;
extern Sub Sys_ushort;
extern Sub Sys_int;
extern Sub Sys_uint;
extern Sub Sys_int16;
extern Sub Sys_uint16;
extern Sub Sys_int32;
extern Sub Sys_uint32;
extern Sub Sys_long;
extern Sub Sys_ulong;
extern Sub Sys_longlong;
extern Sub Sys_ulonglong;
extern Sub Sys_float;
extern Sub Sys_double;
extern Sub Sys_longdouble;
extern Sub Sys_size_t;
extern Sub Sys_ssize_t;
extern Sub Sys_off_t;

extern Sub Sys_Null;
extern Sub Sys_Boolean;
extern Sub Sys_Int;
extern Sub Sys_Pointer;
extern Sub Sys_String;
extern Sub Sys_Array;
extern Sub Sys_Dict;
extern Sub Sys_Set;
extern Sub Sys_Range;
extern Sub Sys_Class;
extern Sub Sys_Queue;
extern Sub Sys_Stream;
extern Sub Sys_Proc;
extern Sub Sys_Any;


/** SysEImplement
	Functions that always throw EImplement.
*/
extern Any SysEImplement0 ();
extern Any SysEImplement1 (Any a1);
extern Any SysEImplement2 (Any a1, Any a2);
extern Any SysEImplement3 (Any a1, Any a2, Any a3);
extern Any SysEImplement4 (Any a1, Any a2, Any a3, Any a4);
extern Any SysEImplementn (int argc, Any *argv);


/** SysSort
	Use this as a replacement of qsort.
*/
/* maximum size of item that can be sorted */
#define SYS_SORT_MAXSIZE SIZEOF_DOUBLE

/* comparison function prototype */
typedef int (*sysosort_f) (const void *i, const void *j, Any obj, Any self);
typedef int (*syssort_f ) (const void *i, const void *j);

/* possible return value of syssort_f */
#define SYS_SORT_ERROR   -2
#define SYS_SORT_SMALLER -1
#define SYS_SORT_EQUAL    0
#define SYS_SORT_GREATER  1

extern int sys_osort (sysosort_f f, void *p, long size, long count, Any obj, Any self);
extern int sys_sort  (syssort_f  f, void *p, long size, long count);

/* return a sort result returned by a call to a Qu function/method */
#define SYS_RETURN_SORT_RESULT(res) \
BEGIN \
	if ((res) == 0) \
		return SYS_SORT_ERROR; \
	else \
	{ \
		long vAl = IntegerToi (res); \
		if (errno) \
		{ \
			ThreadThrow (ECmpClass, 0); \
			return SYS_SORT_ERROR; \
		} \
		return (vAl < 0) ? SYS_SORT_SMALLER : ((vAl > 0) ? SYS_SORT_GREATER : SYS_SORT_EQUAL); \
	} \
END



/***/
/*--------------------------------------------------------------------------*/

/* This is called internally by qu.c to initialize the Sys module */
extern void SysInit ();

#endif /*_H_Sys_*/
