/* Thread.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Thread_
#define _H_Thread_ 1


/** Thread Mutex Cond
	The final Thread, Mutex, and Cond classes.

	All script level methods are not published since you should never use them
	from C code.
*/
extern Class ThreadClass;
extern Class MutexClass;
extern Class CondClass;


/** ThreadState
	Use ThreadState to create a new thread state.

	This is used for printing or constructing strings of containers that may
	be recursive such as Array.

	Acceptable keys one of THREAD_STATE_XXX.
*/
extern void ThreadStateBegin (String key, Any value);
extern int  ThreadStateExist (String key, Any value);
extern void ThreadStateEnd   (String key, Any value);

#define __ThreadStateBegin(k, v) \
BEGIN \
	Any tHrEaDkEy   = k; \
	Any tHrEaDvAlUe = v; \
	ThreadStateBegin (k, v)

#define __ThreadStateEnd \
	ThreadStateEnd (tHrEaDkEy, tHrEaDvAlUe); \
END

#define THREAD_STATE_PRINT      StringBYTE ('p')  /* __print */
#define THREAD_STATE_STRING     StringBYTE ('s')  /* __string */
#define THREAD_STATE_JOIN       StringBYTE ('j')  /* join */
#define THREAD_STATE_JOIN_KEY   StringBYTE ('k')  /* join key (Dict only) */
#define THREAD_STATE_JOIN_VALUE StringBYTE ('v')  /* join value (Dict only) */


/** TThread
	This represent an instance of the Thread class.
*/
struct TThread
{
	OBJ_HEAD;
	long    tid;      /* Thread ID */
	int     ticks;    /* For the scheduler, see interp.c */
	int     depth;    /* To check recursive function call limit */
	long    atomic;   /* Atomic block counter, initially 0 */

	/* These are zeroed once the thread runs */
	Array   args;     /* Args to pass to the start function, initally 0 */
	Sub     sub;      /* Start function, initally 0 */

	/* Internally used by the builtin functions buffer/collect/clear */
	Array   buffers;  /* Array of String objects, initally 0 */

	/* These are set everytime an exception is thrown.
		When a joinable thread dies with an exception, they are used to rethrow
		the exception when the method 'join' is called.

		!!! If ewhere is a C function name, it must be __FUNCTION__.
		!!! It may not be a dynamic/allocated string.
	*/
	Class   eclass; /* exception class */
	Any     emsg;   /* the message, may be 0 */
	void   *ewhere; /* the Module, Sub or Class, or (char *) C function name */
	long    eline;  /* the line number */

	/* This is the result set by InterpRunThread. If an error occured then
		eclass etc. are set and this is 0. This is the result returned
		by the 'join' method. */
	Any     result;

	/* To mark the C stack */
	Any    *stacklo;
	Any    *stackhi;

	/* thread state - managed internally, don't touch! */
	Array   state;

	/* User data - a Dict or 0 */
	Dict    data;

	/* to support Framework - don't touch! */
	Array   extra;

	/* to support html - don't touch! */
	Any     html;
};


/** ThreadFlag
	Flags used by Thread objects.

	Do not access these flags directly.
	Use the accessor macros provided instead.
	If the accessor macros is not defined then it means you should not touch
	the flag.
*/
#define ThreadFlagALIVE     Flag0  /* alive? */
#define ThreadFlagDETACHED  Flag1  /* in detached mode? */
#define ThreadFlagEFUNCTION Flag2  /* ewhere is a char * instead of an object */
#define ThreadFlagBLOCKING  Flag15 /* DEBUG: thread is blocking? */


/** ThreadIsBlocking ThreadSetBlocking ThreadUnsetBlocking
	These are strictly for use by the GC and the interpreter.

	It is used to assert that the thread is not in blocking mode when
	allocating/freeing memory and used to verify the GcLock status.
	Thus they are only meaningful in DEBUG mode.
*/
#define ThreadIsBlocking(self)    FlagTest  (self, ThreadFlagBLOCKING)
#define ThreadSetBlocking(self)   FlagSet   (self, ThreadFlagBLOCKING)
#define ThreadUnsetBlocking(self) FlagUnset (self, ThreadFlagBLOCKING)


/** ThreadIsEfunction ThreadSetEfunction ThreadSetEobject
	These are strictly for use by the interpreter.

	If the ThreadFlagEFUNCTION flag is set then it means that the field
	Thread->ewhere is a C function name thus a C{char *} that comes from
	__FUNCTION__. Otherwise it is a Mod, Class or Sub object.
*/
#define ThreadIsEfunction(self)  FlagTest  (self, ThreadFlagEFUNCTION)
#define ThreadSetEfunction(self) FlagSet   (self, ThreadFlagEFUNCTION)
#define ThreadSetEobject(self)   FlagUnset (self, ThreadFlagEFUNCTION)


/** ThreadMain
	This is the main thread.
*/
extern Thread ThreadMain;


/** ThreadCurrent
	This is the thread which currently holds the ={GcLock}.
	It is set by Interp.c.
*/
extern Thread volatile ThreadCurrent;


/** ThreadMulti
	Initially 0. Set to 1 once another thread is created.

	This should help speed-up single threaded applications because
	they don't have to do a context switch.
*/
extern int volatile ThreadMulti;


/** ThreadSelf
	Get the Thread object of this thread.
*/
extern Thread ThreadSelf ();


/** ThreadIs
	Whether or not the object is a Thread.
*/
#define ThreadIs(o) (ClassOf(o) == ThreadClass)


/** __Atomic
	Use __AtomicBegin and __AtomicEnd to start and end an atomic state.

	@warn
	You must not use these within ={__Blocking}Begin/End.
*/
#define __AtomicBegin BEGIN ThreadCurrent->atomic++; END
#define __AtomicEnd   BEGIN ThreadCurrent->atomic--; END


/** ThreadThrow
	Throwing exceptions from C code.

	v{kls} must be an exception Class or NULL. NULL means the
	Exception class.

	v{msg} may be 0 which means to use the class default message.

	v{cfunc} must be __FUNCTION__. Never call Thread_Throw and
	Thread_Rethrow directly. Use the macros instead.

	Make sure v{obj} is either a Module, Class or Sub.
*/
extern void Thread_Throw   (Class kls, Any msg, const char *cfunc);
extern void Thread_Rethrow (const char *cfunc);
extern void ThreadThrowEx  (Class kls, Any msg, Any obj, long line);

#define ThreadThrow(kls,msg) Thread_Throw (kls, msg, __FUNCTION__)
#define ThreadRethrow()      Thread_Rethrow (__FUNCTION__)

#define ThreadSetEmsgz(msgz) \
BEGIN \
	if (ThreadCurrent) \
		ThreadCurrent->emsg = StringNewz (msgz); \
END

#define ThreadSetEmsg(msg) \
BEGIN \
	if (ThreadCurrent) \
		ThreadCurrent->emsg = msg; \
END


/** ThreadErrStr
	Get the last exception as a string.
*/
extern String ThreadErrStr ();


/** ThreadPrintErr ThreadPrintTrace
	Print the last exception.

	ThreadPrintTrace takes into account ={SysTrace}.
*/
extern void ThreadPrintErr   ();
extern void ThreadPrintTrace ();


/** TheadEStop
	Whether or not the last exception was EStop.
	This is useful for iterators.

	@warn
	Assumes ThreadCurrent thus do not use at Qu initialization time.
*/
#define ThreadEStop() (ThreadCurrent && ThreadCurrent->eclass == EStopClass)


/** ThreadErrIs
	Whether the last exception is a specified class.
*/
#define ThreadErrIs(kls) (ThreadCurrent && ThreadCurrent->eclass == kls##Class)


/** ThreadCheckExternal
	Do the best we can to test whether this thread is our thread.
	Return 1 if it is, 0 if not.
*/
extern  int ThreadCheckExternal ();


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize Thread, Mutex and Cond. */
extern void ThreadSetup ();
extern void ThreadInit  ();

#endif /*_H_Thread_*/
