/* Var.h - Copyright (c) Marc Krisnanto */

#ifndef _H_Var_
#define _H_Var_ 1


/** Var
	This is the final Var class.
*/
extern Class VarClass;


/** TVar
	This represents an instance of the Var class.

	@warn
	Instance properties, once created are stored in C{object->attrs}
	as the values, not as Var objects.
*/
struct TVar
{
	OBJ_HEAD;
	String name;   /* Name - must be the first field */
	Symbol parent; /* Block where declared - Module/Class/Sub */
	Any    value;  /* Value, never 0, initially nil */
	Any    type;   /* Validator or 0 */
	int    index;  /* Local index, if VarFlagLEXICAL set */
};


/** VarFlag
	Flags used by Var objects.

	Do not access these flags directly.
	Use the accessor macros provided instead.
*/
#define VarFlagFINAL     SymbolFlagFINAL    /* final property */
#define VarFlagCLASS     SymbolFlagCLASS    /* class property */
#define VarFlagINST      SymbolFlagINST     /* instance property */
#define VarFlagPRIVATE   SymbolFlagPRIVATE  /* class/derived property */
#define VarFlagLOCAL     SymbolFlagLOCAL    /* local to block and child block */
#define VarFlagLEXICAL   Flag10             /* local sub variable (lexical) */
#define VarFlagSTATIC    Flag11             /* local sub static */
#define VarFlagARG       Flag12             /* function argument */
#define VarFlagVARARG    Flag13             /* function vararg */
#define VarFlagFLATTENER Flag14             /* argument flattener */
#define VarFlagDOCENTRY  Flag15             /* is __doc__ */
#define VarFlagFROZEN    Flag16             /* cant modify */
#define VarFlagRDWR      Flag17             /* read-only or read-write? */
#define VarFlagTYPEOF    Flag18             /* from Array|Dict|Set of x statement? */
#define VarFlagSTRUCT    Flag19             /* is struct */
#define VarFlagENUM      Flag20             /* is enum */
#define VarFlagENUMITEM  Flag21             /* is an enum item */
#define VarFlagALIAS     Flag22             /* documentation hack for alias */


/** VarIsAlias
	Whether this is an alias documentation hack.
*/
#define VarIsAlias(self) FlagTest (self, VarFlagALIAS)


/** VarIsStruct VarIsTypeof VarIsEnum VarIsEnumitem
	Various validators.

	VarIsStruct checks whether it comes from a C{struct} statement.

	VarIsEnum checks whether the variable comes from an C{enum} statement.

	VarIsEnumitem checks whether the variable is an C{enum} item.

	VarIsTypeof checks whether it is a C{Array|Dict|Set of Foo} validator,
	i.e:
	@code
		var foo: Array of String
		var bar: Dict of @baz

	@block
	Thus instead of having to create validator methods, e.g. Array.String,
	Set.Int, Dict.Array, etc., we use a more flexible validator Var object
	which fields are:
	@ul
	.name   = '*' (not important)
	.value  = ArrayClass | SetClass | DictClass
	.type   = any valid validator
	.flag  |= VarFlagTYPEOF
*/
#define VarIsStruct(self)    FlagTest (self, VarFlagSTRUCT)
#define VarIsEnum(self)      FlagTest (self, VarFlagENUM)
#define VarIsEnumitem(self)  FlagTest (self, VarFlagENUMITEM)
#define VarIsTypeof(self)    FlagTest (self, VarFlagTYPEOF)


/** VarIsClass
	Whether this is a class property.
*/
#define VarIsClass(self) FlagTest (self, VarFlagCLASS)


/** VarIsInst
	Whether this is a class property.
*/
#define VarIsInst(self) FlagTest (self, VarFlagINST)


/** VarIsLocal
	Whether this is a local variable known to block and child blocks.
*/
#define VarIsLocal(self) FlagTest (self, VarFlagLOCAL)


/** VarIsLexical
	Whether this is a local Sub variable.

	Note that lexical variables only lives in compile-time.
	They are indexed and live in the Frame stack at run-time.
*/
#define VarIsLexical(self) FlagTest (self, VarFlagLEXICAL)


/** VarIsStatic
	Whether this is a Sub static variable.
*/
#define VarIsStatic(self) FlagTest (self, VarFlagSTATIC)


/** VarIsArg VarIsVararg
	Whether a variable is a Sub argument.

	VarIsVararg checks whether it is a variable argument specifier,
	i.e. C{c} in C{sub foo (a, b, *c)}.
*/
#define VarIsArg(self)    FlagTest (self, VarFlagARG | VarFlagVARARG)
#define VarIsVararg(self) FlagTest (self, VarFlagVARARG)


/** VarIsFlattener
	Whether this is an argument flattener.

	This must only be created by the interpreter to flatten argument to functions,
	i.e. v{bar} in C{foo, *bar, baz}.

	The use of flattener variables allows for small and calculatable stack size.
*/
#define VarIsFlattener(self) FlagTest (self, VarFlagFLATTENER)


/** VarIsDocentry
	Whether this variable comes from a __doc__ statement.
*/
#define VarIsDocentry(self) FlagTest (self, VarFlagDOCENTRY)


/** VarIsPrivate
	Whether this is a private variable known to block/childs, or a property
	known to class/subclasses.
*/
#define VarIsPrivate(self) FlagTest (self, VarFlagPRIVATE)


/** VarIsFrozen VarFreeze
	VarIsFrozen checks whether the value of the variable may be modified (constant).

	VarFreeze makes the variable a constant.
*/
#define VarIsFrozen(self) FlagTest (self, VarFlagFROZEN)
#define VarFreeze(self)   FlagSet  (self, VarFlagFROZEN)


/** VarIsRdwr
	Whether the global variable or property is read/write (read-only otherwise).
*/
#define VarIsRdwr(self) FlagTest (self, VarFlagRDWR)


/** VarNew VarNewEx
	Create a new Var object. Make sure that v{block} is a ={Module},
	={Class} or ={Sub} object.

	Use VarNewEx to type the variable. v{type} must be a ={Class}, ={Sub},
	={Dict}, ={Array}, or ={Set} object.
*/
extern Var VarNew   (Symbol block, String name, long flags);
extern Var VarNewEx (Symbol block, String name, long flags, Any type);

#define VarNewRdwr(b, name)  VarNew (b, name, VarFlagRDWR)
#define VarNewConst(b, name) VarNew (b, name, VarFlagFROZEN)


/** VarNewFlattener
	Create a new flattener variable, i.e. v{c} from C{a, b, *c}.
*/
#define VarNewFlattener() \
	VarNew ((Symbol) SysModule, StringBYTE ('*'), \
		VarFlagLOCAL | VarFlagFROZEN | VarFlagFLATTENER)

/** VarIs
	Whether or not the object is a Var.
*/
#define VarIs(o) (ClassOf(o) == VarClass)


/** VarModule VarClassOf
	VarModule returns the ={Module} object where the variable is declared.

	VarClassOf returns the ={Class} object where the property is declared. If
	the Var is not a property then 0 is returned, no exception thrown.
*/
extern Module VarModule  (Var self);
extern Class  VarClassOf (Var self);


/** VarGetTag VarSetTag
	VarGetTag returns a value in the C{__tag} property.
	Returns 0 if none.

	VarSetTag set a value in the C{__tag} property.

	VarGetTagDict gets all in a Dict.
	Returns 0 if none.
*/
extern Any  VarGetTag     (Var self, String name);
extern void VarSetTag     (Var self, String name, Any value);
extern Dict VarGetTagDict (Var self);


/** VarEnum
	Does what the compiler does with the k{enum} keyword. This creates a
	constant named v{name} in v{where} which contains a frozen ={Dict}
	of the enumerated values. All enumerated values will be Var objects
	in v{where} as well.

	All entries must be a pair of C{const char *} key and values.

	For VarEnum the values must be C{int} the ={Int} class range.
	For VarEnumul the values must be C{unsigned long} values.
	The last argument must be NULL to mark the end of the entries.

	For VarEnumEx you must specify the types.
	See ={StrucTypeToChar} for details.

	@warn
	v{where} must be a ={Module} or ={Class}, otherwise FATAL is issued.
	VarFlagFROZEN is added to ={Var} flags and VarFlagCLASS is automatically added
	if v{where} is a Class.

	The ={Dict} and Var objects are static objects by means of ={GcStatic}.
	You must never modify the content.

	VarEnumEx only supports C{abqiIlLfsx}.
	Others will give you a ={FATAL} error.
	If you use "x" make sure the object is a static object otherwise you get
	a ={FATAL} error as well.
*/
extern Var VarEnum   (Symbol where, const char *name, const char *first, ...);
extern Var VarEnumul (Symbol where, const char *name, const char *first, ...);
extern Var VarEnumEx (Symbol where, const char *name, const char *spec, ...);


/***/
/*--------------------------------------------------------------------------*/

/* These are called internally by Class.c to initialize the Var class */
extern void VarSetup ();
extern void VarInit  ();

/* for interp.c */
extern Any VarSetValue (Var self, Any val);

/* for Reduce.c */
extern int VarAccessible (Var self, Symbol setter);

#endif /*_H_Var_*/
