/* attr.h - Copyright (c) Marc Krisnanto */

#ifndef _H_attr_
#define _H_attr_ 1


/** attr_s
	This represents an attribute table.

	It is used to store object attributes.
	It is a special case of a Dict where all keys are (and must be) ={String} objects.

	The struct is intentionally hidden.
	See C{attr.c} if you are curious.
*/
#ifndef ATTR_S_DECLARED
#define ATTR_S_DECLARED
typedef struct attr_s attr_s;  /* see obj.h */
#endif


/** AttrAlloc
	Allocate a new attribute table.
*/
extern attr_s *AttrAlloc ();


/** AttrFree AttrXfree
	Free an attribute table.

	AttrXfree simply calls AttrFree if its argument is not NULL.
*/
extern void AttrFree (attr_s *t);
#define AttrXfree(t) \
BEGIN \
	if (t) \
		AttrFree (t);\
END


/** AttrMark AttrXmark
	Mark the items in the attribute table.

	These are called by the GC. You must not call this function directly unless
	you are using the table for something else then the v{attrs} field in
	={OBJ_HEAD}.

	AttrXmark simply calls AttrMark if its argument is not NULL.
*/
extern void AttrMark (attr_s *t);
#define AttrXmark(t) \
BEGIN \
	if (t) \
		AttrMark (t); \
END


/** AttrFind
	Find a value in the attribute table.
*/
extern Any AttrFind (attr_s *t, String key);


/** AttrAdd AttrSet
	AttrAdd adds an entry without checking for an existing key. If the key
	already exists then there will be two entries with the same key, so be
	careful. The only reason AttrAdd is provided is for efficiency, i.e.
	when you know an entry with the same key does not exist (e.g. when you
	just created the table). Normally you would use AttrSet instead.

	AttrSet checks for an existing key. If the key already exists, its
	value is set and 1 is returned. If not, a new entry is added and 0
	is returned instead. If the table is fixed then FATAL is issued.

	AttrSetEx is the same but does it even if the table is fixed.
	It always succeed.
*/
extern void AttrAdd   (attr_s *t, String key, Any val);
extern int  AttrSet   (attr_s *t, String key, Any val);

/* EXPERIMENTAL - dotted sub */
extern void AttrSetEx (attr_s *t, String key, Any val);


/** AttrDel
	Delete an entry, return 1 if the entry existed and 0 otherwise.
*/
extern int AttrDel (attr_s *t, String key);


/** AttrLen
	Return the number of items.
*/
extern long AttrLen (attr_s *t);


/** AttrWalk
	Walk through the items.

	The first call must set C{i = 0}.
	@xmp
		String key;
		Any    val;
		long   i = 0;

		while ((val = AttrWalk (MyFooAttr, &key, &i)) != 0)
		{
			... do something with key, val ...
		}

	@warn
	It is not safe to delete nor add an entry inside the loop.
*/
extern Any AttrWalk (attr_s *t, String *key, long *i);


/** AttrMergeToDict
	Export entries to a Dict.
*/
extern void AttrMergeToDict (attr_s *a, Dict d);


/** AttrEndDeclSub
	This is only to be called by ={SubEndDecl}.
	It frees all lexical ={Var} objects to save memory at runtime.
*/
extern void AttrEndDeclSub (Sub u);


/** AttrInstGet AttrInstSet
	Get/set instance attribute.

	AttrInstGet finds an instance attribute value.

	AttrInstSet sets an instance attribute value.

	v{restrict} means whether or not private attributes may be accessed.
	If 1, then only public attributes are looked up.

	@note
	These should really be in C{Object.c} but we need to do some internal
	v{attr} stuffs: the values are stored in a flat array which is much
	more efficient.

	@warn
	self may not be an Int.
*/
extern Any AttrInstGet (Obj self, String name, int restrict);
extern Any AttrInstSet (Obj self, String name, Any val, int restrict);

extern Any AttrInstGetNoThrow (Obj self, String key, int restrict);

/** __AttrNoThrow
	Use these to prevent throwing a useless exception.
*/
extern volatile int AttrNoThrow;
#define __AttrNoThrowBegin BEGIN AttrNoThrow++; END
#define __AttrNoThrowEnd   BEGIN AttrNoThrow--; END


/***/
#endif /*_H_attr_*/
