/* gc.h - Copyright (c) Marc Krisnanto */

#ifndef _H_gc_
#define _H_gc_ 1


/** GC_SIZE_LIMIT
	This is the biggest possible size of an object struct.

	All Txxx structs size must be smaller or equal this limit.

	If you ever create an object with a struct bigger than this then you
	will have to modify gc.c and you are probably doing something stupid.
*/
#define GC_SIZE_LIMIT 1024


/** GC_MAX_ITEMS
	The maximum number of items that can be allocated.

	Note that this value is guaranteed to be <= ={IntMAX}.

	All container objects should use this value instead of defining their own
	even if they can actually contain more than this value.
*/
#define GC_MAX_ITEMS ((INT_MAX >> 1) / SIZEOF_VOID_P)


/** GC_MEM_MAXSIZE GC_MEM_CHECKSIZE
	GC_MEM_MAXSIZE is the maximum size of memory you are allowed to allocate.

	Use GC_MEM_CHECKSIZE to check. It is not perfect but at least it reminds
	you that you must check the size.
*/
#define GC_MEM_MAXSIZE (INT_MAX >> 1)
#define GC_MEM_CHECKSIZE(expr) \
BEGIN \
	if (!(expr)) \
		THROW (EMemory); \
END


/** GcDisable
	Whether or not GC is enabled.

	Increment the count to disable, decrement to restore.
*/
extern int volatile GcDisable;


/** __GcCritical
	Always enclose the creation of composite objects within these two macros.

	This ensures that the GC doesn't get triggered and accidentally call your
	mark function with an incomplete object.

	In the following example, GC can occur right after C{GcNew (Foo)}. This
	will trigger a call to C{_foo_mark} with C{self->dict} and C{self->set}
	not yet initialized.

	@xmp
		void _foo_mark (Foo self)
		{
			GcMark (self->dict);
			GcMark (self->set);
		}

	  __GcCriticalBegin;
		a = GcNew (Foo);
		a->dict = DictNew (4);
		a->set = SetNew (8);
	  __GcCriticalEnd;
		a->mem = GcMalloc (10);

	@code
	That is generally simpler than writing code such as the following example.
	It also eliminates the FlagTest in C{_foo_mark} thus is faster.

	@xmp
		void _foo_mark (Foo self)
		{
			if (FlagTest (self, FooINITIALIZED))
			{
				GcMark (self->dict);
				GcMark (self->set);
			}
		}

		a = GcNew (Foo);
		a->dict = DictNew (4);
		a->set = SetNew (8);
		FlagSet (a, FooINITIALIZED);
		a->mem = GcMalloc (10);

	@note
	Note that GcMemMalloc et.al. do not trigger GC thus is safe to do
	out of the enclosing macros.
*/
#define __GcCriticalBegin (GcDisable++)
#define __GcCriticalEnd   (GcDisable--)


/** GcNew GcRoot GcStatic GcNewIsa
	Object allocation routines.

	GcRoot creates a root object. A root object is always marked and never
	freed. It may refer to other objects.

	GcStatic creates a static object. A static object is never marked and
	never freed. It must not reference other objects. You can use ={GcIsStatic}
	to test whether or not an object is a static object.

	GcNew and GcNewIsa create a normal (non-root non-static) object.
	This object is marked and freed automatically by the GC. It may refer
	to other objects, of course.

	You never need to free an object because that is taken care by the GC
	automatically when the object is no longer referenced. You must never
	call ={GcDelete} unless you know exactly what you are doing. GcDelete
	forces the object to be deleted and is intended to be used by the
	interpreter only.

	@note
	Use the macros if possible.

	Thus unlike previous versions that requires you to call GcFree from
	the class free function, starting from version 1.08 this is no
	longer necessary. In fact, GcFree is now gone.

	Allocation clears C{object->flags} except for GcStatic which sets
	FlagGCSTATIC (see ={Flag}).

	@warn
	You might need to use ={__GcCriticalBegin}.
*/
extern Any Gc_Root   (Class c, size_t size);
extern Any Gc_Static (Class c, size_t size);
extern Any Gc_New    (Class c, size_t size);

#define GcRoot(kls)   ((kls) Gc_Root (kls##Class, kls##Class->size))
#define GcStatic(kls) ((kls) Gc_Static (kls##Class, kls##Class->size))
#define GcNew(kls)    ((kls) Gc_New (kls##Class, kls##Class->size))
#define GcNewIsa(c)   Gc_New (c, (c)->size)


/** GcIsStatic
	Whether or not an object is static.

	For Int objects, this function returns true.
*/
extern int GcIsStatic (Any o);


/** GcDelete
	Force an object to be deleted.

	@warn
	Use with care. Make sure there are no other object referencing this object.
	You would normally only use this with temporary objects.

	This function is designed to for the interpreter in case the -u option
	is set, i.e. to destroy Frame objects as soon as they are no longer used.
*/
extern void GcDelete (Obj o);


/** GcMemMalloc GcMemCalloc GcMemRealloc GcMemFree GcMemCopy
	These are the replacement routines for malloc etc.

	Note that you may not allocate a memory bigger than ={GC_MEM_MAXSIZE}.
	You must check for this yourself before you call the routines.
*/
#ifdef DEBUG_SEGV
extern void *GcMem_Malloc  (size_t size, const char *func);
extern void *GcMem_Calloc  (size_t nelts, size_t size, const char *func);
extern void *GcMem_Realloc (void *p, size_t size, const char *func);
extern void  GcMem_Free    (void *p, const char *func);
extern void *GcMem_Copy    (void *p, const char *func);
#define GcMemMalloc(z)     GcMem_Malloc  (z, __FUNCTION__)
#define GcMemCalloc(n,z)   GcMem_Calloc  (n, z, __FUNCTION__)
#define GcMemRealloc(p,z)  GcMem_Realloc (p, z, __FUNCTION__)
#define GcMemFree(p)       GcMem_Free    (p, __FUNCTION__)
#define GcMemCopy(p)       GcMem_Copy    (p, __FUNCTION__)
#else
extern void *GcMemMalloc  (size_t size);
extern void *GcMemCalloc  (size_t nelts, size_t size);
extern void *GcMemRealloc (void *p, size_t size);
extern void  GcMemFree    (void *p);
extern void *GcMemCopy    (void *p);
#endif /*DEBUG_SEGV*/


/** GcMemStatic
	At initialization Qu allocates a malloc'ed space for used as static memory.

	Static memory can only be allocated. It can not be resized nor freed.
	You must never use GcMemRealloc nor GcMemFree with the pointer returned
	by GcMemStatic.
*/
extern void *GcMemStatic (size_t size);


/** GcNeedReclaim
	Whether or not we need to start GC.

	Set GcNeedReclaim to 1 to force GC as soon as possible.
	To immediately start GC call ={GcReclaim} instead.
*/
extern int volatile GcNeedReclaim;


/** GcReclaim
	Start collecting garbage.

	To simply have GC to run as soon as possible, use ={GcNeedReclaim}
	instead.
*/
extern void GcReclaim ();


/** GcRoots GcStatics GcRegulars
	GcRoots returns the number of root objects, i.e. those allocated with
	={GcRoot}.

	GcStatics returns the number of static objects, i.e. those allocated with
	={GcStatic}.

	GcRegulars returns the number of non-static non-root objects, i.e. those
	allocated with ={GcNew}.

	@note
	These functions may return 0 because the GC may not keep track of the number
	of objects for performance reasons.
*/
extern long GcRegulars ();
extern long GcRoots    ();
extern long GcStatics  ();


/** GcLimit
	Return the current limit (in bytes) when GC should occur.
*/
extern long GcLimit ();


/** GcMark GcXmark
	Mark referenced objects.

	Call these functions from within the v{mark} methods. See plenty of
	examples in the source code.
*/
extern void GcMark  (Any o);
extern void GcXmark (Any o);


/** GcMutex
	This is the one and only global lock.
	Only one thread hold this lock at any given time.

	@warn
	Never acquire the lock directly.
	Use the ={__Blocking}Begin/End macros instead.
*/
extern thread_mutex_t GcMutex;
#define GcLock()   thread_mutex_lock (&GcMutex)
#define GcUnlock() thread_mutex_unlock (&GcMutex)


/** GcMemAllocated
	Read-only: the current amount of active memory.
	This may be used by the GC to determine whether or not GC must start.
*/
extern unsigned long volatile GcMemAllocated;


/** GcStat
	Prints some GC information on v{stderr}.
*/
extern void GcStat ();


/** GcMemTrap
	Set this to a function that must be called when memory allocation fails.

	@warn
	This is intended for the interpreter. You should have no reasons to
	change it.
*/
typedef void (*gcmemtrap_f) ();
extern gcmemtrap_f GcMemTrap;


/***/
/* This is called internally by qu.c to initialiaze the GC */
extern void GcInit ();

/* for Thread.c to mark the C stack */
extern void GcMarkCstack (Thread self, Any *lo, Any *hi);

/* for Weak.c to add a Weak to self */
extern int GcSetWeak (Obj self, Weak w);

/* for Weak.c to remove a Weak from self - called by weak_unref */
extern void GcDelWeak (Obj self, Weak w);


#endif /*_H_gc_*/
