/* interp.h - Copyright (c) Marc Krisnanto */

#ifndef _H_interp_
#define _H_interp_ 1


/** InterpRunMain
	Run the main thread. This is called by the main program. Do not call
	directly. See C{qu.c}.
*/
extern Any InterpRunMain (Module m);


/** InterpRunThread
	All new threads enter here. This function is called from ={Thread}.c.
	Do not call directly.
*/
extern void InterpRunThread (Thread t, Any self);


/** InterpFrameNext
	This is called to get the next value from a generator ={Frame}.

	It is only meant to be called by ={FrameNext}.
*/
extern Any InterpFrameNext (Frame f);


/** InterpWhere InterpSelf InterpModule InterpWhereStr InterpScope
	Get info from the running code.

	InterpWhere returns the block where the code is currently running.
	It returns the associated ={Module}, ={Class} or ={Sub} object.

	InterpWhereStr returns a nicely formatted string.

	InterpSelf returns the current value of self.

	InterpModule returns the ={Module} object of the current running code.

	InterpScope returns a Class if the code is currently running from within
	a Class, the Module otherwise.

	@warn
	It is an error to call these functions before the main program runs.
*/
extern Symbol InterpWhere    ();
extern String InterpWhereStr ();
extern Any    InterpSelf     ();
extern Module InterpModule   ();
extern Symbol InterpScope    ();



/** InterpErrStr InterpPrintErr
	Get the last exception as a string or print it.
*/
extern String InterpErrStr   ();
extern void   InterpPrintErr ();


/** InterpSelf
	Return the associated 'self' object of the current running ={Frame}.

	@warn
	It is an error to call this function before the main program runs.
*/
extern Any InterpSelf ();


/** InterpAddFinalizer
	Add an object for finalization, i.e. have the interpreter call it's
	__destroy method.

	You must not call this function directly. It is called by C{gc.c}.

	The function call can happen at any time. There is absolutely no guarantee
	when and whether or not it is actually called (e.g. abort may be issued
	before the function is called).
*/
extern void InterpAddFinalizer (Any o);


/** InterpUnsafe __UNSAFE__
	Whether or not the call comes from an untrusted eval module.

	Put __UNSAFE__ on top of a C function to throw ESecurity if the call came from
	an untrusted eval module.

	This has the same effect as k{__unsafe} at program level.

	@xmp
		static Int Foo_bar (String baz)
		{
			int a, b;

			__UNSAFE__;
			...
		}
*/
extern int InterpUnsafe ();
#define __UNSAFE__ REQUIRE (ESecurity, !InterpUnsafe())



/** __Blocking
	Use the macros (not the functions) to do a blocking operation.

	Use InterpCheckSignal to check for termination signals every now and
	then while blocking, i.e. __BlockingCheckSignal so we are a bit more
	responsive to, say, C{Ctrl-c}.

	@warn
	Only one thread has the interpreter lock at any given time.

	@block
	__BlockingBegin releases the lock while __BlockingEnd acquires the lock.

	A thread may allocate objects or allocate memory only if it owns the
	interpreter lock.

	={GcReclaim} may be run by any thread at any time. Thus in between
	__BlockingBegin and __BlockingEnd, another thread may run GC. This
	means that you must never allocate objects or allocate memory in
	between __BlockingBegin and __BlockingEnd. Even accessing an object
	is safe only if it is immutable such as String, Float, etc.

	The pairs must be visible in one file.

	If you have issued __BlockingBegin then you must issue __BlockingEnd
	before calling a function unless you are certain that the function does
	not allocate objects/memory. If the function is not within the same C file,
	you must never assume anything.

	If you do not issue __BlockingEnd before calling a function and the
	function happens to issue __BlockingBegin then you have a deadlock.

	As a rule of thumb, only perform system calls in between __BlockingBegin
	and __BlockingEnd.
*/
extern void Interp_lock   ();
extern void Interp_unlock ();
#define __BlockingBegin Interp_unlock ()
#define __BlockingEnd   Interp_lock ()

#define __BlockingCheckSignal \
BEGIN \
	if (SigTerminated || SigInterrupt) \
	{ \
		__BlockingEnd; \
		if (SigTerminated) \
			THROW (EQuit); \
		THROW (EInterrupt); \
	} \
END


#define __BlockingCheckSignalv(retval) \
BEGIN \
	if (SigTerminated || SigInterrupt) \
	{ \
		__BlockingEnd; \
		if (SigTerminated) \
			THROWV (EQuit, retval); \
		THROWV (EInterrupt, retval); \
	} \
END


/** InterpEval
	Evaluate a code string.

	v{mod} is the Module object where eval occurs.

	v{src} is the code string.

	v{imports} is a list of symbol to export.

	v{options} is a combination of INTERP_EVAL.
*/
extern Any InterpEval (Module mod, String src, Dict imports, long options);

/* options is one of these */
#define INTERP_EVAL_EXPR       0x0001 /* compile as expression */
#define INTERP_EVAL_MODULE     0x0002 /* compile as Module */
#define INTERP_EVAL_DOC        0x0008 /* compile as Module, do not initialize */

/* possibly combined with one of these */
#define INTERP_EVAL_TRUSTED    0x1000 /* trusted source? */
#define INTERP_EVAL_GETIMPORT  0x2000 /* get back imports result */


/***/
/* This is internally called by qu.c to initialize the interpreter */
extern void InterpInit ();

/* Inform the interpreter of a signal */
extern volatile int InterpGotSignal;

/* The current running frame. Highly toxic, use with care! */
extern volatile Frame InterpFrame;

/* For the JIT */
extern Any _interp_run_frame (Frame f);

/*
	When a thread->ticks reaches this limit, then its time to switch to
	other threads.
*/
#define INTERP_THREAD_TICKS 10001

/*
	Run finalizers.
	For the interpreter only. Don't call otherwise.
*/
extern volatile int   InterpInFinalize;
extern volatile Array InterpFinalizeList;
extern void Interp_runfinalize ();
#define InterpRunFinalize() \
BEGIN \
	if (!InterpInFinalize && InterpFinalizeList->len) \
	{ \
		InterpInFinalize = 1; \
		Interp_runfinalize (); \
		InterpInFinalize = 0; \
	} \
END


/*
	Callers - see Sub.c

	c = frame and frame stack is on the C stack (non-generator)
	a = frame and frame stack are allocated (non-generator)
	g = frame and frame stack is allocated (generator)
*/
extern Any _interp_call_cn (Sub u, Any self, int argc, Any *argv);
extern Any _interp_call_c4 (Sub u, Any self, Any a1, Any a2, Any a3, Any a4);
extern Any _interp_call_c3 (Sub u, Any self, Any a1, Any a2, Any a3);
extern Any _interp_call_c2 (Sub u, Any self, Any a1, Any a2);
extern Any _interp_call_c1 (Sub u, Any self, Any a1);
extern Any _interp_call_c0 (Sub u, Any self);

extern Any _interp_call_an (Sub u, Any self, int argc, Any *argv);
extern Any _interp_call_a4 (Sub u, Any self, Any a1, Any a2, Any a3, Any a4);
extern Any _interp_call_a3 (Sub u, Any self, Any a1, Any a2, Any a3);
extern Any _interp_call_a2 (Sub u, Any self, Any a1, Any a2);
extern Any _interp_call_a1 (Sub u, Any self, Any a1);
extern Any _interp_call_a0 (Sub u, Any self);

extern Frame _interp_call_gn (Sub u, Any self, int argc, Any *argv);
extern Frame _interp_call_g4 (Sub u, Any self, Any a1, Any a2, Any a3, Any a4);
extern Frame _interp_call_g3 (Sub u, Any self, Any a1, Any a2, Any a3);
extern Frame _interp_call_g2 (Sub u, Any self, Any a1, Any a2);
extern Frame _interp_call_g1 (Sub u, Any self, Any a1);
extern Frame _interp_call_g0 (Sub u, Any self);

extern Any _interp_call_once_cn (Sub u, Any self, int argc, Any *argv);
extern Any _interp_call_once_c4 (Sub u, Any self, Any a1, Any a2, Any a3, Any a4);
extern Any _interp_call_once_c3 (Sub u, Any self, Any a1, Any a2, Any a3);
extern Any _interp_call_once_c2 (Sub u, Any self, Any a1, Any a2);
extern Any _interp_call_once_c1 (Sub u, Any self, Any a1);
extern Any _interp_call_once_c0 (Sub u, Any self);

extern Any _interp_call_once_an (Sub u, Any self, int argc, Any *argv);
extern Any _interp_call_once_a4 (Sub u, Any self, Any a1, Any a2, Any a3, Any a4);
extern Any _interp_call_once_a3 (Sub u, Any self, Any a1, Any a2, Any a3);
extern Any _interp_call_once_a2 (Sub u, Any self, Any a1, Any a2);
extern Any _interp_call_once_a1 (Sub u, Any self, Any a1);
extern Any _interp_call_once_a0 (Sub u, Any self);

extern Any _interp_call_jit_cn (Sub u, Any self, int argc, Any *argv);
extern Any _interp_call_jit_c4 (Sub u, Any self, Any a1, Any a2, Any a3, Any a4);
extern Any _interp_call_jit_c3 (Sub u, Any self, Any a1, Any a2, Any a3);
extern Any _interp_call_jit_c2 (Sub u, Any self, Any a1, Any a2);
extern Any _interp_call_jit_c1 (Sub u, Any self, Any a1);
extern Any _interp_call_jit_c0 (Sub u, Any self);

extern Any _interp_call_jit_an (Sub u, Any self, int argc, Any *argv);
extern Any _interp_call_jit_a4 (Sub u, Any self, Any a1, Any a2, Any a3, Any a4);
extern Any _interp_call_jit_a3 (Sub u, Any self, Any a1, Any a2, Any a3);
extern Any _interp_call_jit_a2 (Sub u, Any self, Any a1, Any a2);
extern Any _interp_call_jit_a1 (Sub u, Any self, Any a1);
extern Any _interp_call_jit_a0 (Sub u, Any self);

#endif /*_H_interp_*/
