/* threads.h - Copyright (c) Marc Krisnanto */

/*
	This is the low-level interface to the underlying system thread library.
	If you are developing thread modules and need more functionalities, then
	please add them here.

	Note that this header file must not include anything from a high-level
	interface. Only system libraries are allowed.
*/

#ifndef _H_threads_
#define _H_threads_ 1

#include <pthread.h>
#ifdef HAVE_SCHED_H
#include <sched.h>
#endif


/** THREAD_STACK_MIN
	Minimum stack size.
*/
#ifdef PTHREAD_STACK_MIN
#define THREAD_STACK_MIN (PTHREAD_STACK_MIN + 32768)
#else
#define THREAD_STACK_MIN 49152
#endif


/** thread_t thread_key_t thread_mutex_t thread_cond_t
	The basic types.
*/
#define thread_t        pthread_t
#define thread_key_t    pthread_key_t
#define thread_mutex_t  pthread_mutex_t
#define thread_cond_t   pthread_cond_t


/** thread_create
	Create new thread.
	@args
	.tid         the new thread will have this ID
	.start       the thread start function
	.arg         argument to pass to the start function
	.stacksize   the stack size to use
	.detach      create the thread in detached state?
*/
extern int thread_create (thread_t *tid, void *(*start)(void *), void *arg,
                          int stacksize, int detach);


/** thread_yield
	Give up CPU to another thread.

	This is the function used by the interpreter for context switching.
*/
#ifdef HAVE_SCHED_YIELD
#define thread_yield() sched_yield()
#else
extern void thread_yield ();
#endif


/** threadMethod
	See the pthread manual for details.
*/
#define thread_sigmask(h,n,o)       pthread_sigmask(h,n,o)
#define thread_self()               pthread_self ()
#define thread_exit(v)              pthread_exit (v)
#define thread_kill(t,s)            pthread_kill (t, s)
#define thread_detach(t)            pthread_detach (t)
#define thread_join(t, p)           pthread_join (t, p)

#define thread_specific_create(k,f) pthread_key_create (k, f)
#define thread_specific_set(k,v)    pthread_setspecific (k, v)
#define thread_specific_get(k)      pthread_getspecific (k)

#define THREAD_MUTEX_INITIALIZER    PTHREAD_MUTEX_INITIALIZER
#define thread_mutex_init(m)        pthread_mutex_init (m, NULL)
#define thread_mutex_destroy(m)     pthread_mutex_destroy (m)
#define thread_mutex_lock(m)        pthread_mutex_lock (m)
#define thread_mutex_trylock(m)     pthread_mutex_trylock (m)
#define thread_mutex_unlock(m)      pthread_mutex_unlock (m)

#define THREAD_COND_INITIALIZER     PTHREAD_COND_INITIALIZER
#define thread_cond_init(c)         pthread_cond_init (c, NULL)
#define thread_cond_destroy(c)      pthread_cond_destroy (c)
#define thread_cond_signal(c)       pthread_cond_signal (c)
#define thread_cond_broadcast(c)    pthread_cond_broadcast (c)
#define thread_cond_wait(c, m)      pthread_cond_wait (c, m)


/***/
#endif /*_H_threads_*/
